package front

import (
	"database/sql"
	"encoding/json"
	"fmt"
	"html/template"
	"io"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
)

type View struct {
	app *core.Core
}

type Template struct {
	templates *template.Template
}

func (t *Template) Render(w io.Writer, name string, data interface{}, c echo.Context) error {
	return t.templates.ExecuteTemplate(w, name, data)
}

func NewView(app *core.Core) *View {
	return &View{app: app}
}

func (v *View) GetIndex(c echo.Context) error {
	return c.Render(http.StatusOK, "index.html", nil)
}

func (v *View) GetTreeRoot(c echo.Context) error {
	c.SetParamNames("DirID")
	c.SetParamValues(fmt.Sprintf("%d", v.app.Config.RootDir))
	return v.GetTreeList(c)
}

func (v *View) GetTreeList(c echo.Context) error {
	id, err := strconv.Atoi(c.Param("DirID"))
	if err != nil {
		return c.NoContent(http.StatusBadRequest)
	}
	dir, err := v.app.Nodes.Get(int(id))
	if err == sql.ErrNoRows {
		return c.NoContent(http.StatusNotFound)
	}
	dirs, err := v.app.Nodes.FindByDir(id)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if dirs == nil {
		dirs = make([]models.Node, 0)
	}
	actions, err := v.app.Actions.FindByDir(id)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if actions == nil {
		actions = make([]models.Action, 0)
	}
	planners, err := v.app.Planners.FindByDir(id)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if planners == nil {
		planners = make([]models.Planner, 0)
	}
	configs, err := v.app.Configs.FindByDir(id)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if configs == nil {
		configs = make([]models.Config, 0)
	}
	var resources []models.Resource
	if err := v.app.WithRoTx(c.Request().Context(), func(tx *sql.Tx) error {
		resources, err = v.app.Resources.FindByDirTx(tx, id)
		return err
	}); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	if resources == nil {
		resources = make([]models.Resource, 0)
	}
	secrets, err := v.app.Secrets.FindByDir(id)
	if err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	vars := map[string]interface{}{
		"Dir":       dir,
		"Dirs":      dirs,
		"Actions":   actions,
		"Planners":  planners,
		"Configs":   configs,
		"Resources": resources,
		"Secrets":   secrets,
	}
	return c.Render(http.StatusOK, "tree.html", vars)
}

func (v *View) GetHostList(c echo.Context) error {
	hosts, err := v.app.Hosts.All()
	if err != nil {
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Render(http.StatusOK, "hosts.html", hosts)
}

func (v *View) GetSecretList(c echo.Context) error {
	return c.Render(http.StatusOK, "secrets.html", v.app.Config.Secrets)
}

func (v *View) GetSecret(c echo.Context) error {
	secretID, err := strconv.Atoi(c.Param("SecretID"))
	if err != nil {
		c.Logger().Warn(err)
		return c.NoContent(http.StatusBadRequest)
	}
	secret, err := v.app.Secrets.Get(secretID)
	if err != nil {
		if err != sql.ErrNoRows {
			c.Logger().Error(err)
			return c.NoContent(http.StatusInternalServerError)
		}
		return c.NoContent(http.StatusBadRequest)
	}
	var data models.YavSecretData
	if err := json.Unmarshal(secret.Data, &data); err != nil {
		c.Logger().Error(err)
		return c.NoContent(http.StatusInternalServerError)
	}
	return c.Redirect(
		http.StatusMovedPermanently,
		fmt.Sprintf("https://yav.yandex-team.ru/secret/%s", data.SecretID),
	)
}

func (v *View) Register(e *echo.Echo, middleware []echo.MiddlewareFunc) {
	setupTemplates(*v.app.Config.Server, e)
	e.GET("/tree", v.GetTreeRoot, middleware...)
	e.GET("/tree/:DirID", v.GetTreeList, middleware...)
	e.GET("/secrets", v.GetSecretList, middleware...)
	e.GET("/action/:ActionID", v.GetActionInfo, middleware...)
	e.GET("/action/:ActionID/edit", v.GetActionEdit, middleware...)
	e.POST("/action/:ActionID/edit", v.PostActionEdit, middleware...)
	e.GET("/action/:ActionID/run", v.GetActionRun, middleware...)
	e.POST("/action/:ActionID/run", v.PostActionRun, middleware...)
	e.GET("/action/:ActionID/plan", v.GetActionPlan, middleware...)
	e.POST("/action/:ActionID/plan", v.PostActionPlan, middleware...)
	e.GET("/action/:ActionID/tree", v.GetActionTreeRoot, middleware...)
	e.GET("/action/:ActionID/tree/:DirID", v.GetActionTreeList, middleware...)
	e.POST("/action/:ActionID/tree/:DirID", v.PostActionTreeList, middleware...)
	e.GET("/planner/:PlannerID", v.GetPlannerInfo, middleware...)
	e.GET("/planner/:PlannerID/edit", v.GetPlannerEdit, middleware...)
	e.POST("/planner/:PlannerID/edit", v.PostPlannerEdit, middleware...)
	e.POST("/planner/:PlannerID/enable", v.PostPlannerEnable, middleware...)
	e.POST("/planner/:PlannerID/disable", v.PostPlannerDisable, middleware...)
	e.GET("/planner/:PlannerID/tree", v.GetPlannerTreeRoot, middleware...)
	e.GET("/planner/:PlannerID/tree/:DirID", v.GetPlannerTreeList, middleware...)
	e.POST("/planner/:PlannerID/tree/:DirID", v.PostPlannerTreeList, middleware...)
	e.GET("/config/:ConfigID", v.GetConfigInfo, middleware...)
	e.GET("/config/:ConfigID/edit", v.GetConfigEdit, middleware...)
	e.POST("/config/:ConfigID/edit", v.PostConfigEdit, middleware...)
	e.GET("/config/:ConfigID/tree", v.GetConfigTreeRoot, middleware...)
	e.GET("/config/:ConfigID/tree/:DirID", v.GetConfigTreeList, middleware...)
	e.POST("/config/:ConfigID/tree/:DirID", v.PostConfigTreeList, middleware...)
	e.GET("/task/:TaskID", v.GetTaskInfo, middleware...)
	e.GET("/secret/:SecretID", v.GetSecret, middleware...)
	e.GET("/queue", v.GetQueue, middleware...)
}
