package models

import (
	"database/sql"
	"fmt"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type AccountKind int

const (
	StaffAccount AccountKind = 1
	GroupAccount AccountKind = 2
)

func (k AccountKind) String() string {
	switch k {
	case StaffAccount:
		return "staff"
	case GroupAccount:
		return "group"
	default:
		return fmt.Sprintf("AccountKind(%d)", k)
	}
}

func (k AccountKind) MarshalText() ([]byte, error) {
	return []byte(k.String()), nil
}

type Account struct {
	ID          int         `db:"id"`
	Kind        AccountKind `db:"kind"`
	Login       string      `db:"login"`
	PassportUID NInt64      `db:"passport_uid"`
	CreateTime  int64       `db:"create_time"`
}

func (o Account) ObjectID() objects.ID {
	return o.ID
}

func (o Account) Clone() Account {
	return o
}

type AccountEvent struct {
	baseEvent
	Account
}

func (e AccountEvent) Object() objects.Object {
	return e.Account
}

func (e AccountEvent) WithObject(o objects.Object) ObjectEvent {
	e.Account = o.(Account)
	return e
}

type AccountStore struct {
	baseStore
	accounts      map[int]Account
	byLogin       indexString
	byPassportUID indexInt64
}

func (s *AccountStore) Get(id int) (Account, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if account, ok := s.accounts[id]; ok {
		return account.Clone(), nil
	}
	return Account{}, sql.ErrNoRows
}

func (s *AccountStore) GetByPassportUID(uid int64) (Account, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	for id := range s.byPassportUID[uid] {
		if account, ok := s.accounts[int(id)]; ok && account.Kind == StaffAccount {
			return account.Clone(), nil
		}
	}
	return Account{}, sql.ErrNoRows
}

func (s *AccountStore) FindByLogin(login string) ([]Account, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var accounts []Account
	for id := range s.byLogin[login] {
		if account, ok := s.accounts[id]; ok {
			accounts = append(accounts, account.Clone())
		}
	}
	return accounts, nil
}

func (s *AccountStore) CreateTx(
	tx gosql.Runner, account *Account, options ...EventOption,
) error {
	result, err := s.CreateObjectEvent(tx, AccountEvent{
		makeBaseEvent(CreateEvent, options...),
		*account,
	})
	if err != nil {
		return err
	}
	*account = result.Object().(Account)
	return nil
}

func (s *AccountStore) UpdateTx(
	tx gosql.Runner, account Account, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, AccountEvent{
		makeBaseEvent(UpdateEvent, options...),
		account,
	})
	return err
}

func (s *AccountStore) RemoveTx(
	tx gosql.Runner, id int, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, AccountEvent{
		makeBaseEvent(RemoveEvent, options...),
		Account{ID: id},
	})
	return err
}

func (s *AccountStore) reset() {
	s.accounts = map[int]Account{}
	s.byPassportUID = indexInt64{}
	s.byLogin = indexString{}
}

func (s *AccountStore) onCreateObject(o objects.Object) {
	account := o.(Account)
	s.accounts[account.ID] = account
	s.byPassportUID.create(int64(account.PassportUID), int64(account.ID))
	s.byLogin.create(account.Login, account.ID)
}

func (s *AccountStore) onRemoveObject(id objects.ID) {
	if account, ok := s.accounts[id.(int)]; ok {
		s.byPassportUID.remove(int64(account.PassportUID), int64(account.ID))
		s.byLogin.remove(account.Login, account.ID)
		delete(s.accounts, account.ID)
	}
}

func NewAccountStore(db *gosql.DB, table, eventTable string) *AccountStore {
	impl := &AccountStore{}
	impl.baseStore = makeBaseStore(
		impl, db, Account{}, table, AccountEvent{}, eventTable,
	)
	return impl
}
