package models

import (
	"database/sql"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type AccountNodeRole struct {
	// ID.
	ID int `db:"id"`
	// AccountID.
	AccountID int `db:"account_id"`
	// NodeID.
	NodeID int `db:"node_id"`
	// Roles.
	RoleID int `db:"role_id"`
}

func (o AccountNodeRole) ObjectID() objects.ID {
	return o.ID
}

func (o AccountNodeRole) Clone() AccountNodeRole {
	return o
}

type AccountNodeRoleEvent struct {
	baseEvent
	AccountNodeRole
}

func (e AccountNodeRoleEvent) Object() objects.Object {
	return e.AccountNodeRole
}

func (e AccountNodeRoleEvent) WithObject(o objects.Object) ObjectEvent {
	e.AccountNodeRole = o.(AccountNodeRole)
	return e
}

type AccountNodeRoleStore struct {
	baseStore
	roles     map[int]AccountNodeRole
	byAccount indexInt
	byNode    indexInt
}

func (s *AccountNodeRoleStore) Get(id int) (AccountNodeRole, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if role, ok := s.roles[id]; ok {
		return role.Clone(), nil
	}
	return AccountNodeRole{}, sql.ErrNoRows
}

func (s *AccountNodeRoleStore) FindByAccount(accountID int) ([]AccountNodeRole, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var roles []AccountNodeRole
	for id := range s.byAccount[accountID] {
		if role, ok := s.roles[id]; ok {
			roles = append(roles, role.Clone())
		}
	}
	return roles, nil
}

func (s *AccountNodeRoleStore) FindByNode(nodeID int) ([]AccountNodeRole, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var roles []AccountNodeRole
	for id := range s.byNode[nodeID] {
		if role, ok := s.roles[id]; ok {
			roles = append(roles, role.Clone())
		}
	}
	return roles, nil
}

func (s *AccountNodeRoleStore) FindByAccountNodes(
	accountID int, nodeIDs map[int]struct{},
) ([]AccountNodeRole, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var roles []AccountNodeRole
	for id := range s.byAccount[accountID] {
		if role, ok := s.roles[id]; ok {
			if _, ok := nodeIDs[role.NodeID]; ok {
				roles = append(roles, role.Clone())
			}
		}
	}
	return roles, nil
}

func (s *AccountNodeRoleStore) CreateTx(
	tx gosql.Runner, role *AccountNodeRole, options ...EventOption,
) error {
	result, err := s.CreateObjectEvent(tx, AccountNodeRoleEvent{
		makeBaseEvent(CreateEvent, options...),
		*role,
	})
	if err != nil {
		return err
	}
	*role = result.Object().(AccountNodeRole)
	return nil
}

func (s *AccountNodeRoleStore) UpdateTx(
	tx gosql.Runner, role AccountNodeRole, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, AccountNodeRoleEvent{
		makeBaseEvent(UpdateEvent, options...),
		role,
	})
	return err
}

func (s *AccountNodeRoleStore) RemoveTx(
	tx gosql.Runner, id int, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, AccountNodeRoleEvent{
		makeBaseEvent(RemoveEvent, options...),
		AccountNodeRole{ID: id},
	})
	return err
}

func (s *AccountNodeRoleStore) reset() {
	s.roles = map[int]AccountNodeRole{}
	s.byAccount = indexInt{}
	s.byNode = indexInt{}
}

func (s *AccountNodeRoleStore) onCreateObject(o objects.Object) {
	role := o.(AccountNodeRole)
	s.roles[role.ID] = role
	s.byAccount.create(role.AccountID, role.ID)
	s.byNode.create(role.NodeID, role.ID)
}

func (s *AccountNodeRoleStore) onRemoveObject(id objects.ID) {
	if role, ok := s.roles[id.(int)]; ok {
		s.byAccount.remove(role.AccountID, role.ID)
		s.byNode.remove(role.NodeID, role.ID)
		delete(s.roles, role.ID)
	}
}

func NewAccountNodeRoleStore(
	db *gosql.DB, table, eventTable string,
) *AccountNodeRoleStore {
	impl := &AccountNodeRoleStore{}
	impl.baseStore = makeBaseStore(
		impl, db, AccountNodeRole{}, table, AccountNodeRoleEvent{}, eventTable,
	)
	return impl
}
