package models

import (
	"database/sql"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type AccountRole struct {
	// ID.
	ID int `db:"id"`
	// AccountID.
	AccountID int `db:"account_id"`
	// Roles.
	RoleID int `db:"role_id"`
}

func (o AccountRole) ObjectID() objects.ID {
	return o.ID
}

func (o AccountRole) Clone() AccountRole {
	return o
}

type AccountRoleEvent struct {
	baseEvent
	AccountRole
}

func (e AccountRoleEvent) Object() objects.Object {
	return e.AccountRole
}

func (e AccountRoleEvent) WithObject(o objects.Object) ObjectEvent {
	e.AccountRole = o.(AccountRole)
	return e
}

type AccountRoleStore struct {
	baseStore
	roles     map[int]AccountRole
	byAccount indexInt
}

func (s *AccountRoleStore) Get(id int) (AccountRole, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if role, ok := s.roles[id]; ok {
		return role.Clone(), nil
	}
	return AccountRole{}, sql.ErrNoRows
}

func (s *AccountRoleStore) FindByAccount(accountID int) ([]AccountRole, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var roles []AccountRole
	for id := range s.byAccount[accountID] {
		if role, ok := s.roles[id]; ok {
			roles = append(roles, role.Clone())
		}
	}
	return roles, nil
}

func (s *AccountRoleStore) CreateTx(
	tx gosql.Runner, role *AccountRole, options ...EventOption,
) error {
	result, err := s.CreateObjectEvent(tx, AccountRoleEvent{
		makeBaseEvent(CreateEvent, options...),
		*role,
	})
	if err != nil {
		return err
	}
	*role = result.Object().(AccountRole)
	return nil
}

func (s *AccountRoleStore) UpdateTx(
	tx gosql.Runner, role AccountRole, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, AccountRoleEvent{
		makeBaseEvent(UpdateEvent, options...),
		role,
	})
	return err
}

func (s *AccountRoleStore) RemoveTx(
	tx gosql.Runner, id int, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, AccountRoleEvent{
		makeBaseEvent(RemoveEvent, options...),
		AccountRole{ID: id},
	})
	return err
}

func (s *AccountRoleStore) reset() {
	s.roles = map[int]AccountRole{}
	s.byAccount = indexInt{}
}

func (s *AccountRoleStore) onCreateObject(o objects.Object) {
	role := o.(AccountRole)
	s.roles[role.ID] = role
	s.byAccount.create(role.AccountID, role.ID)
}

func (s *AccountRoleStore) onRemoveObject(id objects.ID) {
	if role, ok := s.roles[id.(int)]; ok {
		s.byAccount.remove(role.AccountID, role.ID)
		delete(s.roles, role.ID)
	}
}

func NewAccountRoleStore(
	db *gosql.DB, table, eventTable string,
) *AccountRoleStore {
	impl := &AccountRoleStore{}
	impl.baseStore = makeBaseStore(
		impl, db, AccountRole{}, table, AccountRoleEvent{}, eventTable,
	)
	return impl
}
