package models

import (
	"database/sql"
	"database/sql/driver"
	"encoding/json"
	"fmt"
	"strings"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type OptionType string

const (
	StringOption  OptionType = "String"
	IntegerOption OptionType = "Integer"
	FloatOption   OptionType = "Float"
	SecretOption  OptionType = "Secret"
	ConfigOption  OptionType = "Config"
	JSONOption    OptionType = "JSON"
	// Deprecated.
	JSONQuery OptionType = "JSONQuery"
)

// Deprecated.
type SecretValue struct {
	Store string `json:""`
	Name  string `json:""`
}

type ActionOption struct {
	// Type contains type of option.
	Type OptionType `json:""`
	// Title contains human readable name of option.
	Title string `json:",omitempty"`
	// Description contains description of option for humans.
	Description string `json:",omitempty"`
	// Should human see this option, or this option should be hidden.
	Visible bool `json:""`
	// Editable describes can human edit this option. When option is not
	// editable, you do not need to pass it's default value to api handler.
	Editable bool `json:""`
	// Required describes can option value be an empty (null).
	Required bool `json:""`
	// Settings contains type special settings.
	Settings interface{} `json:",omitempty"`
	// Value contains value that corresponds to type.
	Value interface{} `json:",omitempty"`
	// Priority describes order of option.
	Priority int `json:""`
}

func (o *ActionOption) UnmarshalJSON(bytes []byte) error {
	var opt struct {
		Type        OptionType          `json:""`
		Title       string              `json:""`
		Description string              `json:""`
		Visible     bool                `json:""`
		Editable    bool                `json:""`
		Required    bool                `json:""`
		Settings    interface{}         `json:""`
		Value       *optionGenericValue `json:""`
		Priority    int                 `json:""`
	}
	if err := json.Unmarshal(bytes, &opt); err != nil {
		return err
	}
	if opt.Value != nil {
		value, err := getOptionValue(opt.Type, *opt.Value)
		if err != nil {
			return err
		}
		o.Value = value
	} else {
		o.Value = nil
	}
	o.Type = opt.Type
	o.Title = opt.Title
	o.Description = opt.Description
	o.Visible = opt.Visible
	o.Editable = opt.Editable
	o.Required = opt.Required
	o.Settings = opt.Settings
	o.Priority = opt.Priority
	return nil
}

type ActionOptions map[string]ActionOption

func (o ActionOptions) Value() (driver.Value, error) {
	bytes, err := json.Marshal(o)
	if err != nil {
		return nil, err
	}
	return string(bytes), err
}

func (o *ActionOptions) Scan(data interface{}) error {
	switch value := data.(type) {
	case []byte:
		return json.Unmarshal(value, o)
	case string:
		return json.Unmarshal([]byte(value), o)
	case nil:
		return nil
	default:
		return fmt.Errorf("unsupported type: %T", value)
	}
}

func (o ActionOptions) TaskOptions(opts TaskOptions) (TaskOptions, error) {
	res := TaskOptions{}
	// First step validates that passed task options are correct.
	for name, option := range opts {
		opt, ok := o[name]
		if !ok {
			// Ignore unknown options for compatibility.
			continue
		}
		if opt.Type != option.Type {
			return nil, fmt.Errorf("incorrect type of option %q", name)
		}
		if opt.Required && option.Value == nil {
			return nil, fmt.Errorf("required option %q is empty", name)
		}
		res[name] = option
	}
	// Second step validates that default values are also correct.
	for name, option := range o {
		if _, ok := opts[name]; ok {
			continue
		}
		if option.Required && option.Value == nil {
			return nil, fmt.Errorf("required option %q is empty", name)
		}
		res[name] = TaskOption{Type: option.Type, Value: option.Value}
	}
	return res, nil
}

type Action struct {
	// ID contains ID of action.
	ID int `db:"id" json:""`
	// DirID contains ID of directory.
	DirID NInt `db:"dir_id" json:",omitempty"`
	// OwnerID contains ID of owner.
	OwnerID NInt `db:"owner_id" json:",omitempty"`
	// CreateTime contains creation time.
	CreateTime int64 `db:"create_time" json:""`
	// Options contains action options.
	Options ActionOptions `db:"options" json:""`
	// Title contains action title.
	Title string `db:"title" json:""`
	// Description contains action description.
	Description string `db:"description" json:",omitempty"`
}

// ObjectID returns ID of action.
func (o Action) ObjectID() objects.ID {
	return o.ID
}

func (o Action) clone() Action {
	optionsCopy := ActionOptions{}
	for key, value := range o.Options {
		optionsCopy[key] = value
	}
	o.Options = optionsCopy
	return o
}

type ActionEvent struct {
	baseEvent
	Action
}

func (e ActionEvent) Object() objects.Object {
	return e.Action
}

func (e ActionEvent) WithObject(o objects.Object) ObjectEvent {
	e.Action = o.(Action)
	return e
}

type ActionStore struct {
	baseStore
	actions map[int]Action
	byDir   indexInt
}

func (s *ActionStore) Get(id int) (Action, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if action, ok := s.actions[id]; ok {
		return action.clone(), nil
	}
	return Action{}, sql.ErrNoRows
}

func (s *ActionStore) FindByDir(id int) ([]Action, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var actions []Action
	for id := range s.byDir[id] {
		if action, ok := s.actions[id]; ok {
			actions = append(actions, action.clone())
		}
	}
	return actions, nil
}

func (s *ActionStore) FindByQuery(query string) ([]Action, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var actions []Action
	for _, action := range s.actions {
		options, err := json.Marshal(action.Options)
		if err != nil {
			return nil, err
		}
		if strings.Contains(action.Title, query) || strings.Contains(string(options), query) {
			actions = append(actions, action)
		}
	}
	return actions, nil
}

func (s *ActionStore) CreateTx(
	tx gosql.Runner, action *Action, options ...EventOption,
) error {
	result, err := s.CreateObjectEvent(tx, ActionEvent{
		makeBaseEvent(CreateEvent, options...),
		*action,
	})
	if err != nil {
		return err
	}
	*action = result.Object().(Action)
	return nil
}

func (s *ActionStore) UpdateTx(
	tx gosql.Runner, action Action, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, ActionEvent{
		makeBaseEvent(UpdateEvent, options...),
		action,
	})
	return err
}

func (s *ActionStore) RemoveTx(
	tx gosql.Runner, id int, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, ActionEvent{
		makeBaseEvent(RemoveEvent, options...),
		Action{ID: id},
	})
	return err
}

func (s *ActionStore) Validate(action Action) error {
	errList := FieldListError{}
	if action.DirID == 0 {
		errList = append(errList, FieldError{"DirID", InvalidField})
	}
	if len(action.Title) < 1 {
		errList = append(errList, FieldError{"Title", TooShortField})
	}
	if len(action.Title) > 64 {
		errList = append(errList, FieldError{"Title", TooLongField})
	}
	if len(action.Description) > 4096 {
		errList = append(errList, FieldError{"Description", TooLongField})
	}
	if action.Options == nil {
		errList = append(errList, FieldError{"Options", InvalidField})
	} else {
		if _, ok := action.Options["@command"]; !ok {
			// Deprecated.
			if _, ok := action.Options["@Command"]; !ok {
				if _, ok := action.Options["__Command"]; !ok {
					errList = append(errList, FieldError{"Options", InvalidField})
				}
			}
		}
	}
	return errList.AsError()
}

func (s *ActionStore) reset() {
	s.actions = map[int]Action{}
	s.byDir = indexInt{}
}

func (s *ActionStore) onCreateObject(o objects.Object) {
	action := o.(Action)
	s.actions[action.ID] = action
	s.byDir.create(int(action.DirID), action.ID)
}

func (s *ActionStore) onRemoveObject(id objects.ID) {
	if action, ok := s.actions[id.(int)]; ok {
		s.byDir.remove(int(action.DirID), action.ID)
		delete(s.actions, action.ID)
	}
}

func NewActionStore(db *gosql.DB, table, eventTable string) *ActionStore {
	impl := &ActionStore{}
	impl.baseStore = makeBaseStore(
		impl, db, Action{}, table, ActionEvent{}, eventTable,
	)
	return impl
}
