package models

import (
	"database/sql"
	"database/sql/driver"
	"encoding/base64"
	"encoding/json"
	"errors"
	"fmt"
	"strings"
	"sync"
	"time"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db"
	"a.yandex-team.ru/zootopia/library/go/db/events"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type (
	// ID represents ID of object.
	ID = int64
	// NID represents nullable ID of object.
	NID = NInt64
)

// EventType represents event type.
type EventType string

const (
	// CreateEvent represents create event.
	CreateEvent EventType = "Create"
	// RemoveEvent represents remove event.
	RemoveEvent EventType = "Remove"
	// UpdateEvent represents update event.
	UpdateEvent EventType = "Update"
)

// ObjectEvent represents event for object with ID.
type ObjectEvent interface {
	events.Event
	// EventType should return type of event.
	EventType() EventType
	// Object should return object.
	Object() objects.Object
	// WithObject should return event copy with replaced object.
	WithObject(objects.Object) ObjectEvent
}

// baseEvent represents basic event for objects.
type baseEvent struct {
	BaseEventID     ID        `db:"event_id" json:"EventID"`
	BaseEventType   EventType `db:"event_type" json:"EventType"`
	BaseEventTime   int64     `db:"event_time" json:"EventTime"`
	BaseEventUserID NID       `db:"event_user_id" json:"EventUserID,omitempty"`
	BaseEventTaskID NID       `db:"event_task_id" json:"EventTaskID,omitempty"`
}

// EventID returns ID of event.
func (e baseEvent) EventID() ID {
	return e.BaseEventID
}

// SetEventID sets ID of event.
func (e *baseEvent) SetEventID(id ID) {
	e.BaseEventID = id
}

// EventType returns type of event.
func (e baseEvent) EventType() EventType {
	return e.BaseEventType
}

// SetEventType returns type of event.
func (e *baseEvent) SetEventType(typ EventType) {
	e.BaseEventType = typ
}

// EventTime returns time of event.
func (e baseEvent) EventTime() time.Time {
	return time.Unix(e.BaseEventTime, 0)
}

// SetEventTime sets time of event.
func (e *baseEvent) SetEventTime(ts time.Time) {
	e.BaseEventTime = ts.Unix()
}

type baseStoreImpl interface {
	reset()
	onCreateObject(objects.Object)
	onRemoveObject(objects.ID)
	onUpdateObject(objects.Object)
}

type stubStoreImpl struct{}

func (stubStoreImpl) reset() {}

func (stubStoreImpl) onCreateObject(objects.Object) {}

func (stubStoreImpl) onRemoveObject(objects.ID) {}

func (stubStoreImpl) onUpdateObject(objects.Object) {}

// Manager represents cached store.
type Manager interface {
	InitTx(tx gosql.Runner) error
	SyncTx(tx gosql.Runner) error
}

type baseStore struct {
	impl     baseStoreImpl
	db       *gosql.DB
	objects  objects.Store
	events   events.Store
	consumer events.Consumer
	mutex    sync.RWMutex
}

// BeginEventID returns ID of last event in the store.
func (s *baseStore) BeginEventID() int64 {
	return s.consumer.BeginEventID()
}

var (
	withReadOnly = gosql.WithTxOptions(&sql.TxOptions{
		ReadOnly: true,
	})
	withReadWrite = gosql.WithTxOptions(&sql.TxOptions{
		// Force REPEATABLE READ for Postgres.
		Isolation: sql.LevelRepeatableRead,
	})
)

func (s *baseStore) InitTx(tx gosql.Runner) error {
	switch tx := tx.(type) {
	case *sql.Tx:
		return s.initTx(tx)
	case gosql.TxBeginner:
		return gosql.WithTx(tx, s.initTx, withReadOnly)
	default:
		panic(fmt.Errorf("unsupported type: %T", tx))
	}
}

func (s *baseStore) initTx(tx *sql.Tx) error {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	if err := s.initEvents(tx); err != nil {
		return err
	}
	if err := s.initObjects(tx); err != nil {
		return err
	}
	return nil
}

func (s *baseStore) SyncTx(tx gosql.Runner) error {
	switch tx := tx.(type) {
	case *sql.Tx:
		return s.syncTx(tx)
	case gosql.TxBeginner:
		return gosql.WithTx(tx, s.syncTx, withReadOnly)
	default:
		panic(fmt.Errorf("unsupported type: %T", tx))
	}
}

func (s *baseStore) syncTx(tx *sql.Tx) error {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	return s.consumer.ConsumeEvents(tx, s.consumeEvent)
}

func (s *baseStore) FindObject(
	tx *sql.Tx, where string, args ...interface{},
) (objects.Object, error) {
	rows, err := s.objects.FindObjects(tx, where, args...)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = rows.Close()
	}()
	if !rows.Next() {
		return nil, sql.ErrNoRows
	}
	if err := rows.Err(); err != nil {
		return nil, err
	}
	return rows.Object(), nil
}

func (s *baseStore) FindObjects(
	tx *sql.Tx, where string, args ...interface{},
) ([]objects.Object, error) {
	rows, err := s.objects.FindObjects(tx, where, args...)
	if err != nil {
		return nil, err
	}
	defer func() {
		_ = rows.Close()
	}()
	var list []objects.Object
	for rows.Next() {
		list = append(list, rows.Object())
	}
	if err := rows.Err(); err != nil {
		return nil, err
	}
	return list, nil
}

// CreateObjectEvent creates object event in persistent store.
func (s *baseStore) CreateObjectEvent(
	tx gosql.Runner, event ObjectEvent,
) (ObjectEvent, error) {
	switch tx := tx.(type) {
	case *sql.Tx:
		return s.createObjectEvent(tx, event)
	case gosql.TxBeginner:
		var object ObjectEvent
		if err := gosql.WithTx(tx, func(tx *sql.Tx) (err error) {
			object, err = s.createObjectEvent(tx, event)
			return
		}, withReadWrite); err != nil {
			return nil, err
		}
		return object, nil
	default:
		panic(fmt.Errorf("unsupported type: %T", tx))
	}
}

func (s *baseStore) createObjectEvent(
	tx *sql.Tx, event ObjectEvent,
) (ObjectEvent, error) {
	switch object := event.Object(); event.EventType() {
	case CreateEvent:
		object, err := s.objects.CreateObject(tx, object)
		if err != nil {
			return nil, err
		}
		event = event.WithObject(object)
	case RemoveEvent:
		if err := s.objects.RemoveObject(tx, object.ObjectID()); err != nil {
			return nil, err
		}
	case UpdateEvent:
		object, err := s.objects.UpdateObject(tx, object)
		if err != nil {
			return nil, err
		}
		event = event.WithObject(object)
	}
	result, err := s.events.CreateEvent(tx, event)
	if err != nil {
		return nil, err
	}
	return result.(ObjectEvent), err
}

const oldGapDistance = 2000

func (s *baseStore) initEvents(tx *sql.Tx) error {
	lastID, err := s.events.LastEventID(tx)
	if err != nil {
		// LastEventID can return sql.ErrNoRows if there is no events.
		if err != sql.ErrNoRows {
			return err
		}
		lastID = 1
	}
	// lastID should be always greater than 0.
	if lastID > oldGapDistance {
		lastID -= oldGapDistance
	} else {
		lastID = 1
	}
	// Create consumer with delayed begin EventID for graceful
	// detection of all gaps.
	s.consumer = events.NewConsumer(s.events, lastID)
	return s.consumer.ConsumeEvents(tx, ignoreEvent)
}

// ignoreEvent does nothing.
func ignoreEvent(events.Event) error {
	return nil
}

func (s *baseStore) initObjects(tx *sql.Tx) error {
	rows, err := s.objects.ReadObjects(tx)
	if err != nil {
		return err
	}
	defer func() {
		_ = rows.Close()
	}()
	s.impl.reset()
	for rows.Next() {
		s.impl.onCreateObject(rows.Object())
	}
	return rows.Err()
}

func (s *baseStore) onUpdateObject(o objects.Object) {
	s.impl.onRemoveObject(o.ObjectID())
	s.impl.onCreateObject(o)
}

func (s *baseStore) consumeEvent(event events.Event) error {
	switch v := event.(ObjectEvent); v.EventType() {
	case CreateEvent:
		s.impl.onCreateObject(v.Object())
	case RemoveEvent:
		s.impl.onRemoveObject(v.Object().ObjectID())
	case UpdateEvent:
		s.impl.onUpdateObject(v.Object())
	default:
		return fmt.Errorf("unexpected event type: %v", v.EventType())
	}
	return nil
}

type EventOption func(*baseEvent)

func WithUser(id int) EventOption {
	return func(event *baseEvent) {
		event.BaseEventUserID = NID(id)
	}
}

func WithTask(id int64) EventOption {
	return func(event *baseEvent) {
		event.BaseEventTaskID = NID(id)
	}
}

func makeBaseEvent(eventType EventType, options ...EventOption) baseEvent {
	event := baseEvent{
		BaseEventType: eventType,
		BaseEventTime: time.Now().Unix(),
	}
	for _, option := range options {
		option(&event)
	}
	return event
}

func getDialect(conn *gosql.DB) db.Dialect {
	switch conn.Driver {
	case gosql.PostgresDriver:
		return db.Postgres
	case gosql.SQLiteDriver:
		return db.SQLite
	default:
		panic("unsupported driver")
	}
}

func makeBaseStore(
	impl baseStoreImpl, db *gosql.DB,
	object objects.Object, table string,
	event ObjectEvent, eventTable string,
) baseStore {
	dialect := getDialect(db)
	return baseStore{
		impl:    impl,
		db:      db,
		objects: objects.NewStore(object, "id", table, dialect),
		events:  events.NewStore(event, "event_id", eventTable, dialect),
	}
}

type indexInt64 map[int64]map[int64]struct{}

func (m indexInt64) create(key, id int64) {
	if _, ok := m[key]; !ok {
		m[key] = map[int64]struct{}{}
	}
	m[key][id] = struct{}{}
}

func (m indexInt64) remove(key, id int64) {
	delete(m[key], id)
	if len(m[key]) == 0 {
		delete(m, key)
	}
}

type indexInt map[int]map[int]struct{}

func (m indexInt) create(key, id int) {
	if _, ok := m[key]; !ok {
		m[key] = map[int]struct{}{}
	}
	m[key][id] = struct{}{}
}

func (m indexInt) remove(key, id int) {
	delete(m[key], id)
	if len(m[key]) == 0 {
		delete(m, key)
	}
}

type indexString map[string]map[int]struct{}

func (m indexString) create(key string, id int) {
	if _, ok := m[key]; !ok {
		m[key] = map[int]struct{}{}
	}
	m[key][id] = struct{}{}
}

func (m indexString) remove(key string, id int) {
	delete(m[key], id)
	if len(m[key]) == 0 {
		delete(m, key)
	}
}

// NInt64 represents nullable integer.
type NInt64 int64

func (v NInt64) eqOp(n int) string {
	if v == 0 {
		return "IS NULL"
	}
	return fmt.Sprintf("= $%d", n)
}

func (v NInt64) has() bool {
	return v != 0
}

// Value returns SQL value or NULL, if value is zero.
func (v NInt64) Value() (driver.Value, error) {
	if v == 0 {
		return nil, nil
	}
	return int64(v), nil
}

// Scan scans integer or NULL.
func (v *NInt64) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = 0
	case int64:
		*v = NInt64(x)
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

// NInt represents nullable integer.
type NInt int

func (v NInt) eqOp(n int) string {
	if v == 0 {
		return "IS NULL"
	}
	return fmt.Sprintf("= $%d", n)
}

func (v NInt) has() bool {
	return v != 0
}

// Value returns SQL value or NULL, if value is zero.
func (v NInt) Value() (driver.Value, error) {
	if v == 0 {
		return nil, nil
	}
	return int64(v), nil
}

// Scan scans integer or NULL.
func (v *NInt) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = 0
	case int64:
		*v = NInt(x)
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
	return nil
}

// JSON represents raw JSON value.
type JSON []byte

// MarshalJSON mashals JSON.
func (v JSON) MarshalJSON() ([]byte, error) {
	if len(v) == 0 {
		return []byte("null"), nil
	}
	if !json.Valid(v) {
		return nil, errors.New("invalid json value")
	}
	return v, nil
}

// UnmarshalJSON unmarshals JSON.
func (v *JSON) UnmarshalJSON(bytes []byte) error {
	if len(bytes) == 0 {
		*v = nil
		return nil
	}
	if !json.Valid(bytes) {
		return errors.New("invalid json value")
	}
	*v = JSON(bytes)
	return nil
}

// Value returns SQL value.
func (v JSON) Value() (driver.Value, error) {
	value, err := v.MarshalJSON()
	if err != nil {
		return nil, err
	}
	return string(value), nil
}

// Scan scans SQL value.
func (v *JSON) Scan(value interface{}) error {
	switch x := value.(type) {
	case nil:
		*v = nil
		return nil
	case []byte:
		return v.UnmarshalJSON(x)
	case string:
		return v.UnmarshalJSON([]byte(x))
	default:
		return fmt.Errorf("unsupported value: %T", x)
	}
}

// Clone creates copy of JSON object.
func (v JSON) Clone() JSON {
	c := make(JSON, len(v))
	copy(c, v)
	return c
}

const (
	InvalidField  = "Invalid"
	TooShortField = "TooShort"
	TooLongField  = "TooLong"
)

type FieldError struct {
	Name string `json:""`
	Type string `json:""`
}

func (e FieldError) Error() string {
	return fmt.Sprintf("field %q has %q error", e.Name, e.Type)
}

type FieldListError []FieldError

func (e FieldListError) Error() string {
	var result strings.Builder
	for _, err := range e {
		if result.Len() > 0 {
			result.WriteString(", ")
		}
		result.WriteString(err.Error())
	}
	return result.String()
}

func (e FieldListError) AsError() error {
	if len(e) == 0 {
		return nil
	}
	return e
}

// Base64 represents base64 value.
type Base64 []byte

// MarshalText marshals text.
func (v Base64) MarshalText() ([]byte, error) {
	return []byte(base64.StdEncoding.EncodeToString(v)), nil
}

// UnmarshalText unmarshals text.
func (v *Base64) UnmarshalText(bytes []byte) (err error) {
	*v, err = base64.StdEncoding.DecodeString(string(bytes))
	return
}

// Value returns SQL value.
func (v Base64) Value() (driver.Value, error) {
	return base64.StdEncoding.EncodeToString(v), nil
}

// Scan scans SQL value.
func (v *Base64) Scan(data interface{}) (err error) {
	switch value := data.(type) {
	case nil:
		*v = nil
	case []byte:
		*v, err = base64.StdEncoding.DecodeString(string(value))
	case string:
		*v, err = base64.StdEncoding.DecodeString(value)
	default:
		return fmt.Errorf("unsupported type: %T", value)
	}
	return
}

func intOrNil(id int) *int {
	if id == 0 {
		return nil
	}
	return &id
}

type baseObject struct {
	ID ID `json:"id"`
}

func (o baseObject) ObjectID() ID {
	return o.ID
}

func (o *baseObject) SetObjectID(id ID) {
	o.ID = id
}
