package models

import (
	"database/sql"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type Config struct {
	ID          int    `db:"id" json:""`
	DirID       NInt   `db:"dir_id" json:",omitempty"`
	Title       string `db:"title" json:""`
	Description string `db:"description" json:",omitempty"`
	Data        string `db:"data" json:""`
}

func (o Config) ObjectID() objects.ID {
	return o.ID
}

func (o Config) Clone() Config {
	return o
}

type ConfigEvent struct {
	baseEvent
	Config
}

func (e ConfigEvent) Object() objects.Object {
	return e.Config
}

func (e ConfigEvent) WithObject(o objects.Object) ObjectEvent {
	e.Config = o.(Config)
	return e
}

type ConfigStore struct {
	baseStore
	configs map[int]Config
	byDir   indexInt
}

func (s *ConfigStore) Get(id int) (Config, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if config, ok := s.configs[id]; ok {
		return config.Clone(), nil
	}
	return Config{}, sql.ErrNoRows
}

func (s *ConfigStore) FindByDir(id int) ([]Config, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var configs []Config
	for id := range s.byDir[id] {
		if config, ok := s.configs[id]; ok {
			configs = append(configs, config.Clone())
		}
	}
	return configs, nil
}

func (s *ConfigStore) CreateTx(
	tx gosql.Runner, config *Config, options ...EventOption,
) error {
	result, err := s.CreateObjectEvent(tx, ConfigEvent{
		makeBaseEvent(CreateEvent, options...),
		*config,
	})
	if err != nil {
		return err
	}
	*config = result.Object().(Config)
	return nil
}

func (s *ConfigStore) UpdateTx(
	tx gosql.Runner, config Config, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, ConfigEvent{
		makeBaseEvent(UpdateEvent, options...),
		config,
	})
	return err
}

func (s *ConfigStore) RemoveTx(
	tx gosql.Runner, id int, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, ConfigEvent{
		makeBaseEvent(RemoveEvent, options...),
		Config{ID: id},
	})
	return err
}

func (s *ConfigStore) Validate(config Config) error {
	errList := FieldListError{}
	if config.DirID == 0 {
		errList = append(errList, FieldError{"DirID", InvalidField})
	}
	if len(config.Title) < 1 {
		errList = append(errList, FieldError{"Title", TooShortField})
	}
	if len(config.Title) > 64 {
		errList = append(errList, FieldError{"Title", TooLongField})
	}
	if len(config.Description) > 4096 {
		errList = append(errList, FieldError{"Description", TooLongField})
	}
	if len(config.Data) > 1048576 {
		errList = append(errList, FieldError{"Data", TooLongField})
	}
	return errList.AsError()
}

func (s *ConfigStore) reset() {
	s.configs = map[int]Config{}
	s.byDir = indexInt{}
}

func (s *ConfigStore) onCreateObject(o objects.Object) {
	config := o.(Config)
	s.configs[config.ID] = config
	s.byDir.create(int(config.DirID), config.ID)
}

func (s *ConfigStore) onRemoveObject(id objects.ID) {
	if config, ok := s.configs[id.(int)]; ok {
		s.byDir.remove(int(config.DirID), config.ID)
		delete(s.configs, config.ID)
	}
}

func NewConfigStore(db *gosql.DB, table, eventTable string) *ConfigStore {
	impl := &ConfigStore{}
	impl.baseStore = makeBaseStore(
		impl, db, Config{}, table, ConfigEvent{}, eventTable,
	)
	return impl
}
