package models

import (
	"database/sql"
	"database/sql/driver"
	"encoding/json"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/drive/library/go/gosql"
)

type HostTags map[string]string

type HostConfig struct {
	Active       bool `json:"active"`
	EnableLeader bool `json:"enable_leader"`
	// Labels contains host labels from configuration.
	Labels map[string]string `json:"labels"`
}

func (v *HostConfig) Scan(src interface{}) error {
	switch t := src.(type) {
	case string:
		return json.Unmarshal([]byte(t), v)
	case []byte:
		return json.Unmarshal(t, v)
	case nil:
		*v = HostConfig{}
		return nil
	default:
		return fmt.Errorf("incompatible type %T", t)
	}
}

func (v HostConfig) Value() (driver.Value, error) {
	bytes, err := json.Marshal(v)
	if err != nil {
		return nil, err
	}
	return string(bytes), nil
}

type HostState struct {
	Labels       map[string]string `json:"labels"`
	RunningTasks []string          `json:"running_tasks"`
}

// Scan scans host state from SQL.
func (v *HostState) Scan(src interface{}) error {
	switch t := src.(type) {
	case string:
		return json.Unmarshal([]byte(t), v)
	case []byte:
		return json.Unmarshal(t, v)
	case nil:
		*v = HostState{}
		return nil
	default:
		return fmt.Errorf("incompatible type %T", t)
	}
}

// Value returns host state for SQL.
func (v HostState) Value() (driver.Value, error) {
	bytes, err := json.Marshal(v)
	if err != nil {
		return nil, err
	}
	return string(bytes), nil
}

// Host represents host.
type Host struct {
	ID       int        `db:"id"`
	Name     string     `db:"name"`
	PingTime int64      `db:"ping_time"`
	Config   HostConfig `db:"config"`
	State    HostState  `db:"state"`
}

// Running checks that host is actually running.
func (m Host) Running() bool {
	return m.PingTime+int64(lockTimeout.Seconds()) > time.Now().Unix()
}

// Offline checks that host is offine.
func (m Host) Offline() bool {
	return m.PingTime+int64(lockTimeout.Seconds())*2 < time.Now().Unix()
}

// Active checks that host is actually active.
func (m Host) Active() bool {
	return m.Running() && m.Config.Active
}

// HasRunningTask returns if host now runs specified task.
func (m Host) HasRunningTask(name string) bool {
	if !m.Running() {
		return false
	}
	for _, task := range m.State.RunningTasks {
		if task == name {
			return true
		}
	}
	return false
}

type HostStore struct {
	db    *gosql.DB
	table string
}

func NewHostStore(db *gosql.DB, table string) *HostStore {
	return &HostStore{db: db, table: table}
}

func (o *HostTags) Scan(src interface{}) error {
	switch t := src.(type) {
	case string:
		return json.Unmarshal([]byte(t), o)
	case []byte:
		return json.Unmarshal(t, o)
	case nil:
		return nil
	default:
		return errors.New("incompatible type for `HostTags`")
	}
}

func (o HostTags) Value() (driver.Value, error) {
	if o == nil {
		return driver.Value("{}"), nil
	}
	bytes, err := json.Marshal(o)
	if err != nil {
		return nil, err
	}
	return string(bytes), nil
}

func (s *HostStore) GetTx(tx gosql.Runner, id int) (Host, error) {
	var host Host
	names, scans := gosql.StructNameValues(&host, true)
	query, values := s.db.Select(s.table).
		Names(names...).Where(gosql.Column("id").Equal(id)).Build()
	row := tx.QueryRow(query, values...)
	if err := row.Scan(scans...); err != nil {
		return Host{}, err
	}
	return host, nil
}

func (s *HostStore) GetByNameTx(tx gosql.Runner, name string) (Host, error) {
	var host Host
	names, scans := gosql.StructNameValues(&host, true)
	query, values := s.db.Select(s.table).
		Names(names...).Where(gosql.Column("name").Equal(name)).Build()
	row := tx.QueryRow(query, values...)
	if err := row.Scan(scans...); err != nil {
		return Host{}, err
	}
	return host, nil
}

func (s *HostStore) CreateTx(tx gosql.Runner, host Host) (Host, error) {
	host.PingTime = time.Now().Unix()
	names, values := gosql.StructNameValues(host, false, "id")
	query, values := s.db.Insert(s.table).
		Names(names...).Values(values...).Build()
	row := tx.QueryRow(query+` RETURNING "id"`, values...)
	if err := row.Scan(&host.ID); err != nil {
		return Host{}, err
	}
	return host, nil
}

func (s *HostStore) UpdateTx(
	tx gosql.Runner, host Host, columns ...string,
) error {
	names, values := gosql.StructNameValues(host, false, "id")
	if len(columns) > 0 {
		allowed := map[string]struct{}{}
		for _, column := range columns {
			allowed[column] = struct{}{}
		}
		newLen := 0
		for i := 0; i < len(names); i++ {
			if _, ok := allowed[names[i]]; !ok {
				continue
			}
			names[newLen] = names[i]
			values[newLen] = values[i]
			newLen++
		}
		names, values = names[:newLen], values[:newLen]
	}
	query, values := s.db.Update(s.table).
		Names(names...).Values(values...).
		Where(gosql.Column("id").Equal(host.ID)).Build()
	res, err := tx.Exec(query, values...)
	if err != nil {
		return err
	}
	affected, err := res.RowsAffected()
	if err != nil {
		return err
	}
	if affected != 1 {
		return sql.ErrNoRows
	}
	return nil
}

func (s *HostStore) FindTx(
	tx gosql.Runner, where gosql.BoolExpr,
) ([]Host, error) {
	query, values := s.db.Select(s.table).
		Names(gosql.StructNames(Host{})...).Where(where).Build()
	rows, err := tx.Query(query, values...)
	if err != nil {
		return nil, err
	}
	var hosts []Host
	for rows.Next() {
		var host Host
		if err := rows.Scan(gosql.StructValues(&host, true)...); err != nil {
			return nil, err
		}
		hosts = append(hosts, host)
	}
	return hosts, rows.Err()
}

func (s *HostStore) All() ([]Host, error) {
	return s.FindTx(s.db.RO, nil)
}

func (s *HostStore) FindActive() ([]Host, error) {
	nodes, err := s.All()
	if err != nil {
		return nil, err
	}
	newLen := 0
	for i := 0; i < len(nodes); i++ {
		if nodes[i].Active() {
			nodes[newLen] = nodes[i]
			newLen++
		}
	}
	return nodes[:newLen], nil
}

// Ping tries to ping specified host.
//
// Host object will be updated only when query is succeeded.
func (s *HostStore) Ping(host *Host) error {
	newHost := Host{
		ID:       host.ID,
		PingTime: time.Now().Unix(),
		State:    host.State,
	}
	if err := s.UpdateTx(
		s.db, newHost, "ping_time", "state",
	); err != nil {
		return err
	}
	host.PingTime = newHost.PingTime
	return nil
}

// ReloadTx tries to reload information about host.
//
// Host will be updated only on successful attempt.
func (s *HostStore) ReloadTx(tx gosql.Runner, host *Host) error {
	newHost, err := s.GetTx(tx, host.ID)
	if err != nil {
		return err
	}
	*host = newHost
	return nil
}

// Reload tries to reload information about host.
func (s *HostStore) Reload(host *Host) error {
	return s.ReloadTx(s.db.RO, host)
}
