package models

import (
	"encoding/json"
	"fmt"
	"strings"
)

type JSONQueryValues map[string]interface{}

const (
	JSONQueryLogicalOrType    = "LogicalOr"
	JSONQueryLogicalAndType   = "LogicalAnd"
	JSONQueryIsNilType        = "IsNil"
	JSONQueryIsNotNilType     = "IsNotNil"
	JSONQueryInType           = "In"
	JSONQueryNotInType        = "NotIn"
	JSONQueryEqualType        = "Equal"
	JSONQueryNotEqualType     = "NotEqual"
	JSONQueryLessType         = "Less"
	JSONQueryGreaterType      = "Greater"
	JSONQueryLessEqualType    = "LessEqual"
	JSONQueryGreaterEqualType = "GreaterEqual"
)

type JSONQueryOperator interface {
	Calculate(values JSONQueryValues) bool
}

type JSONQueryLogicalOperator struct {
	Operands []JSONQueryOperator `json:""`
}

func (o JSONQueryLogicalOperator) marshalJSON(t string) ([]byte, error) {
	data := struct {
		JSONQueryLogicalOperator
		Type string `json:""`
	}{Type: t, JSONQueryLogicalOperator: o}
	return json.Marshal(data)
}

func (o *JSONQueryLogicalOperator) UnmarshalJSON(data []byte) error {
	var raw struct {
		Operands []json.RawMessage `json:""`
	}
	if err := json.Unmarshal(data, &raw); err != nil {
		return err
	}
	o.Operands = nil
	for _, r := range raw.Operands {
		op, err := jsonQueryUnmarshalJSON(r)
		if err != nil {
			return err
		}
		o.Operands = append(o.Operands, op)
	}
	return nil
}

type JSONQueryLogicalOr struct {
	JSONQueryLogicalOperator
}

func (o JSONQueryLogicalOr) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryLogicalOrType)
}

func (o *JSONQueryLogicalOr) Calculate(values JSONQueryValues) bool {
	for _, operand := range o.Operands {
		if operand.Calculate(values) {
			return true
		}
	}
	return false
}

type JSONQueryLogicalAnd struct {
	JSONQueryLogicalOperator
}

func (o JSONQueryLogicalAnd) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryLogicalAndType)
}

func (o *JSONQueryLogicalAnd) Calculate(values JSONQueryValues) bool {
	for _, operand := range o.Operands {
		if !operand.Calculate(values) {
			return false
		}
	}
	return true
}

type JSONQueryCompareOperator struct {
	Field string      `json:""`
	Value interface{} `json:""`
}

func (o JSONQueryCompareOperator) marshalJSON(t string) ([]byte, error) {
	data := struct {
		JSONQueryCompareOperator
		Type string `json:""`
	}{Type: t, JSONQueryCompareOperator: o}
	return json.Marshal(data)
}

// returns :
//   0 - if field is equal
//   1 - if field greater
//   -1 - if field less
//   2  - if field is not equal
//   -2 - if field does not exists
func (o JSONQueryCompareOperator) compare(values JSONQueryValues) int {
	val, ok := values[o.Field]
	if !ok {
		return -2
	}
	return compare(val, o.Value)
}

func compareFloat(lhs float64, rhs interface{}) int {
	var val float64
	switch v := rhs.(type) {
	case float64:
		val = v
	case float32:
		val = float64(v)
	case int:
		val = float64(v)
	case int64:
		val = float64(v)
	case int32:
		val = float64(v)
	case int16:
		val = float64(v)
	case int8:
		val = float64(v)
	case uint64:
		val = float64(v)
	case uint32:
		val = float64(v)
	case uint16:
		val = float64(v)
	case uint8:
		val = float64(v)
	case bool:
		if v {
			val = 1
		} else {
			val = 0
		}
	default:
		return 2
	}
	if lhs < val {
		return -1
	} else if lhs > val {
		return 1
	} else {
		return 0
	}
}

func compareInt(lhs int64, rhs interface{}) int {
	var val int64
	switch v := rhs.(type) {
	case int:
		val = int64(v)
	case int64:
		val = v
	case int32:
		val = int64(v)
	case int16:
		val = int64(v)
	case int8:
		val = int64(v)
	case uint64:
		val = int64(v)
	case uint32:
		val = int64(v)
	case uint16:
		val = int64(v)
	case uint8:
		val = int64(v)
	case float64:
		return compareFloat(float64(lhs), rhs)
	case float32:
		return compareFloat(float64(lhs), rhs)
	case bool:
		if v {
			val = 1
		} else {
			val = 0
		}
	default:
		return 2
	}
	if lhs < val {
		return -1
	} else if lhs > val {
		return 1
	} else {
		return 0
	}
}

func compareString(lhs string, rhs interface{}) int {
	var val string
	switch v := rhs.(type) {
	case string:
		val = v
	case []byte:
		val = string(v)
	default:
		return 2
	}
	if lhs < val {
		return -1
	} else if lhs > val {
		return 1
	} else {
		return 0
	}
}

func compare(lhs, rhs interface{}) int {
	switch v := lhs.(type) {
	case nil:
		if rhs == nil {
			return 0
		} else {
			return 2
		}
	case int:
		return compareInt(int64(v), rhs)
	case int64:
		return compareInt(v, rhs)
	case int32:
		return compareInt(int64(v), rhs)
	case int16:
		return compareInt(int64(v), rhs)
	case int8:
		return compareInt(int64(v), rhs)
	case uint64:
		return compareInt(int64(v), rhs)
	case uint32:
		return compareInt(int64(v), rhs)
	case uint16:
		return compareInt(int64(v), rhs)
	case uint8:
		return compareInt(int64(v), rhs)
	case float64:
		return compareFloat(v, rhs)
	case float32:
		return compareFloat(float64(v), rhs)
	case string:
		return compareString(v, rhs)
	case []byte:
		return compareString(string(v), rhs)
	case bool:
		if v {
			return compareInt(1, rhs)
		} else {
			return compareInt(0, rhs)
		}
	default:
		return 2
	}
}

func contains(lhs, rhs interface{}) bool {
	switch v := lhs.(type) {
	case []string:
		for _, val := range v {
			if compareString(val, rhs) == 0 {
				return true
			}
		}
	case []int:
		for _, val := range v {
			if compareInt(int64(val), rhs) == 0 {
				return true
			}
		}
	case []int64:
		for _, val := range v {
			if compareInt(val, rhs) == 0 {
				return true
			}
		}
	case []int32:
		for _, val := range v {
			if compareInt(int64(val), rhs) == 0 {
				return true
			}
		}
	case []int16:
		for _, val := range v {
			if compareInt(int64(val), rhs) == 0 {
				return true
			}
		}
	case []int8:
		for _, val := range v {
			if compareInt(int64(val), rhs) == 0 {
				return true
			}
		}
	case []uint64:
		for _, val := range v {
			if compareInt(int64(val), rhs) == 0 {
				return true
			}
		}
	case []uint32:
		for _, val := range v {
			if compareInt(int64(val), rhs) == 0 {
				return true
			}
		}
	case []uint16:
		for _, val := range v {
			if compareInt(int64(val), rhs) == 0 {
				return true
			}
		}
	case []uint8:
		for _, val := range v {
			if compareInt(int64(val), rhs) == 0 {
				return true
			}
		}
	case []float64:
		for _, val := range v {
			if compareFloat(val, rhs) == 0 {
				return true
			}
		}
	case []float32:
		for _, val := range v {
			if compareFloat(float64(val), rhs) == 0 {
				return true
			}
		}
	case []interface{}:
		for _, val := range v {
			if compare(val, rhs) == 0 {
				return true
			}
		}
	}
	return false
}

type JSONQueryIsNil struct {
	Field string
}

func (o JSONQueryIsNil) marshalJSON(t string) ([]byte, error) {
	data := struct {
		JSONQueryIsNil
		Type string `json:""`
	}{Type: t, JSONQueryIsNil: o}
	return json.Marshal(data)
}

func (o JSONQueryIsNil) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryIsNilType)
}

func (o *JSONQueryIsNil) Calculate(values JSONQueryValues) bool {
	val, ok := values[o.Field]
	return !ok || val == nil
}

type JSONQueryIsNotNil struct {
	JSONQueryIsNil
}

func (o JSONQueryIsNotNil) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryIsNotNilType)
}

func (o *JSONQueryIsNotNil) Calculate(values JSONQueryValues) bool {
	return !o.JSONQueryIsNil.Calculate(values)
}

type JSONQueryIn struct {
	JSONQueryCompareOperator
}

func (o JSONQueryIn) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryInType)
}

func (o *JSONQueryIn) Calculate(values JSONQueryValues) bool {
	val, ok := values[o.Field]
	if !ok {
		return false
	}
	switch v := o.Value.(type) {
	case string:
		for _, part := range strings.Split(v, ",") {
			if contains(val, part) {
				return true
			}
		}
		return false
	default:
		return contains(val, o.Value)
	}
}

type JSONQueryNotIn struct {
	JSONQueryCompareOperator
}

func (o JSONQueryNotIn) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryNotInType)
}

func (o *JSONQueryNotIn) Calculate(values JSONQueryValues) bool {
	val, ok := values[o.Field]
	if !ok {
		return false
	}
	switch v := o.Value.(type) {
	case string:
		for _, part := range strings.Split(v, ",") {
			if !contains(val, part) {
				return true
			}
		}
		return false
	default:
		return !contains(val, o.Value)
	}
}

type JSONQueryEqual struct {
	JSONQueryCompareOperator
}

func (o JSONQueryEqual) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryEqualType)
}

func (o *JSONQueryEqual) Calculate(values JSONQueryValues) bool {
	return o.compare(values) == 0
}

type JSONQueryNotEqual struct {
	JSONQueryCompareOperator
}

func (o JSONQueryNotEqual) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryNotEqualType)
}

func (o *JSONQueryNotEqual) Calculate(values JSONQueryValues) bool {
	return o.compare(values) != 0
}

type JSONQueryLess struct {
	JSONQueryCompareOperator
}

func (o JSONQueryLess) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryLessType)
}

func (o *JSONQueryLess) Calculate(values JSONQueryValues) bool {
	return o.compare(values) == -1
}

type JSONQueryGreater struct {
	JSONQueryCompareOperator
}

func (o JSONQueryGreater) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryGreaterType)
}

func (o *JSONQueryGreater) Calculate(values JSONQueryValues) bool {
	return o.compare(values) == 1
}

type JSONQueryLessEqual struct {
	JSONQueryCompareOperator
}

func (o JSONQueryLessEqual) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryLessEqualType)
}

func (o *JSONQueryLessEqual) Calculate(values JSONQueryValues) bool {
	return o.compare(values) == 0 || o.compare(values) == -1
}

type JSONQueryGreaterEqual struct {
	JSONQueryCompareOperator
}

func (o JSONQueryGreaterEqual) MarshalJSON() ([]byte, error) {
	return o.marshalJSON(JSONQueryGreaterEqualType)
}

func (o *JSONQueryGreaterEqual) Calculate(values JSONQueryValues) bool {
	return o.compare(values) == 0 || o.compare(values) == 1
}

type JSONQueryValue struct {
	JSONQueryOperator
}

func jsonQueryUnmarshalJSON(data []byte) (JSONQueryOperator, error) {
	var raw struct {
		Type string `json:""`
	}
	if err := json.Unmarshal(data, &raw); err != nil {
		return nil, err
	}
	var op JSONQueryOperator
	switch raw.Type {
	case JSONQueryLogicalOrType:
		op = &JSONQueryLogicalOr{}
	case JSONQueryLogicalAndType:
		op = &JSONQueryLogicalAnd{}
	case JSONQueryIsNilType:
		op = &JSONQueryIsNil{}
	case JSONQueryIsNotNilType:
		op = &JSONQueryIsNotNil{}
	case JSONQueryInType:
		op = &JSONQueryIn{}
	case JSONQueryNotInType:
		op = &JSONQueryNotIn{}
	case JSONQueryEqualType:
		op = &JSONQueryEqual{}
	case JSONQueryNotEqualType:
		op = &JSONQueryNotEqual{}
	case JSONQueryLessType:
		op = &JSONQueryLess{}
	case JSONQueryGreaterType:
		op = &JSONQueryGreater{}
	case JSONQueryLessEqualType:
		op = &JSONQueryLessEqual{}
	case JSONQueryGreaterEqualType:
		op = &JSONQueryGreaterEqual{}
	default:
		return nil, fmt.Errorf(
			"unsupported operator '%s'", raw.Type,
		)
	}
	if err := json.Unmarshal(data, op); err != nil {
		return nil, err
	}
	return op, nil
}

func (v JSONQueryValue) MarshalJSON() ([]byte, error) {
	return json.Marshal(v.JSONQueryOperator)
}

func (v *JSONQueryValue) UnmarshalJSON(data []byte) error {
	var err error
	v.JSONQueryOperator, err = jsonQueryUnmarshalJSON(data)
	return err
}
