package models

import (
	"database/sql"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type Node struct {
	ID           int    `db:"id"          json:""`
	NodeID       NInt   `db:"node_id"     json:"DirID,omitempty"`
	CreateTime   int64  `db:"create_time" json:""`
	Title        string `db:"name"        json:""`
	Description  string `db:"description" json:",omitempty"`
	InheritRoles bool   `db:"inherit_roles"`
}

func (o Node) ObjectID() objects.ID {
	return o.ID
}

func (o Node) Clone() Node {
	return o
}

type NodeEvent struct {
	baseEvent
	Node
}

func (e NodeEvent) Object() objects.Object {
	return e.Node
}

func (e NodeEvent) WithObject(o objects.Object) ObjectEvent {
	e.Node = o.(Node)
	return e
}

type NodeStore struct {
	baseStore
	nodes  map[int]Node
	byNode indexInt
}

func (s *NodeStore) Get(id int) (Node, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if node, ok := s.nodes[id]; ok {
		return node.Clone(), nil
	}
	return Node{}, sql.ErrNoRows
}

func (s *NodeStore) FindByDir(id int) ([]Node, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var nodes []Node
	for id := range s.byNode[id] {
		if node, ok := s.nodes[id]; ok {
			nodes = append(nodes, node.Clone())
		}
	}
	return nodes, nil
}

func (s *NodeStore) CreateTx(
	tx gosql.Runner, node Node, options ...EventOption,
) (Node, error) {
	result, err := s.CreateObjectEvent(tx, NodeEvent{
		makeBaseEvent(CreateEvent, options...),
		node,
	})
	if err != nil {
		return Node{}, err
	}
	return result.Object().(Node), nil
}

func (s *NodeStore) UpdateTx(
	tx gosql.Runner, node Node, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, NodeEvent{
		makeBaseEvent(UpdateEvent, options...),
		node,
	})
	return err
}

func (s *NodeStore) DeleteTx(
	tx gosql.Runner, id int, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, NodeEvent{
		makeBaseEvent(RemoveEvent, options...),
		Node{ID: id},
	})
	return err
}

func (s *NodeStore) reset() {
	s.nodes = map[int]Node{}
	s.byNode = indexInt{}
}

func (s *NodeStore) onCreateObject(o objects.Object) {
	node := o.(Node)
	s.nodes[node.ID] = node
	s.byNode.create(int(node.NodeID), node.ID)
}

func (s *NodeStore) onRemoveObject(id objects.ID) {
	if node, ok := s.nodes[id.(int)]; ok {
		s.byNode.remove(int(node.NodeID), node.ID)
		delete(s.nodes, node.ID)
	}
}

func NewNodeStore(db *gosql.DB, table, eventTable string) *NodeStore {
	impl := &NodeStore{}
	impl.baseStore = makeBaseStore(
		impl, db, Node{}, table, NodeEvent{}, eventTable,
	)
	return impl
}
