package models

import (
	"database/sql"
	"database/sql/driver"
	"encoding/json"
	"errors"
	"strings"
	"time"

	"a.yandex-team.ru/drive/library/go/cron"
	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type PlannerSettings struct {
	RepeatDelay   *int           `json:""`
	FailureDelay  *int           `json:""`
	Schedule      *cron.Schedule `json:""`
	AllowOverlaps bool           `json:""`
	ListenSuccess []int          `json:""`
	ListenFailure []int          `json:""`
	FailureMails  []string       `json:""`
	Enabled       bool           `json:""`
	// NotifyFailureThreshold contains amount of failures that
	// allowed before notification.
	NotifyFailureThreshold int `json:"notify_failure_threshold,omitempty"`
}

func (o *PlannerSettings) Scan(src interface{}) error {
	switch t := src.(type) {
	case string:
		return json.Unmarshal([]byte(t), o)
	case []byte:
		return json.Unmarshal(t, o)
	case nil:
		return nil
	default:
		return errors.New("incompatible type for PlannerSettings")
	}
}

func (o PlannerSettings) Value() (driver.Value, error) {
	bytes, err := json.Marshal(o)
	if err != nil {
		return nil, err
	}
	return driver.Value(string(bytes)), err
}

func (o PlannerSettings) Clone() PlannerSettings {
	if o.RepeatDelay != nil {
		value := *o.RepeatDelay
		o.RepeatDelay = &value
	}
	if o.FailureDelay != nil {
		value := *o.FailureDelay
		o.FailureDelay = &value
	}
	if o.Schedule != nil {
		value := *o.Schedule
		o.Schedule = &value
	}
	return o
}

type Planner struct {
	ID          int             `db:"id"          json:""`
	ActionID    int             `db:"action_id"   json:""`
	DirID       int             `db:"dir_id"      json:",omitempty"`
	OwnerID     int             `db:"owner_id"    json:",omitempty"`
	Title       string          `db:"title"       json:""`
	Description string          `db:"description" json:",omitempty"`
	CreateTime  int64           `db:"create_time" json:""`
	Options     TaskOptions     `db:"options"     json:""`
	Settings    PlannerSettings `db:"settings"    json:""`
}

// ObjectID returns ID of planner.
func (o Planner) ObjectID() objects.ID {
	return o.ID
}

func (o Planner) Clone() Planner {
	o.Options = o.Options.Clone()
	o.Settings = o.Settings.Clone()
	return o
}

func (o Planner) Next(from int64) int64 {
	// 10 years like infinite timeout
	to := from + time.Now().AddDate(10, 0, 0).Unix()
	if o.Settings.RepeatDelay != nil {
		if from+int64(*o.Settings.RepeatDelay) < to {
			to = from + int64(*o.Settings.RepeatDelay)
		}
	}
	if o.Settings.Schedule != nil {
		next := o.Settings.Schedule.
			Next(time.Unix(from, 0).In(time.UTC)).
			Unix()
		if next < to {
			to = next
		}
	}
	return to
}

type PlannerEvent struct {
	baseEvent
	Planner
}

func (e PlannerEvent) Object() objects.Object {
	return e.Planner
}

func (e PlannerEvent) WithObject(o objects.Object) ObjectEvent {
	e.Planner = o.(Planner)
	return e
}

type PlannerStore struct {
	baseStore
	planners map[int]Planner
	byDir    indexInt
}

func (s *PlannerStore) Get(id int) (Planner, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if planner, ok := s.planners[id]; ok {
		return planner.Clone(), nil
	}
	return Planner{}, sql.ErrNoRows
}

func (s *PlannerStore) All() ([]Planner, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var planners []Planner
	for _, planner := range s.planners {
		planners = append(planners, planner.Clone())
	}
	return planners, nil
}

func (s *PlannerStore) FindByDir(id int) ([]Planner, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var planners []Planner
	for id := range s.byDir[id] {
		if planner, ok := s.planners[id]; ok {
			planners = append(planners, planner.Clone())
		}
	}
	return planners, nil
}

func (s *PlannerStore) FindByQuery(query string) ([]Planner, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var planners []Planner
	for _, planner := range s.planners {
		options, err := json.Marshal(planner.Options)
		if err != nil {
			return nil, err
		}
		if strings.Contains(planner.Title, query) || strings.Contains(string(options), query) {
			planners = append(planners, planner)
		}
	}
	return planners, nil
}

func (s *PlannerStore) CreateTx(
	tx gosql.Runner, planner *Planner, options ...EventOption,
) error {
	result, err := s.CreateObjectEvent(tx, PlannerEvent{
		makeBaseEvent(CreateEvent, options...),
		*planner,
	})
	if err != nil {
		return err
	}
	*planner = result.Object().(Planner)
	return nil
}

func (s *PlannerStore) UpdateTx(
	tx gosql.Runner, planner Planner, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, PlannerEvent{
		makeBaseEvent(UpdateEvent, options...),
		planner,
	})
	return err
}

func (s *PlannerStore) RemoveTx(
	tx gosql.Runner, id int, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, PlannerEvent{
		makeBaseEvent(RemoveEvent, options...),
		Planner{ID: id},
	})
	return err
}

func (s *PlannerStore) reset() {
	s.planners = map[int]Planner{}
	s.byDir = indexInt{}
}

func (s *PlannerStore) onCreateObject(o objects.Object) {
	planner := o.(Planner)
	s.planners[planner.ID] = planner
	s.byDir.create(int(planner.DirID), planner.ID)
}

func (s *PlannerStore) onRemoveObject(id objects.ID) {
	if planner, ok := s.planners[id.(int)]; ok {
		s.byDir.remove(int(planner.DirID), planner.ID)
		delete(s.planners, planner.ID)
	}
}

func NewPlannerStore(db *gosql.DB, table, eventTable string) *PlannerStore {
	impl := &PlannerStore{}
	impl.baseStore = makeBaseStore(
		impl, db, Planner{}, table, PlannerEvent{}, eventTable,
	)
	return impl
}
