package models

import (
	"database/sql"

	"a.yandex-team.ru/drive/library/go/gosql"
)

// PlannerState represents planner state.
type PlannerState struct {
	PlannerID      int    `db:"planner_id"`
	NextTime       NInt64 `db:"next_time"`
	FailureCounter int    `db:"failure_counter"`
}

// PlannerStateStore represents store for planner states.
type PlannerStateStore struct {
	db    *gosql.DB
	table string
}

// NewPlannerStateStore creates new instance of store.
func NewPlannerStateStore(db *gosql.DB, table string) *PlannerStateStore {
	return &PlannerStateStore{db: db, table: table}
}

// GetTx finds planner state by planner ID.
func (s *PlannerStateStore) GetTx(
	tx gosql.Runner, id int,
) (PlannerState, error) {
	var state PlannerState
	names, scans := gosql.StructNameValues(&state, true)
	query, values := s.db.Select(s.table).
		Names(names...).Where(gosql.Column("planner_id").Equal(id)).Build()
	row := tx.QueryRow(query, values...)
	if err := row.Scan(scans...); err != nil {
		return PlannerState{}, err
	}
	return state, nil
}

// Get finds planner state by planner ID.
func (s *PlannerStateStore) Get(id int) (PlannerState, error) {
	return s.GetTx(s.db, id)
}

func (s *PlannerStateStore) All() ([]PlannerState, error) {
	return s.FindTx(s.db, nil)
}

// CreateTx creates planner state with specified ID.
func (s *PlannerStateStore) CreateTx(
	tx gosql.Runner, state PlannerState,
) error {
	names, values := gosql.StructNameValues(state, false)
	query, values := s.db.Insert(s.table).
		Names(names...).Values(values...).Build()
	res, err := tx.Exec(query, values...)
	if err != nil {
		return err
	}
	affected, err := res.RowsAffected()
	if err != nil {
		return err
	}
	if affected != 1 {
		return sql.ErrNoRows
	}
	return nil
}

// UpdateTx update specified planner state.
func (s *PlannerStateStore) UpdateTx(
	tx gosql.Runner, state PlannerState, columns ...string,
) error {
	names, values := gosql.StructNameValues(state, false, "planner_id")
	if len(columns) > 0 {
		allowed := map[string]struct{}{}
		for _, column := range columns {
			allowed[column] = struct{}{}
		}
		newLen := 0
		for i := 0; i < len(names); i++ {
			if _, ok := allowed[names[i]]; !ok {
				continue
			}
			names[newLen] = names[i]
			values[newLen] = values[i]
			newLen++
		}
		names, values = names[:newLen], values[:newLen]
	}
	query, values := s.db.Update(s.table).
		Names(names...).Values(values...).
		Where(gosql.Column("planner_id").Equal(state.PlannerID)).Build()
	res, err := tx.Exec(query, values...)
	if err != nil {
		return err
	}
	affected, err := res.RowsAffected()
	if err != nil {
		return err
	}
	if affected != 1 {
		return sql.ErrNoRows
	}
	return nil
}

func (s *PlannerStateStore) DeleteTx(tx gosql.Runner, id int) error {
	query, values := s.db.Delete(s.table).
		Where(gosql.Column("planner_id").Equal(id)).Build()
	res, err := tx.Exec(query, values...)
	if err != nil {
		return err
	}
	affected, err := res.RowsAffected()
	if err != nil {
		return err
	}
	if affected != 1 {
		return sql.ErrNoRows
	}
	return nil
}

func (s *PlannerStateStore) UpsertTx(
	tx gosql.Runner, state PlannerState, columns ...string,
) error {
	err := s.UpdateTx(tx, state, columns...)
	if err == sql.ErrNoRows {
		return s.CreateTx(tx, state)
	}
	return err
}

func (s *PlannerStateStore) Upsert(state PlannerState, columns ...string) error {
	return gosql.WithTx(s.db, func(tx *sql.Tx) error {
		return s.UpsertTx(tx, state, columns...)
	})
}

// FindTx finds planners with specified query.
func (s *PlannerStateStore) FindTx(
	tx gosql.Runner, where gosql.BoolExpr,
) ([]PlannerState, error) {
	query, values := s.db.Select(s.table).
		Names(gosql.StructNames(PlannerState{})...).Where(where).Build()
	rows, err := tx.Query(query, values...)
	if err != nil {
		return nil, err
	}
	var states []PlannerState
	for rows.Next() {
		var state PlannerState
		if err := rows.Scan(gosql.StructValues(&state, true)...); err != nil {
			return nil, err
		}
		states = append(states, state)
	}
	return states, rows.Err()
}
