package models

import (
	"crypto/aes"
	"crypto/cipher"
	"crypto/rand"
	"database/sql"
	"database/sql/driver"
	"encoding/base64"
	"encoding/hex"
	"encoding/json"
	"fmt"
	"io"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/runner/config"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

type ResourceMeta map[string]interface{}

func (m ResourceMeta) Value() (driver.Value, error) {
	bytes, err := json.Marshal(m)
	if err != nil {
		return nil, err
	}
	return string(bytes), nil
}

func (m *ResourceMeta) Scan(data interface{}) error {
	switch value := data.(type) {
	case string:
		return json.Unmarshal([]byte(value), m)
	case []byte:
		return json.Unmarshal(value, m)
	case nil:
		return nil
	default:
		return fmt.Errorf("unsupported type: %T", value)
	}
}

const sizeMeta = "@Size"
const contentTypeMeta = "@ContentType"

// Resource represents resource.
type Resource struct {
	ID     int64  `db:"id" json:""`
	DirID  NInt   `db:"dir_id" json:",omitempty"`
	TaskID NInt64 `db:"task_id" json:",omitempty"`
	// Title contains title of resource.
	Title string `db:"title" json:""`
	// Meta contains metadata of resource.
	//
	// All meta starting with "@" are reserved by system:
	//   - "@Size" - contains size of resource in bytes;
	//   - "@ContentType" - used for specifying content type of resource.
	Meta ResourceMeta `db:"meta" json:""`
	// Key contains name of data key in MDS.
	Key string `db:"key" json:"-"`
	// Secret contains secret of data in MDS.
	Secret string `db:"secret" json:"-"`
}

// ObjectID returns ID of resource.
func (o Resource) ObjectID() objects.ID {
	return o.ID
}

func (o Resource) clone() Resource {
	return o
}

func (o Resource) SizeMeta() (int64, bool) {
	if meta, ok := o.Meta[sizeMeta]; ok {
		switch value := meta.(type) {
		case int64:
			return value, true
		}
	}
	return 0, false
}

func (o *Resource) SetSize(size int64) {
	if o.Meta == nil {
		o.Meta = ResourceMeta{}
	}
	o.Meta[sizeMeta] = size
}

func (o Resource) ContentTypeMeta() (string, bool) {
	if meta, ok := o.Meta[contentTypeMeta]; ok {
		switch value := meta.(type) {
		case string:
			return value, true
		case []byte:
			return string(value), true
		}
	}
	return "", false
}

func (o *Resource) GenerateKey() error {
	bytes := make([]byte, 20)
	if _, err := rand.Read(bytes); err != nil {
		return err
	}
	o.Key = hex.EncodeToString(bytes)
	return nil
}

func (o *Resource) GenerateSecret() error {
	bytes := make([]byte, 16)
	if _, err := rand.Read(bytes); err != nil {
		return err
	}
	o.Secret = base64.StdEncoding.EncodeToString(bytes)
	return nil
}

type ResourceEvent struct {
	baseEvent
	Resource
}

func (e ResourceEvent) Object() objects.Object {
	return e.Resource
}

func (e ResourceEvent) WithObject(o objects.Object) ObjectEvent {
	e.Resource = o.(Resource)
	return e
}

type ResourceStore struct {
	baseStore
	// ResourceStore directly retrieves data from store.
	stubStoreImpl
	config     config.MDS
	uploader   *s3manager.Uploader
	downloader *s3manager.Downloader
}

// GetTx returns resource with specified ID.
func (m *ResourceStore) GetTx(tx *sql.Tx, id int64) (Resource, error) {
	object, err := m.FindObject(tx, `"id" = $1`, id)
	if err != nil {
		return Resource{}, err
	}
	return object.(Resource), nil
}

// FindByDirTx returns resources with specified dir ID.
func (m *ResourceStore) FindByDirTx(
	tx *sql.Tx, id int,
) ([]Resource, error) {
	list, err := m.FindObjects(tx, `"dir_id" = $1`, id)
	if err != nil {
		return nil, err
	}
	var resources []Resource
	for _, object := range list {
		resources = append(resources, object.(Resource))
	}
	return resources, nil
}

// FindByTaskTx returns resources with specified task ID.
func (m *ResourceStore) FindByTaskTx(
	tx *sql.Tx, id int64,
) ([]Resource, error) {
	list, err := m.FindObjects(tx, `"task_id" = $1`, id)
	if err != nil {
		return nil, err
	}
	var resources []Resource
	for _, object := range list {
		resources = append(resources, object.(Resource))
	}
	return resources, nil
}

func (m *ResourceStore) CreateTx(
	tx *sql.Tx, resource Resource, options ...EventOption,
) (Resource, error) {
	result, err := m.CreateObjectEvent(tx, ResourceEvent{
		makeBaseEvent(CreateEvent, options...),
		resource,
	})
	if err != nil {
		return Resource{}, err
	}
	return result.Object().(Resource), nil
}

func (m *ResourceStore) UpdateTx(
	tx *sql.Tx, resource Resource, options ...EventOption,
) error {
	_, err := m.CreateObjectEvent(tx, ResourceEvent{
		makeBaseEvent(UpdateEvent, options...),
		resource,
	})
	return err
}

func (m *ResourceStore) RemoveTx(
	tx *sql.Tx, id int64, options ...EventOption,
) error {
	_, err := m.CreateObjectEvent(tx, ResourceEvent{
		makeBaseEvent(RemoveEvent, options...),
		Resource{ID: id},
	})
	return err
}

func (m *ResourceStore) Validate(resource Resource) error {
	errList := FieldListError{}
	// If TaskID is zero this means that resource was created by user.
	if resource.DirID == 0 && resource.TaskID == 0 {
		errList = append(errList, FieldError{"DirID", InvalidField})
	}
	if len(resource.Title) < 1 {
		errList = append(errList, FieldError{"Title", TooShortField})
	}
	if len(resource.Title) > 64 {
		errList = append(errList, FieldError{"Title", TooLongField})
	}
	return errList.AsError()
}

func (m *ResourceStore) Upload(
	resource Resource, reader io.Reader,
) error {
	key, err := hex.DecodeString(m.config.Key.Secret())
	if err != nil {
		return err
	}
	block, err := aes.NewCipher(key)
	if err != nil {
		return err
	}
	iv, err := base64.StdEncoding.DecodeString(resource.Secret)
	if err != nil {
		return err
	}
	stream := cipher.NewCFBEncrypter(block, iv)
	_, err = m.uploader.Upload(&s3manager.UploadInput{
		Bucket: aws.String(m.config.Bucket),
		Key:    aws.String(m.config.Prefix + resource.Key),
		Body:   &cipher.StreamReader{S: stream, R: reader},
	})
	return err
}

type fakeWriterAt struct {
	io.Writer
}

func (w fakeWriterAt) WriteAt(p []byte, _ int64) (int, error) {
	return w.Writer.Write(p)
}

func (m *ResourceStore) Download(
	resource Resource, writer io.Writer,
) error {
	key, err := hex.DecodeString(m.config.Key.Secret())
	if err != nil {
		return err
	}
	block, err := aes.NewCipher(key)
	if err != nil {
		return err
	}
	iv, err := base64.StdEncoding.DecodeString(resource.Secret)
	if err != nil {
		return err
	}
	stream := cipher.NewCFBDecrypter(block, iv)
	_, err = m.downloader.Download(
		&fakeWriterAt{cipher.StreamWriter{S: stream, W: writer}},
		&s3.GetObjectInput{
			Bucket: aws.String(m.config.Bucket),
			Key:    aws.String(m.config.Prefix + resource.Key),
		},
	)
	return err
}

const mdsRegion = "yandex"

func NewResourceStore(
	db *gosql.DB, table, eventTable string, mds config.MDS,
) *ResourceStore {
	sess, err := session.NewSession(&aws.Config{
		Endpoint: aws.String(mds.BaseURL),
		Region:   aws.String(mdsRegion),
		Credentials: credentials.NewStaticCredentials(
			mds.AccessKey.Secret(), mds.Secret.Secret(), "",
		),
	})
	if err != nil {
		panic(err)
	}
	downloader := s3manager.NewDownloader(sess)
	downloader.Concurrency = 1
	impl := &ResourceStore{
		config:     mds,
		uploader:   s3manager.NewUploader(sess),
		downloader: downloader,
	}
	impl.baseStore = makeBaseStore(
		stubStoreImpl{}, db, Resource{}, table, ResourceEvent{}, eventTable,
	)
	return impl
}
