package models

import (
	"database/sql"
	"encoding/json"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

const (
	// Role permissions.
	ObserveRolePermission = "observe_role"
	CreateRolePermission  = "create_role"
	UpdateRolePermission  = "update_role"
	DeleteRolePermission  = "delete_role"
	// Account role permissions
	ObserveAccountRolePermission = "observe_account_role"
	CreateAccountRolePermission  = "create_account_role"
	DeleteAccountRolePermission  = "delete_account_role"
	// Host permissions.
	ObserveHostPermission = "observe_host"
	UpdateHostPermission  = "update_host"
	// Dir permissions.
	ObserveNodePermission = "observe_node"
	CreateNodePermission  = "create_node"
	UpdateNodePermission  = "update_node"
	DeleteNodePermission  = "delete_node"
	// Dir role permissions.
	ObserveNodeRolePermission = "observe_node_role"
	CreateNodeRolePermission  = "create_node_role"
	DeleteNodeRolePermission  = "delete_node_role"
	// Action permissions.
	ObserveActionPermission = "observe_action"
	CreateActionPermission  = "create_action"
	UpdateActionPermission  = "update_action"
	DeleteActionPermission  = "delete_action"
	RunActionPermission     = "run_action"
	// Planner permissions.
	ObservePlannerPermission = "observe_planner"
	CreatePlannerPermission  = "create_planner"
	UpdatePlannerPermission  = "update_planner"
	DeletePlannerPermission  = "delete_planner"
	RunPlannerPermission     = "run_planner"
	// Secret permissions.
	ObserveSecretPermission = "observe_secret"
	CreateSecretPermission  = "create_secret"
	UpdateSecretPermission  = "update_secret"
	DeleteSecretPermission  = "delete_secret"
	// Config permissions.
	ObserveConfigPermission = "observe_config"
	CreateConfigPermission  = "create_config"
	UpdateConfigPermission  = "update_config"
	DeleteConfigPermission  = "delete_config"
	// Task permissions.
	ObserveTaskPermission = "observe_task"
	AbortTaskPermission   = "abort_task"
	// LegacyAuthPermission.
	LegacyAuthPermission = "legacy_auth"
)

var allPermissions = map[string]struct{}{
	ObserveRolePermission:        {},
	CreateRolePermission:         {},
	UpdateRolePermission:         {},
	DeleteRolePermission:         {},
	ObserveAccountRolePermission: {},
	CreateAccountRolePermission:  {},
	DeleteAccountRolePermission:  {},
	ObserveHostPermission:        {},
	UpdateHostPermission:         {},
	ObserveNodePermission:        {},
	CreateNodePermission:         {},
	UpdateNodePermission:         {},
	DeleteNodePermission:         {},
	ObserveNodeRolePermission:    {},
	CreateNodeRolePermission:     {},
	DeleteNodeRolePermission:     {},
	ObserveActionPermission:      {},
	CreateActionPermission:       {},
	UpdateActionPermission:       {},
	DeleteActionPermission:       {},
	RunActionPermission:          {},
	ObservePlannerPermission:     {},
	CreatePlannerPermission:      {},
	UpdatePlannerPermission:      {},
	DeletePlannerPermission:      {},
	RunPlannerPermission:         {},
	ObserveSecretPermission:      {},
	CreateSecretPermission:       {},
	UpdateSecretPermission:       {},
	DeleteSecretPermission:       {},
	ObserveConfigPermission:      {},
	CreateConfigPermission:       {},
	UpdateConfigPermission:       {},
	DeleteConfigPermission:       {},
	ObserveTaskPermission:        {},
	AbortTaskPermission:          {},
	LegacyAuthPermission:         {},
}

// HasPermission checks that permission exists.
func HasPermission(name string) bool {
	_, ok := allPermissions[name]
	return ok
}

type Role struct {
	ID          int    `db:"id"`
	Name        string `db:"name"`
	Permissions JSON   `db:"permissions"`
}

func (o Role) ObjectID() objects.ID {
	return o.ID
}

func (o Role) Clone() Role {
	o.Permissions = o.Permissions.Clone()
	return o
}

func (o Role) GetPermissions() ([]string, error) {
	var permissions []string
	if err := json.Unmarshal(o.Permissions, &permissions); err != nil {
		return nil, err
	}
	return permissions, nil
}

func (o *Role) SetPermissions(permissions []string) error {
	data, err := json.Marshal(permissions)
	if err != nil {
		return err
	}
	o.Permissions = data
	return nil
}

type RoleEvent struct {
	baseEvent
	Role
}

func (e RoleEvent) Object() objects.Object {
	return e.Role
}

func (e RoleEvent) WithObject(o objects.Object) ObjectEvent {
	e.Role = o.(Role)
	return e
}

type RoleStore struct {
	baseStore
	roles  map[int]Role
	byName indexString
}

func (s *RoleStore) Get(id int) (Role, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if role, ok := s.roles[id]; ok {
		return role.Clone(), nil
	}
	return Role{}, sql.ErrNoRows
}

func (s *RoleStore) GetByName(name string) (Role, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	for id := range s.byName[name] {
		if role, ok := s.roles[id]; ok {
			return role.Clone(), nil
		}
	}
	return Role{}, sql.ErrNoRows
}

func (s *RoleStore) All() ([]Role, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var roles []Role
	for _, role := range s.roles {
		roles = append(roles, role.Clone())
	}
	return roles, nil
}

func (s *RoleStore) CreateTx(
	tx gosql.Runner, role *Role, options ...EventOption,
) error {
	result, err := s.CreateObjectEvent(tx, RoleEvent{
		makeBaseEvent(CreateEvent, options...),
		*role,
	})
	if err != nil {
		return err
	}
	*role = result.Object().(Role)
	return nil
}

func (s *RoleStore) UpdateTx(
	tx gosql.Runner, role Role, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, RoleEvent{
		makeBaseEvent(UpdateEvent, options...),
		role,
	})
	return err
}

func (s *RoleStore) RemoveTx(
	tx gosql.Runner, id int, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, RoleEvent{
		makeBaseEvent(RemoveEvent, options...),
		Role{ID: id},
	})
	return err
}

func (s *RoleStore) reset() {
	s.roles = map[int]Role{}
	s.byName = indexString{}
}

func (s *RoleStore) onCreateObject(o objects.Object) {
	role := o.(Role)
	s.roles[role.ID] = role
	s.byName.create(role.Name, role.ID)
}

func (s *RoleStore) onRemoveObject(id objects.ID) {
	if role, ok := s.roles[id.(int)]; ok {
		s.byName.remove(role.Name, role.ID)
		delete(s.roles, role.ID)
	}
}

func NewRoleStore(db *gosql.DB, table, eventTable string) *RoleStore {
	impl := &RoleStore{}
	impl.baseStore = makeBaseStore(
		impl, db, Role{}, table, RoleEvent{}, eventTable,
	)
	return impl
}
