package models

import (
	"context"
	"database/sql"
	"encoding/json"
	"fmt"

	"a.yandex-team.ru/drive/library/go/gosql"
	"a.yandex-team.ru/drive/runner/config"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/yav"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/zootopia/library/go/db/objects"
)

// SecretType represents type of secret.
type SecretType string

const (
	// YavSecret represents secret from yav.yandex-team.ru
	YavSecret SecretType = "Yav"
)

type YavSecretData struct {
	SecretID  string `json:""`
	VersionID string `json:",omitempty"`
	Key       string `json:""`
}

type Secret struct {
	ID    int        `db:"id" json:""`
	DirID NInt       `db:"dir_id" json:",omitempty"`
	Title string     `db:"title" json:""`
	Type  SecretType `db:"type" json:""`
	Data  JSON       `db:"data" json:""`
}

func (o Secret) ObjectID() objects.ID {
	return o.ID
}

func (o Secret) Clone() Secret {
	o.Data = o.Data.Clone()
	return o
}

type SecretEvent struct {
	baseEvent
	Secret
}

func (e SecretEvent) Object() objects.Object {
	return e.Secret
}

func (e SecretEvent) WithObject(o objects.Object) ObjectEvent {
	e.Secret = o.(Secret)
	return e
}

type SecretStore struct {
	baseStore
	secrets map[int]Secret
	byDir   indexInt
	yav     yav.Client
	config  config.Yav
}

func (s *SecretStore) Get(id int) (Secret, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	if secret, ok := s.secrets[id]; ok {
		return secret.Clone(), nil
	}
	return Secret{}, sql.ErrNoRows
}

func (s *SecretStore) FindByDir(id int) ([]Secret, error) {
	s.mutex.RLock()
	defer s.mutex.RUnlock()
	var secrets []Secret
	for id := range s.byDir[id] {
		if secret, ok := s.secrets[id]; ok {
			secrets = append(secrets, secret.Clone())
		}
	}
	return secrets, nil
}

func (s *SecretStore) CreateTx(
	tx gosql.Runner, secret *Secret, options ...EventOption,
) error {
	result, err := s.CreateObjectEvent(tx, SecretEvent{
		makeBaseEvent(CreateEvent, options...),
		*secret,
	})
	if err != nil {
		return err
	}
	*secret = result.Object().(Secret)
	return nil
}

func (s *SecretStore) UpdateTx(
	tx gosql.Runner, secret Secret, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, SecretEvent{
		makeBaseEvent(UpdateEvent, options...),
		secret,
	})
	return err
}

func (s *SecretStore) RemoveTx(
	tx gosql.Runner, id int, options ...EventOption,
) error {
	_, err := s.CreateObjectEvent(tx, SecretEvent{
		makeBaseEvent(RemoveEvent, options...),
		Secret{ID: id},
	})
	return err
}

// SecretValue returns value of specified secret.
func (s *SecretStore) SecretValue(secret Secret) (string, error) {
	switch secret.Type {
	case YavSecret:
		var data YavSecretData
		if err := json.Unmarshal(secret.Data, &data); err != nil {
			return "", err
		}
		versionID := data.SecretID
		if data.VersionID != "" {
			versionID = data.VersionID
		}
		resp, err := getYavVersion(context.TODO(), s.yav, versionID)
		if err != nil {
			return "", err
		}
		for _, value := range resp.Version.Values {
			if value.Key == data.Key {
				return value.Value, nil
			}
		}
		return "", fmt.Errorf("unable to find key %q", data.Key)
	default:
		return "", fmt.Errorf("unsupported secret type: %v", secret.Type)
	}
}

const yavRetries = 3

func getYavVersion(
	context context.Context, client yav.Client, versionID string,
) (*yav.GetVersionResponse, error) {
	var resp *yav.GetVersionResponse
	var err error
	for i := 0; i < yavRetries; i++ {
		select {
		case <-context.Done():
			return nil, context.Err()
		default:
			resp, err = client.GetVersion(context, versionID)
			if err == nil {
				return resp, nil
			}
		}
	}
	return nil, err
}

func (s *SecretStore) Validate(secret Secret) error {
	errList := FieldListError{}
	if secret.DirID == 0 {
		errList = append(errList, FieldError{"DirID", InvalidField})
	}
	if len(secret.Title) < 1 {
		errList = append(errList, FieldError{"Title", TooShortField})
	}
	if len(secret.Title) > 64 {
		errList = append(errList, FieldError{"Title", TooLongField})
	}
	switch secret.Type {
	case YavSecret:
	default:
		errList = append(errList, FieldError{"Type", InvalidField})
	}
	if _, err := s.SecretValue(secret); err != nil {
		errList = append(errList, FieldError{"Data", InvalidField})
	}
	return errList.AsError()
}

func (s *SecretStore) reset() {
	s.secrets = map[int]Secret{}
	s.byDir = indexInt{}
}

func (s *SecretStore) onCreateObject(o objects.Object) {
	secret := o.(Secret)
	s.secrets[secret.ID] = secret
	s.byDir.create(int(secret.DirID), secret.ID)
}

func (s *SecretStore) onRemoveObject(id objects.ID) {
	if secret, ok := s.secrets[id.(int)]; ok {
		s.byDir.remove(int(secret.DirID), secret.ID)
		delete(s.secrets, secret.ID)
	}
}

func NewSecretStore(
	db *gosql.DB, table, eventTable string, cfg config.Yav,
	logger log.Logger,
) (*SecretStore, error) {
	yavClient, err := httpyav.NewClient(
		httpyav.WithOAuthToken(cfg.Token.Secret()),
		httpyav.WithLogger(logger),
	)
	if err != nil {
		return nil, err
	}
	impl := &SecretStore{config: cfg, yav: yavClient}
	impl.baseStore = makeBaseStore(
		impl, db, Secret{}, table, SecretEvent{}, eventTable,
	)
	return impl, nil
}
