package models

import (
	"fmt"
	"time"

	"github.com/jmoiron/sqlx"

	"a.yandex-team.ru/drive/library/go/gosql"
)

type TaskLogType string

const (
	StdoutTaskLog TaskLogType = "Stdout"
	StderrTaskLog TaskLogType = "Stderr"
	SystemTaskLog TaskLogType = "System"
)

type TaskLog struct {
	ID         int64       `db:"id" json:""`
	TaskID     int64       `db:"task_id" json:""`
	Type       TaskLogType `db:"type" json:""`
	Lines      string      `db:"lines" json:""`
	CreateTime int64       `db:"create_time" json:""`
}

type TaskLogStore struct {
	db    *gosql.DB
	dbx   *sqlx.DB
	table string
}

func (s *TaskLogStore) Create(m *TaskLog) error {
	m.CreateTime = time.Now().Unix()
	return s.dbx.Get(
		&m.ID,
		fmt.Sprintf(
			`INSERT INTO "%s" (task_id, type, lines, create_time) VALUES ($1, $2, $3, $4) RETURNING id`,
			s.table,
		),
		m.TaskID, m.Type, m.Lines, m.CreateTime,
	)
}

func (s *TaskLogStore) FindByTask(id int64) (logs []TaskLog, err error) {
	err = s.dbx.Select(
		&logs,
		fmt.Sprintf(
			`SELECT id, task_id, type, lines, create_time FROM "%s" WHERE task_id = $1 ORDER BY id`,
			s.table,
		),
		id,
	)
	return
}

func NewTaskLogStore(db *gosql.DB, table string) *TaskLogStore {
	return &TaskLogStore{db: db, dbx: sqlx.NewDb(db.DB, "pgx"), table: table}
}
