export interface IHost {
	id: number;
	name: string;
	ping_time: number;
	config: any;
	state: any;
	running: boolean;
};

export interface IGetHostsResp {
	hosts: IHost[];
};

export enum OptionType {
	INTEGER = "Integer",
	STRING = "String",
	CONFIG = "Config",
	SECRET = "Secret",
}

export type IOptionValue = number | string | any;

// IStringOptionSettings represents settings for option of type "String".
export interface IStringOptionSettings {
	Multiline?: boolean;
}

export type IOptionSettings = IStringOptionSettings | any;

// IOption represents option.
export interface IOption {
	Type: OptionType;
	Title?: string;
	Description?: string;
	Settings?: IOptionSettings;
	Value?: IOptionValue;
	Required: boolean;
	Visible: boolean;
	Editable: boolean;
}

export type IOptions = {[key: string]: IOption};

export interface ITaskOption {
	Type: OptionType;
	Value: IOptionValue;
}

export type ITaskOptions = {[key: string]: ITaskOption};

export interface IAction {
	id: number;
	node_id?: number;
	title: string;
	description?: string;
	options: IOptions;
};

export interface ITask {
	id: number;
	action_id: number;
	planner_id?: number;
	options: ITaskOptions;
	status: string;
	create_time: number;
	update_time: number;
};

export interface ITasksResp {
	tasks: ITask[] | null;
	next_begin?: number;
};

export interface IPlannerSettings {
	RepeatDelay?: number;
	FailureDelay?: number;
	Schedule?: string;
	FailureMails?: string[];
	ListenSuccess?: number[];
	ListenFailure?: number[];
	Enabled: boolean;
	AllowOverlaps: boolean;
	notify_failure_threshold?: number;
};

export interface IPlanner {
	id: number;
	action_id?: number;
	node_id?: number;
	title: string;
	description: string;
	options: ITaskOptions;
	settings: IPlannerSettings;
	next_time?: number;
};

export type YavSecretData = {
	Key: string;
	SecretID: string;
	VersionID?: string;
};

export type Secret = {
	id: number;
	node_id?: number;
	title: string;
	kind: string;
	data?: YavSecretData;
}

export type Config = {
	id: number;
	node_id?: number;
	title: string;
	description: string;
	data: string;
}

export interface Node {
	id: number;
	node_id?: number;
	name: string;
	description?: string;
	inherit_roles: boolean;
};

export type InvalidField = {
	message: string;
};

export type InvalidFields = {[key: string]: InvalidField};

export type ErrorResponse = {
	message: string;
	invalid_fields?: InvalidFields;
	missing_permissions: string[];
};

const parseResp = (promise: Promise<Response>) => {
	return promise
		.then(resp => Promise.all([resp, resp.json()]))
		.then(([resp, json]) => {
			if (!resp.ok) {
				throw json;
			}
			return json;
		});
};

const POST_JSON_HEADERS = {
	"Content-Type": "application/json; charset=UTF-8",
};

export type UpdateNodeForm = {
	node_id?: number;
	name?: string;
	description?: string;
	inherit_roles?: boolean;
};

export const createNode = (form: UpdateNodeForm) => {
	return parseResp(fetch("/api/v0/nodes", {
		method: "POST",
		headers: POST_JSON_HEADERS,
		body: JSON.stringify(form),
	}));
};

export const observeNode = (id: number) => {
	return parseResp(fetch(`/api/v0/nodes/${id}`, {
		method: "GET",
	}));
};

export const updateNode = (id: number, form: UpdateNodeForm) => {
	return parseResp(fetch(`/api/v0/nodes/${id}`, {
		method: "PATCH",
		headers: POST_JSON_HEADERS,
		body: JSON.stringify(form),
	}));
};

export const observeAction = (id: number) => {
	return parseResp(fetch(`/api/v0/actions/${id}`, {
		method: "GET",
	}));
};

export type UpdateActionForm = {
	node_id?: number;
	title?: string;
	description?: string;
	options?: IOptions;
};

export const createAction = (form: UpdateActionForm) => {
	return parseResp(fetch(`/api/v0/actions`, {
		method: "POST",
		headers: POST_JSON_HEADERS,
		body: JSON.stringify(form),
	}));
};

export const updateAction = (id: number, form: UpdateActionForm) => {
	return parseResp(fetch(`/api/v0/actions/${id}`, {
		method: "PATCH",
		headers: POST_JSON_HEADERS,
		body: JSON.stringify(form),
	}));
};

export const observeSecret = (id: number) => {
	return parseResp(fetch(`/api/v0/secrets/${id}`, {
		method: "GET",
	}));
};

export type UpdateSecretForm = {
	node_id?: number;
	title?: string;
	kind?: string;
	data?: YavSecretData;
};

export const createSecret = (form: UpdateSecretForm) => {
	return parseResp(fetch(`/api/v0/secrets`, {
		method: "POST",
		headers: POST_JSON_HEADERS,
		body: JSON.stringify(form),
	}));
};

export const updateSecret = (id: number, form: UpdateSecretForm) => {
	return parseResp(fetch(`/api/v0/secrets/${id}`, {
		method: "PATCH",
		headers: POST_JSON_HEADERS,
		body: JSON.stringify(form),
	}));
};

export const observeConfig = (id: number) => {
	return parseResp(fetch(`/api/v0/configs/${id}`, {
		method: "GET",
	}));
};

export type UpdateConfigForm = {
	node_id?: number;
	title?: string;
	description?: string;
	data?: string;
};

export const createConfig = (form: UpdateConfigForm) => {
	return parseResp(fetch(`/api/v0/configs`, {
		method: "POST",
		headers: POST_JSON_HEADERS,
		body: JSON.stringify(form),
	}));
};

export const updateConfig = (id: number, form: UpdateConfigForm) => {
	return parseResp(fetch(`/api/v0/configs/${id}`, {
		method: "PATCH",
		headers: POST_JSON_HEADERS,
		body: JSON.stringify(form),
	}));
};

export const abortTask = (id: number) => {
	return parseResp(fetch(`/api/v0/tasks/${id}/abort`, {
		method: "POST",
		headers: POST_JSON_HEADERS,
	}));
};
