import { FC, useState } from "react";
import { Link } from "react-router-dom";
import Block from "../Block";
import Button from "../Button";
import Option from "../Option";
import OptionGroup from "../OptionGroup";
import OptionEditor from "../OptionEditor";
import { OptionType } from "../../api/actions";
import { ErrorResponse, IAction, IOption, IOptions } from "../../api";
import Notice from "../Notice";
import Input from "../Input";
import Field from "../Field";

type OptionsBlockProps = {
    options: IOptions;
    onOptionsChange?(data: IOptions): void;
};

export const OptionsEditorBlock: FC<OptionsBlockProps> = props => {
    const { options, onOptionsChange } = props;
    const [code, setCode] = useState<string>("");
    const onAddOption = () => {
        if (!code || options[code]) {
            return;
        }
        const newOption: IOption = {
            Type: OptionType.STRING,
            // Title: "",
            Required: false,
            Editable: false,
            Visible: false,
        };
        onOptionsChange && onOptionsChange({ ...options, [code]: newOption });
        setCode("");
    };
    return <>
        <OptionGroup title="Options">
            {Object.entries(options).map(([code, option], index) => {
                const deleteOption = () => {
                    const newOptions = { ...(options || {}) };
                    delete newOptions[code];
                    onOptionsChange && onOptionsChange(newOptions);
                };
                return <OptionEditor
                    key={index} optionCode={code} option={option}
                    onOptionChange={(value) => onOptionsChange && onOptionsChange({ ...options, [code]: value })}
                    onDeleteClick={deleteOption}
                />;
            })}
        </OptionGroup>
        <OptionGroup>
            <div className="ui-option-editor">
                <div className="middle">
                    <Field title="New option code:">
                        <Input value={code} onValueChange={setCode} />
                        <Button type="button" onClick={onAddOption}>Add</Button>
                    </Field>
                </div>
            </div>
        </OptionGroup>
    </>;
};

type ActionEditorBlockProps = {
    action?: IAction;
    parentID?: number;
    error?: ErrorResponse;
    onSubmit?(action: IAction): void;
};

const defaultOptions: IOptions = {
    "@command": {
        Type: OptionType.STRING,
        Title: "Command",
        Description: "Command to execute",
        Required: true,
        Visible: false,
        Editable: false,
    },
    "@layers": {
        Type: OptionType.STRING,
        Title: "Layers",
        Description: "Container layers",
        Value: "sandbox:2527306889",
        Required: true,
        Visible: false,
        Editable: false,
        Settings: {
            Multiline: true,
        }
    },
    "@environ": {
        Type: OptionType.STRING,
        Title: "Environment",
        Description: "Environment variables",
        Required: false,
        Visible: false,
        Editable: false,
        Settings: {
            Multiline: true,
        }
    },
    "@files": {
        Type: OptionType.STRING,
        Title: "Files",
        Description: "List of files to download",
        Required: false,
        Visible: false,
        Editable: false,
        Settings: {
            Multiline: true,
        }
    }
};

const ActionEditorBlock: FC<ActionEditorBlockProps> = props => {
    const { action, error, parentID, onSubmit } = props;
    const [title, setTitle] = useState<string>(action?.title || "");
    const [description, setDescription] = useState<string>(action?.description || "");
    const [options, setOptions] = useState<IOptions>(action?.options || defaultOptions)
    return <form onSubmit={(event) => {
        event.preventDefault();
        onSubmit && onSubmit({
            id: action?.id || 0,
            node_id: action?.node_id ?? parentID,
            title: title,
            description: description,
            options: options,
        });
    }}>
        <Block
            className="ui-block-action"
            header={<h2 className="title">{action ? "Edit action" : "Create action"}</h2>}
            footer={action ? <>
                <Button type="submit">Update</Button>
                <Link to={`/actions/${action.id}`}><Button>Back</Button></Link>
            </> : <>
                <Button type="submit">Create</Button>
                <Link to={`/nodes/${parentID}`}><Button>Back</Button></Link>
            </>}
        >
            {error && error.message && <Notice>{error.message}</Notice>}
            <OptionGroup title="Info">
                <Option
                    option={{
                        Type: OptionType.STRING,
                        Title: "Title",
                        Description: "Title of action.",
                        Value: title,
                        Required: true,
                        Visible: true,
                        Editable: true,
                    }}
                    optionCode="title"
                    onValueChange={setTitle}
                    error={error && error.invalid_fields && error.invalid_fields["title"]}
                />
                <Option
                    option={{
                        Type: OptionType.STRING,
                        Title: "Description",
                        Description: "Description for node.",
                        Value: description,
                        Required: false,
                        Visible: true,
                        Editable: true,
                        Settings: {
                            Multiline: true,
                        },
                    }}
                    optionCode="description"
                    onValueChange={setDescription}
                    error={error && error.invalid_fields && error.invalid_fields["description"]}
                />
            </OptionGroup>
            <OptionsEditorBlock options={options} onOptionsChange={setOptions} />
        </Block>
    </form>;
};

export default ActionEditorBlock;
