import React, {FC, useEffect, useState} from "react";
import {Link} from "react-router-dom";
import Block from "../Block";
import {IAction, ITasksResp, ITask} from "../../api";
import Button from "../Button";
import "./index.scss";

export interface IActionTasksBlockProps {
	action: IAction;
};

const formatDuration = (duration: number): string => {
	const seconds = duration % 60;
	duration = (duration - seconds) / 60;
	const minutes = duration % 60;
	duration = (duration - minutes) / 60;
	const hours = duration % 24;
	duration = (duration - hours) / 24;
	let result: string = "";
	if (duration > 0) {
		result += `${duration}d`;
	}
	if (hours > 0) {
		result += `${hours}h`;
	}
	if (minutes > 0) {
		result += `${minutes}m`;
	}
	if (seconds > 0 || !result) {
		result += `${seconds}s`;
	}
	return result;
};

const ActionTasksBlock: FC<IActionTasksBlockProps> = props => {
	const {action} = props;
	const [tasks, setTasks] = useState<ITasksResp>();
	const [loading, setLoading] = useState<boolean>(false);
	const loadMoreTasks = () => {
		setLoading(true);
		tasks && fetch(`/api/v0/actions/${action.id}/tasks?begin=${tasks.next_begin}`)
			.then(response => response.json())
			.then((newTasks: ITasksResp) => {
				setTasks({
					tasks: (tasks.tasks || []).concat(newTasks.tasks || []),
					next_begin: newTasks.next_begin,
				});
				setLoading(false);
			})
			.catch(console.log)
	};
	useEffect(() => {
		setTasks(undefined);
		fetch(`/api/v0/actions/${action.id}/tasks`)
			.then(response => response.json())
			.then((tasks: ITasksResp) => {
				setTasks(tasks);
			})
			.catch(console.log)
	}, [action.id]);
	if (!tasks) {
		return <div/>;
	}
	return <Block
		header={<h2 className="title">Action tasks</h2>}
		className="ui-block-tasks"
		footer={tasks.next_begin && <>{!loading ? <Button onClick={loadMoreTasks}>Load more</Button> : <Button>Loading...</Button>}</>}
	>
		<ul>
			{tasks.tasks && tasks.tasks.map((task: ITask, index: number) => <li className="task" key={index}>
				<span className={`task-status ${task.status}`}>{task.status}</span>
				<Link to={`/tasks/${task.id}`} className="task-id">{task.id}</Link>
				<span className="task-start-time">{(new Date(task.create_time * 1000)).toString()}</span>
				<span className="task-duration">{formatDuration(task.update_time - task.create_time)}</span>
			</li>)}
		</ul>
	</Block>;
};

export default ActionTasksBlock;
