import { FC, useState } from "react";
import { Link } from "react-router-dom";
import Block from "../Block";
import Button from "../Button";
import Option from "../Option";
import OptionGroup from "../OptionGroup";
import { ConfigOptionsBlock } from "../ConfigBlock";
import { OptionType } from "../../api/actions";
import { ErrorResponse, Config } from "../../api";

export interface IConfigEditorBlockProps {
    config?: Config;
    parentID?: number;
    error?: ErrorResponse;
    onSubmit?(config: Config): void;
};

const ConfigEditorBlock: FC<IConfigEditorBlockProps> = props => {
    const { config, parentID, onSubmit } = props;
    const [title, setTitle] = useState<string>(config?.title ?? "");
    const [description, setDescription] = useState<string>(config?.description ?? "");
    const [data, setData] = useState<string>(config?.data ?? "");
    return <form onSubmit={(event) => {
        event.preventDefault();
        onSubmit && onSubmit({
            id: config?.id ?? 0,
            node_id: config?.node_id ?? parentID,
            title: title,
            description: description,
            data: data,
        });
    }}>
        <Block
            header={<h2 className="title">{config ? <>Edit config <span className="object-id">ID: {config.id}</span></> : "Create config"}</h2>}
            footer={config ? <>
                <Button type="submit">Save</Button>
                <Link to={`/configs/${config.id}`}><Button>Back</Button></Link>
            </> : <>
                <Button type="submit">Create</Button>
                <Link to={`/nodes/${parentID}`}><Button>Back</Button></Link>
            </>}
        >
            <OptionGroup title="Info">
                <Option
                    option={{
                        Type: OptionType.STRING,
                        Title: "Title",
                        Description: "Name of config.",
                        Value: title,
                        Required: true,
                        Visible: true,
                        Editable: true,
                    }}
                    optionCode="Title"
                    onValueChange={setTitle}
                />
                <Option
                    option={{
                        Type: OptionType.STRING,
                        Title: "Description",
                        Description: "Description of config.",
                        Value: description,
                        Required: true,
                        Visible: true,
                        Editable: true,
                        Settings: {
                            Multiline: true,
                        }
                    }}
                    optionCode="Description"
                    onValueChange={setDescription}
                />
            </OptionGroup>
            <ConfigOptionsBlock data={data} onDataChange={setData} editable={true} />
        </Block>
    </form>;
};

export default ConfigEditorBlock;
