import { FC, useState } from "react";
import { Link } from "react-router-dom";
import Block from "../Block";
import Button from "../Button";
import Option from "../Option";
import OptionGroup from "../OptionGroup";
import { OptionType } from "../../api/actions";
import { ErrorResponse, Node } from "../../api";
import Notice from "../Notice";

type NodeEditorBlockProps = {
    node?: Node;
    parentID: number;
    error?: ErrorResponse;
    onSubmit?(node: Node): void;
};

const NodeEditorBlock: FC<NodeEditorBlockProps> = props => {
    const { node, error, parentID, onSubmit } = props;
    const [name, setName] = useState<string>(node?.name || "");
    const [description, setDescription] = useState<string>(node?.description || "");
    return <form onSubmit={(event) => {
        event.preventDefault();
        onSubmit && onSubmit({
            id: node?.id || 0,
            node_id: node?.node_id ?? parentID,
            name: name,
            description: description,
            inherit_roles: node?.inherit_roles || true,
        });
    }}>
        <Block
            className="ui-block-node"
            header={<h2 className="title">{node ? "Edit node" : "Create node"}</h2>}
            footer={node ? <>
                <Button type="submit">Update</Button>
                <Link to={`/nodes/${node.id}`}><Button>Back</Button></Link>
            </> : <>
                <Button type="submit">Create</Button>
                <Link to={`/nodes/${parentID}`}><Button>Back</Button></Link>
            </>}
        >
            {error && error.message && <Notice>{error.message}</Notice>}
            <OptionGroup title="Info">
                <Option
                    option={{
                        Type: OptionType.STRING,
                        Title: "Name",
                        Description: "Name of node.",
                        Value: name,
                        Required: true,
                        Visible: true,
                        Editable: true,
                    }}
                    optionCode="name"
                    onValueChange={setName}
                    error={error && error.invalid_fields && error.invalid_fields["name"]}
                />
                <Option
                    option={{
                        Type: OptionType.STRING,
                        Title: "Description",
                        Description: "Description for node.",
                        Value: description,
                        Required: false,
                        Visible: true,
                        Editable: true,
                        Settings: {
                            Multiline: true,
                        },
                    }}
                    optionCode="description"
                    onValueChange={setDescription}
                    error={error && error.invalid_fields && error.invalid_fields["description"]}
                />
            </OptionGroup>
        </Block>
    </form>;
};

export default NodeEditorBlock;
