import { FC, useEffect, useState } from "react";
import Input from "../Input";
import Textarea from "../Textarea";
import { InvalidField } from "../../api";
import { IOption, IOptionValue, OptionType } from "../../api/actions";
import { ISecret } from "../../api/secrets";
import { Link } from "react-router-dom";
import Field from "../Field";
import "./index.scss";

// IOptionProps represents props for Option component.
export type OptionProps = {
	option: IOption;
	optionCode: string;
	onValueChange?(value: IOptionValue): void;
	error?: InvalidField;
}

// StringOptionValue represents value for option of type "String".
const StringOptionValue: FC<OptionProps> = props => {
	const { option, optionCode, onValueChange } = props;
	const { Value = undefined, Settings = undefined, Editable = false, Required = false } = option;
	if (Settings && Settings.Multiline) {
		return <Textarea
			name={optionCode}
			disabled={!Editable}
			required={Required}
			value={Value || ""}
			onValueChange={(value) => onValueChange && onValueChange(value || null)}
		/>;
	}
	return <Input
		type="text"
		name={optionCode}
		disabled={!Editable}
		required={Required}
		value={Value || ""}
		onValueChange={(value) => onValueChange && onValueChange(value || null)}
	/>;
};

const SecretOptionValue: FC<OptionProps> = props => {
	const { option, optionCode, onValueChange } = props;
	const { Value = undefined, Editable = false, Required = false } = option;
	const [secret, setSecret] = useState<ISecret>();
	useEffect(() => {
		setSecret(undefined);
		Value && fetch(`/api/secrets/${Value}`)
			.then(resp => resp.json())
			.then(setSecret)
			.catch(console.log);
	}, [Value]);
	return <>
		<Input
			type="number"
			name={optionCode}
			disabled={!Editable}
			required={Required}
			value={Value || Value === 0 ? String(Value) : ""}
			onValueChange={(value) => onValueChange && onValueChange(value ? Number(value) : null)}
		/>
		{secret && <Link to={`/secrets/${secret.ID}`}>Secret: {secret.Title}</Link>}
	</>;
};

const ConfigOptionValue: FC<OptionProps> = props => {
	const { option, optionCode, onValueChange } = props;
	const { Value = undefined, Editable = false, Required = false } = option;
	const [config, setConfig] = useState<{ ID: string, Title: string }>();
	useEffect(() => {
		setConfig(undefined);
		Value && fetch(`/api/configs/${Value}`)
			.then(resp => resp.json())
			.then(setConfig)
			.catch(console.log);
	}, [Value]);
	return <>
		<Input
			type="number"
			name={optionCode}
			disabled={!Editable}
			required={Required}
			value={Value || Value === 0 ? String(Value) : ""}
			onValueChange={(value) => onValueChange && onValueChange(value ? Number(value) : null)}
		/>
		{config && <Link to={`/configs/${config.ID}`}>Config: {config.Title}</Link>}
	</>;
};

export const OptionValue: FC<OptionProps> = props => {
	const { option, optionCode, onValueChange } = props;
	const { Type, Value = undefined, Editable = false, Required = false } = option;
	switch (Type) {
		case OptionType.STRING:
			return <StringOptionValue {...props} />;
		case OptionType.INTEGER:
			return <Input
				type="number"
				name={optionCode}
				disabled={!Editable}
				required={Required}
				value={Value || Value === 0 ? String(Value) : ""}
				onValueChange={(value) => onValueChange && onValueChange(value ? Number(value) : null)}
			/>;
		case OptionType.SECRET:
			return <SecretOptionValue {...props} />;
		case OptionType.CONFIG:
			return <ConfigOptionValue {...props} />;
	}
	return <span>Unsupported type {Type}</span>;
};

// Option represents option component for option.
const Option: FC<OptionProps> = props => {
	const { option, optionCode, error } = props;
	const { Type, Title = undefined, Description = undefined } = option;
	return <div className="ui-option">
		<div className="left">
			<span className="title" title={optionCode}>{Title || <code>{optionCode}</code>}:</span>
		</div>
		<div className="right">{Type}</div>
		<Field className="middle" description={Description} error={error?.message}>
			<OptionValue {...props} />
		</Field>
	</div>;
};

export default Option;
