import { FC } from "react";
import Input from "../Input";
import Textarea from "../Textarea";
import Field from "../Field";
import { IOption, OptionType } from "../../api/actions";
import { OptionValue } from "../Option";
import "./index.scss";

type OptionEditorProps = {
    optionCode: string;
    option: IOption;
    onOptionChange?(option: IOption): void;
    onDeleteClick?(): void;
};

type OptionSettingsProps = {
    kind: OptionType;
    settings?: any;
    onSettingsChange?(settings: any): void;
};

const StringOptionSettings: FC<OptionSettingsProps> = props => {
    const { settings, onSettingsChange } = props;
    const visualMode = settings?.Multiline ? "textarea" : "input";
    return <>
        <label>
            Visual mode:
            <select name="mode" className="i-select" value={visualMode} onChange={event => { onSettingsChange && onSettingsChange({ ...settings, Multiline: event.target.value === "textarea" }) }}>
                <option value="input">Input</option>
                <option value="textarea">Textarea</option>
            </select>
        </label>
    </>;
};

const OptionSettings: FC<OptionSettingsProps> = props => {
    const { kind } = props;
    switch (kind) {
        case OptionType.STRING:
            return <StringOptionSettings {...props} />
        default:
            return <></>;
    };
};

const OptionEditor: FC<OptionEditorProps> = props => {
    const { optionCode, option, onOptionChange, onDeleteClick } = props;
    return <div className="ui-option-editor">
        <div className="left">
            <span className="code" title={optionCode}><code>{optionCode}</code></span>
            <span className="delete" onClick={onDeleteClick}>Delete</span>
        </div>
        <div className="right">
            <select value={option.Type} onChange={event => { onOptionChange && onOptionChange({ ...option, Type: event.target.value as OptionType }) }}>
                <option value={OptionType.STRING}>String</option>
                <option value={OptionType.INTEGER}>Integer</option>
                <option value={OptionType.CONFIG}>Config</option>
                <option value={OptionType.SECRET}>Secret</option>
            </select>
        </div>
        <div className="middle">
            <Field className="title" title="Title:">
                <Input value={option.Title}
                    onValueChange={value => { onOptionChange && onOptionChange({ ...option, Title: value }) }}
                />
            </Field>
            <Field className="description" title="Description:">
                <Textarea value={option.Description}
                    onValueChange={value => { onOptionChange && onOptionChange({ ...option, Description: value }) }}
                />
            </Field>
            <Field className="parameters" title="Parameters:">
                <label>
                    <input type="checkbox" checked={option.Required}
                        onChange={event => { onOptionChange && onOptionChange({ ...option, Required: event.target.checked }) }}
                    />
                    Required
                </label>
                <label>
                    <input type="checkbox" checked={option.Editable}
                        onChange={event => { onOptionChange && onOptionChange({ ...option, Editable: event.target.checked }) }}
                    />
                    Editable
                </label>
                <label>
                    <input type="checkbox" checked={option.Visible}
                        onChange={event => { onOptionChange && onOptionChange({ ...option, Visible: event.target.checked }) }}
                    />
                    Visible
                </label>
            </Field>
            {option.Type === OptionType.STRING && <Field className="settings" title="Settings:">
                <OptionSettings
                    kind={option.Type} settings={option.Settings}
                    onSettingsChange={value => { onOptionChange && onOptionChange({ ...option, Settings: value }) }}
                />
            </Field>}
            <Field className="value" title="Default value:">
                <OptionValue option={{ ...option, Editable: true, Required: false }} optionCode={optionCode}
                    onValueChange={value => { onOptionChange && onOptionChange({ ...option, Value: value }) }}
                />
            </Field>
        </div>
    </div >;
};

export default OptionEditor;
