import { FC, useState } from "react";
import { Link } from "react-router-dom";
import Block from "../Block";
import Button from "../Button";
import Option from "../Option";
import OptionGroup from "../OptionGroup";
import { OptionType } from "../../api/actions";
import { IPlanner, IAction, ITaskOption } from "../../api";
import { TaskOptionsBlock } from "../TaskBlock";

export interface IPlanActionBlockProps {
	action: IAction;
	onSubmit?(planner: IPlanner): void;
};

const PlanActionBlock: FC<IPlanActionBlockProps> = props => {
	const { action, onSubmit } = props;
	const [title, setTitle] = useState<string>("");
	const [description, setDescription] = useState<string>("");
	const [schedule, setSchedule] = useState<string>("");
	const [repeatDelay, setRepeatDelay] = useState<string>("");
	const [failureDelay, setFailureDelay] = useState<string>("");
	const [listenSuccess, setListenSuccess] = useState<string>("");
	const [listenFailure, setListenFailure] = useState<string>("");
	const [failureMails, setFailureMails] = useState<string>("");
	const [notifyFailureThreshold, setNotifyFailureThreshold] = useState<string>("");
	const [allowOverlaps, setAllowOverlaps] = useState<boolean>(false);
	const [options, setOptions] = useState<{ [key: string]: ITaskOption }>({});
	return <form onSubmit={(event) => {
		event.preventDefault();
		onSubmit && onSubmit({
			id: 0,
			title: title,
			description: description || "",
			options: options,
			settings: {
				Schedule: schedule || undefined,
				RepeatDelay: repeatDelay ? Number(repeatDelay) : undefined,
				FailureDelay: failureDelay ? Number(failureDelay) : undefined,
				ListenSuccess: listenSuccess ? listenSuccess.split(",").map(Number) : undefined,
				ListenFailure: listenFailure ? listenFailure.split(",").map(Number) : undefined,
				FailureMails: failureMails ? failureMails.split(",") : undefined,
				AllowOverlaps: allowOverlaps,
				Enabled: false,
				notify_failure_threshold: notifyFailureThreshold ? Number(notifyFailureThreshold) : undefined,
			},
		});
	}}>
		<Block
			className="ui-block-planner"
			header={<h2 className="title">
				Plan action &laquo;<Link to={`/actions/${action.id}`}>{action.title}</Link>&raquo;
			</h2>}
			footer={<>
				<Button type="submit">Save</Button>
				<Link to={`/actions/${action.id}`}><Button>Back</Button></Link>
			</>}
		>
			<OptionGroup title="Info">
				<Option
					option={{
						Type: OptionType.STRING,
						Title: "Title",
						Description: "Name of planner.",
						Value: title,
						Required: true,
						Visible: true,
						Editable: true,
					}}
					optionCode="Title"
					onValueChange={setTitle}
				/>
				<Option
					option={{
						Type: OptionType.STRING,
						Title: "Description",
						Description: "Description for planner.",
						Value: description,
						Required: false,
						Visible: true,
						Editable: true,
						Settings: {
							Multiline: true,
						},
					}}
					optionCode="Description"
					onValueChange={setDescription}
				/>
				<Option
					option={{
						Type: OptionType.STRING,
						Title: "Schedule",
						Description: "Schedule for planner.",
						Value: schedule,
						Required: false,
						Visible: true,
						Editable: true,
					}}
					optionCode="Schedule"
					onValueChange={setSchedule}
				/>
				<Option
					option={{
						Type: OptionType.INTEGER,
						Title: "Repeat delay",
						Description: "Enter amount of seconds. Leave blank, if you don't want to schedule task by time.",
						Value: repeatDelay,
						Required: false,
						Visible: true,
						Editable: true,
					}}
					optionCode="RepeatDelay"
					onValueChange={setRepeatDelay}
				/>
				<Option
					option={{
						Type: OptionType.INTEGER,
						Title: "Failure delay",
						Description: "Enter amount of seconds. Leave blank, if you don't want to separate failure from success.",
						Value: failureDelay,
						Required: false,
						Visible: true,
						Editable: true,
					}}
					optionCode="FailureDelay"
					onValueChange={setFailureDelay}
				/>
				<div className="ui-option">
					<div className="left">
						<span className="title">Allow overlaps:</span>
					</div>
					<div className="right">Boolean</div>
					<div className="middle">
						<div className="value">
							<select name="AllowOverlaps" className="i-select" value={String(allowOverlaps)} onChange={(event) => setAllowOverlaps(event.target.value === "true")}>
								<option value="false">No</option>
								<option value="true">Yes</option>
							</select>
						</div>
						<div className="description">If you selected «Yes», then planner can run task even if another one is still running.</div>
					</div>
				</div>
				<Option
					option={{
						Type: OptionType.STRING,
						Title: "Listen success",
						Description: "Enter comma separated ID's of planners, that should trigger this planner on success. Example: «1,2,3,4,5».",
						Value: listenSuccess,
						Required: false,
						Visible: true,
						Editable: true,
					}}
					optionCode="ListenSuccess"
					onValueChange={setListenSuccess}
				/>
				<Option
					option={{
						Type: OptionType.STRING,
						Title: "Listen failure",
						Description: "Enter comma separated ID's of planners, that should trigger this planner on failure. Example: «1,2,3,4,5».",
						Value: listenFailure,
						Required: false,
						Visible: true,
						Editable: true,
					}}
					optionCode="ListenFailure"
					onValueChange={setListenFailure}
				/>
				<Option
					option={{
						Type: OptionType.STRING,
						Title: "Failure mails",
						Description: "Enter comma separated login's of users that should have failure email. Example: «user1,user2».",
						Value: failureMails,
						Required: false,
						Visible: true,
						Editable: true,
					}}
					optionCode="FailureMails"
					onValueChange={setFailureMails}
				/>
				<Option
					option={{
						Type: OptionType.INTEGER,
						Title: "Failure notification threshold",
						Description: "Amount of failures that allowed before sending notification",
						Value: notifyFailureThreshold,
						Required: false,
						Visible: true,
						Editable: true,
					}}
					optionCode="notify_failure_threshold"
					onValueChange={setNotifyFailureThreshold}
				/>
			</OptionGroup>
			{action.options && <TaskOptionsBlock options={action.options} taskOptions={options} onTaskOptionsChange={setOptions} editable={true} />}
		</Block>
	</form>;
};

export default PlanActionBlock;
