import { FC, Fragment, useEffect, useState } from "react";
import { Link } from "react-router-dom";
import Block from "../Block";
import Button from "../Button";
import OptionGroup from "../OptionGroup";
import Markdown from "../Markdown";
import { IPlanner, IAction } from "../../api";
import { TaskOptionsBlock } from "../TaskBlock";
import "./index.scss";

export interface IPlannerBlockProps {
	planner: IPlanner;
	onEnableClick?(): void;
	onDisableClick?(): void;
};

const PlannerBlock: FC<IPlannerBlockProps> = props => {
	const { planner, onEnableClick, onDisableClick } = props;
	const [action, setAction] = useState<IAction>();
	useEffect(() => {
		setAction(undefined);
		fetch(`/api/v0/actions/${planner.action_id}`)
			.then(response => response.json())
			.then((action: IAction) => {
				setAction(action);
			})
			.catch(console.log)
	}, [planner.action_id]);
	if (!action) {
		return <div />;
	}
	return <Block
		className="ui-block-planner"
		header={<h2 className="title">
			Planner: {planner.title} for &laquo;<Link to={`/actions/${action.id}`}>{action.title}</Link>&raquo;
			<span className="object-id">ID: {planner.id}</span>
		</h2>}
		footer={<>
			<Link to={`/planners/${planner.id}/run`}><Button>Manual run</Button></Link>
			<Link to={`/planners/${planner.id}/edit`}><Button>Edit</Button></Link>
		</>}
	>
		<p>Go to <a href={`/planner/${planner.id}`}>old interface</a>.</p>
		{planner.description && <OptionGroup title="Description"><Markdown text={planner.description} /></OptionGroup>}
		<OptionGroup title="Info">
			<div className="ui-option">
				<div className="left">
					<span className="title">Status:</span>
				</div>
				<div className="middle">
					<div className="value">{
						planner.settings.Enabled ?
							<Button onClick={onDisableClick}>Disable</Button> :
							<Button onClick={onEnableClick}>Enable</Button>
					}</div>
				</div>
			</div>
			{!!planner.next_time && <div className="ui-option">
				<div className="left">
					<span className="title">Next run:</span>
				</div>
				<div className="middle">
					<div className="value">{(new Date(planner.next_time * 1000)).toString()}</div>
				</div>
			</div>}
			{!!planner.settings.Schedule && <div className="ui-option">
				<div className="left">
					<span className="title">Schedule:</span>
				</div>
				<div className="middle">
					<div className="value">{planner.settings.Schedule}</div>
				</div>
			</div>}
			{!!planner.settings.RepeatDelay && <div className="ui-option">
				<div className="left">
					<span className="title">Repeat delay:</span>
				</div>
				<div className="middle">
					<div className="value">{planner.settings.RepeatDelay}</div>
				</div>
			</div>}
			{!!planner.settings.FailureDelay && <div className="ui-option">
				<div className="left">
					<span className="title">Failure delay:</span>
				</div>
				<div className="middle">
					<div className="value">{planner.settings.FailureDelay}</div>
				</div>
			</div>}
			<div className="ui-option">
				<div className="left">
					<span className="title">Allow overlaps:</span>
				</div>
				<div className="middle">
					<div className="value">{planner.settings.AllowOverlaps ? "Yes" : "No"}</div>
				</div>
			</div>
			{!!planner.settings.FailureMails && <div className="ui-option">
				<div className="left">
					<span className="title">Failure mails:</span>
				</div>
				<div className="middle">
					<div className="value">{planner.settings.FailureMails.map(
						(value: string, index: number) => <Fragment key={index}>
							{index > 0 && ", "}
							<a href={`https://staff.yandex-team.ru/${value}`} target="_blank" rel="noreferrer">{value}</a>
						</Fragment>
					)}</div>
				</div>
			</div>}
			{!!planner.settings.ListenSuccess && <div className="ui-option">
				<div className="left">
					<span className="title">Listen success:</span>
				</div>
				<div className="middle">
					<div className="value">{planner.settings.ListenSuccess.map(
						(value: number, index: number) => <Fragment key={index}>
							{index > 0 && ", "}
							<Link to={`/planners/${value}`}>{value}</Link>
						</Fragment>
					)}</div>
				</div>
			</div>}
			{!!planner.settings.ListenFailure && <div className="ui-option">
				<div className="left">
					<span className="title">Listen failure:</span>
				</div>
				<div className="middle">
					<div className="value">{planner.settings.ListenFailure.map(
						(value: number, index: number) => <Fragment key={index}>
							{index > 0 && ", "}
							<Link to={`/planners/${value}`}>{value}</Link>
						</Fragment>
					)}</div>
				</div>
			</div>}
			{!!planner.settings.notify_failure_threshold && <div className="ui-option">
				<div className="left">
					<span className="title">Failure notification threshold:</span>
				</div>
				<div className="middle">
					<div className="value">{planner.settings.notify_failure_threshold}</div>
				</div>
			</div>}
		</OptionGroup>
		{action.options && planner.options && <TaskOptionsBlock options={action.options} taskOptions={planner.options} editable={false} />}
	</Block>;
};

export default PlannerBlock;
