import { FC } from "react";
import { Link } from "react-router-dom";
import Block from "../Block";
import Button from "../Button";
import Option from "../Option";
import OptionGroup from "../OptionGroup";
import { OptionType } from "../../api/actions";
import { Secret, YavSecretData } from "../../api";

export interface IYavSecretDataBlockProps {
    data: YavSecretData;
    editable: boolean;
    onDataChange?(data: YavSecretData): void;
};

export const YavSecretDataBlock: FC<IYavSecretDataBlockProps> = props => {
    const { data, editable, onDataChange } = props;
    return <OptionGroup title="Data">
        <Option
            option={{
                Type: OptionType.STRING,
                Title: "SecretID",
                Description: "SecretID is ID of secret that commonly starts with sec-.",
                Value: data.SecretID,
                Required: true,
                Visible: true,
                Editable: editable,
            }}
            optionCode="SecretID"
            onValueChange={(value) => onDataChange && onDataChange({ ...data, SecretID: value })}
        />
        <Option
            option={{
                Type: OptionType.STRING,
                Title: "Key",
                Description: "Key of secret value.",
                Value: data.Key,
                Required: true,
                Visible: true,
                Editable: editable,
            }}
            optionCode="Key"
            onValueChange={(value) => onDataChange && onDataChange({ ...data, Key: value })}
        />
        <Option
            option={{
                Type: OptionType.STRING,
                Title: "VersionID",
                Description: "VersionID is ID of secret version that commonly starts with ver-.",
                Value: data.VersionID,
                Required: false,
                Visible: true,
                Editable: editable,
            }}
            optionCode="VersionID"
            onValueChange={(value) => onDataChange && onDataChange({ ...data, VersionID: value })}
        />
    </OptionGroup>;
};

export interface ISecretBlockProps {
    secret: Secret;
};

const YAV_HOST = "https://yav.yandex-team.ru";

const SecretBlock: FC<ISecretBlockProps> = props => {
    const { secret } = props;
    const { id, title, data } = secret;
    return <Block
        header={<h2 className="title">Secret: {title}<span className="object-id">ID: {id}</span></h2>}
        footer={<>
            <Link to={`/secrets/${id}/edit`}><Button>Edit</Button></Link>
            <Button>Remove</Button>
        </>}
    >
        {data && <>Go to <a href={`${YAV_HOST}/secret/${data.SecretID}`} target="_blank" rel="noopener noreferrer">Vault</a>.</>}
        {data && <YavSecretDataBlock data={data} editable={false} />}
    </Block>;
};

export default SecretBlock;
