import React, { FC, useState } from "react";
import { Link } from "react-router-dom";
import Block from "../Block";
import Button from "../Button";
import Option from "../Option";
import OptionGroup from "../OptionGroup";
import { YavSecretDataBlock } from "../SecretBlock";
import { OptionType } from "../../api/actions";
import { ErrorResponse, Secret, YavSecretData } from "../../api";

export interface ISecretEditorBlockProps {
    secret?: Secret;
    parentID?: number;
    error?: ErrorResponse;
    onSubmit?(secret: Secret): void;
};

const SecretEditorBlock: FC<ISecretEditorBlockProps> = props => {
    const { secret, parentID, onSubmit } = props;
    const [title, setTitle] = useState<string>(secret?.title ?? "");
    const [data, setData] = useState<YavSecretData>(secret?.data ?? {
        Key: "",
        SecretID: "",
    });
    return <form onSubmit={(event) => {
        event.preventDefault();
        onSubmit && onSubmit({
            id: secret?.id ?? 0,
            node_id: secret?.node_id ?? parentID,
            title: title,
            kind: "Yav",
            data: data,
        });
    }}>
        <Block
            header={<h2 className="title">{secret ? <>Edit secret <span className="object-id">ID: {secret.id}</span></> : "Create secret"}</h2>}
            footer={secret ? <>
                <Button type="submit">Save</Button>
                <Link to={`/secrets/${secret.id}`}><Button>Back</Button></Link>
            </> : <>
                <Button type="submit">Create</Button>
                <Link to={`/nodes/${parentID}`}><Button>Back</Button></Link>
            </>}
        >
            <OptionGroup title="Info">
                <Option
                    option={{
                        Type: OptionType.STRING,
                        Title: "Title",
                        Description: "Name of secret.",
                        Value: title,
                        Required: true,
                        Visible: true,
                        Editable: true,
                    }}
                    optionCode="Title"
                    onValueChange={setTitle}
                />
            </OptionGroup>
            {data && <YavSecretDataBlock data={data} onDataChange={setData} editable={true} />}
        </Block>
    </form>;
};

export default SecretEditorBlock;
