import { FC, Fragment } from "react";
import { Link } from "react-router-dom";
import Option from "../Option";
import OptionGroup from "../OptionGroup";
import Block from "../Block";
import Button from "../Button";
import { ITask, ITaskOption, IOption, IAction, IPlanner } from "../../api";
import "./index.scss";

export interface ITaskOptionsBlockProps {
	options: { [key: string]: IOption };
	taskOptions: { [key: string]: ITaskOption };
	editable: boolean;
	onTaskOptionsChange?(data: { [key: string]: ITaskOption }): void;
}

export const TaskOptionsBlock: FC<ITaskOptionsBlockProps> = props => {
	const { options, taskOptions, editable, onTaskOptionsChange } = props;
	return <OptionGroup title="Options">
		{Object.entries(options).map(([code, option], index) => {
			if (taskOptions[code]) {
				const taskOption = taskOptions[code];
				option = { ...option, Type: taskOption.Type, Value: taskOption.Value };
			} else if (!option.Visible) {
				return <Fragment key={index} />;
			}
			if (!editable) {
				option = { ...option, Editable: false };
			}
			return <Option
				option={option} optionCode={code} key={index}
				onValueChange={(value) => onTaskOptionsChange && onTaskOptionsChange({ ...taskOptions, [code]: { Type: option.Type, Value: value } })}
			/>;
		})}
	</OptionGroup>;
};

export type ITaskStatusProps = {
	status: string;
};

export const TaskStatus: FC<ITaskStatusProps> = props => {
	const { status } = props;
	return <span className={`ui-task-status ${status}`}>{status}</span>;
};

export const formatDuration = (duration: number): string => {
	const seconds = duration % 60;
	duration = (duration - seconds) / 60;
	const minutes = duration % 60;
	duration = (duration - minutes) / 60;
	const hours = duration % 24;
	duration = (duration - hours) / 24;
	let result: string = "";
	if (duration > 0) {
		result += `${duration}d`;
	}
	if (hours > 0) {
		result += `${hours}h`;
	}
	if (minutes > 0) {
		result += `${minutes}m`;
	}
	if (seconds > 0 || !result) {
		result += `${seconds}s`;
	}
	return result;
};

export interface ITaskBlockProps {
	task: ITask;
	action: IAction;
	planner?: IPlanner;
	stderrLogs?: string;
	stdoutLogs?: string;
	systemLogs?: string;
	onAbortTask?(): void;
};

const TaskBlock: FC<ITaskBlockProps> = props => {
	const { task, action, planner, stderrLogs, stdoutLogs, systemLogs, onAbortTask } = props;
	return <Block
		header={<h2 className="title">
			Task &laquo;<Link to={`/actions/${action.id}`}>{action.title}</Link>&raquo;
			{planner && <> by &laquo;<Link to={`/planners/${planner.id}`}>{planner.title}</Link>&raquo;</>}
			<span className="object-id">ID: {task.id}</span>
		</h2>}
		footer={<>
			<Link to={!!planner ? `/planners/${planner.id}` : `/actions/${action.id}`}><Button>Back</Button></Link>
			{task.status === "Running" && <Button onClick={onAbortTask}>Abort</Button>}
		</>}
	>
		<p>Go to <a href={`/task/${task.id}`}>old interface</a>.</p>
		<OptionGroup title="Info">
			<div className="ui-option">
				<div className="left">
					<span className="title">Status:</span>
				</div>
				<div className="middle">
					<div className="value"><TaskStatus status={task.status} /></div>
				</div>
			</div>
			<div className="ui-option">
				<div className="left">
					<span className="title">Start time:</span>
				</div>
				<div className="middle">
					<div className="value">{(new Date(task.create_time * 1000)).toString()}</div>
				</div>
			</div>
			<div className="ui-option">
				<div className="left">
					<span className="title">Duration:</span>
				</div>
				<div className="middle">
					<div className="value">{formatDuration(task.update_time - task.create_time)}</div>
				</div>
			</div>
		</OptionGroup>
		{stderrLogs && <OptionGroup title="Standard error">
			<pre className="ui-code"><code>{stderrLogs}</code></pre>
		</OptionGroup>}
		{stdoutLogs && <OptionGroup title="Standard output">
			<pre className="ui-code"><code>{stdoutLogs}</code></pre>
		</OptionGroup>}
		{systemLogs && <OptionGroup title="System output">
			<pre className="ui-code"><code>{systemLogs}</code></pre>
		</OptionGroup>}
		{action.options && task.options && <TaskOptionsBlock options={action.options} taskOptions={task.options} editable={false} />}
	</Block>;
};

export default TaskBlock;
