import React, { FC, useContext, useState, useEffect } from "react";
import { Link } from "react-router-dom";
import Block from "../Block";
import { Node } from "../../api";
import "./index.scss";

export type TreeNodeState = {
	treeNodeID?: number;
	setTreeNodeID(nodeID?: number): void;
};

export const TreeNodeContext = React.createContext<TreeNodeState>({
	setTreeNodeID(): void {
		throw new Error("TreeNodeContext is not initialized");
	},
});

export const TreeWrap: FC = props => {
	const { children } = props;
	const [treeNodeID, setTreeNodeID] = useState<number>();
	return <TreeNodeContext.Provider value={{ treeNodeID, setTreeNodeID }}>
		{children}
	</TreeNodeContext.Provider>;
};

export enum TreeNodeType {
	DIR,
	ACTION,
	PLANNER,
	CONFIG,
	SECRET,
	RESOURCE,
};

export type TreeNode = {
	type: TreeNodeType;
	id: number;
	title: string;
	description?: string;
	settings?: any;
};

export interface ITreeListProps {
	parent?: TreeNode;
	nodes?: TreeNode[];
};

export const TreeList: FC<ITreeListProps> = props => {
	const { parent, nodes } = props;
	if (nodes) {
		nodes.sort((lhs, rhs) => {
			const lhsIsDir = lhs.type === TreeNodeType.DIR;
			const rhsIsDir = rhs.type === TreeNodeType.DIR;
			// Dirs has greater priority than other objects.
			if (lhsIsDir !== rhsIsDir) {
				return lhsIsDir ? -1 : 1;
			}
			return lhs.title.localeCompare(rhs.title);
		});
	}
	return <ul className="ui-tree-list">
		{parent && <li className={`tree-node ${getNodeClassName(parent)}`}>
			<Link to={getNodeURL(parent)}>{parent.title}</Link>
		</li>}
		{nodes && nodes.map((node, index) =>
			<li className={`tree-node ${getNodeClassName(node)}`} key={index}>
				{getNodeLink(node)}
			</li>
		)}
	</ul>;
};

const getNodeClassName = (node: TreeNode) => {
	switch (node.type) {
		case TreeNodeType.DIR:
			return "node";
		case TreeNodeType.ACTION:
			return "action";
		case TreeNodeType.PLANNER:
			return "planner";
		case TreeNodeType.CONFIG:
			return "config";
		case TreeNodeType.SECRET:
			return "secret";
		case TreeNodeType.RESOURCE:
			return "resource";
		default:
			return "";
	}
};

const getNodeURL = (node: TreeNode) => {
	const { id } = node;
	switch (node.type) {
		case TreeNodeType.DIR:
			return `/nodes/${id}`;
		case TreeNodeType.ACTION:
			return `/actions/${id}`;
		case TreeNodeType.PLANNER:
			return `/planners/${id}`;
		case TreeNodeType.CONFIG:
			return `/configs/${id}`;
		case TreeNodeType.SECRET:
			return `/secrets/${id}`;
		case TreeNodeType.RESOURCE:
			return `/resources/${id}`;
		default:
			return "";
	}
};

const getNodeLink = (node: TreeNode) => {
	switch (node.type) {
		case TreeNodeType.PLANNER:
			return <Link to={getNodeURL(node)}>
				<span className="icon" />
				<span className={`status ${node.settings && node.settings.Enabled ? "enabled" : "disabled"}`}></span>
				<span className="title" title={node.title}>{node.title}</span>
			</Link>;
		default:
			return <Link to={getNodeURL(node)}>
				<span className="icon" />
				<span className="title" title={node.title}>{node.title}</span>
			</Link>;
	}
};

type Action = {
	ID: number;
	Title: string;
	Description?: string;
};

type Planner = {
	ID: number;
	Title: string;
	Description?: string;
	Settings?: any;
};

type Config = {
	ID: number;
	Title: string;
	Description?: string;
};

type Secret = {
	ID: number;
	Title: string;
	Description?: string;
};

type Resource = {
	ID: number;
	Title: string;
	Description?: string;
};

type Dir = {
	ID: number;
	Title: string;
	Description?: string;
};

type TreeListResponse = {
	ID: number;
	Title: string;
	DirID?: number;
	Actions?: Action[];
	Planners?: Planner[];
	Configs?: Config[];
	Secrets?: Secret[];
	Resources?: Resource[];
	Dirs?: Dir[];
};

export const TreeListWrap: FC = () => {
	const { treeNodeID } = useContext(TreeNodeContext);
	const [state, setState] = useState<ITreeListProps>({});
	const [node, setNode] = useState<Node>();
	useEffect(() => {
		// Clear child nodes before loading.
		setState({});
		// If node is specified we should load child elements.
		treeNodeID && fetch(`/api/tree/list/${treeNodeID}`)
			.then(response => response.json())
			.then((data: TreeListResponse) => {
				let nodes: TreeNode[] = [];
				const {
					DirID, Actions, Planners, Configs,
					Secrets, Resources, Dirs,
				} = data;
				Actions && nodes.push(...Actions.map(action => {
					return {
						type: TreeNodeType.ACTION,
						id: action.ID,
						title: action.Title,
						description: action.Description,
					};
				}));
				Planners && nodes.push(...Planners.map(planner => {
					return {
						type: TreeNodeType.PLANNER,
						id: planner.ID,
						title: planner.Title,
						description: planner.Description,
						settings: planner.Settings,
					};
				}));
				Configs && nodes.push(...Configs.map(config => {
					return {
						type: TreeNodeType.CONFIG,
						id: config.ID,
						title: config.Title,
						description: config.Description,
					};
				}));
				Secrets && nodes.push(...Secrets.map(secret => {
					return {
						type: TreeNodeType.SECRET,
						id: secret.ID,
						title: secret.Title,
						description: secret.Description,
					};
				}));
				Resources && nodes.push(...Resources.map(resource => {
					return {
						type: TreeNodeType.RESOURCE,
						id: resource.ID,
						title: resource.Title,
						description: resource.Description,
					};
				}));
				Dirs && nodes.push(...Dirs.map(dir => {
					return {
						type: TreeNodeType.DIR,
						id: dir.ID,
						title: dir.Title,
						description: dir.Description,
					};
				}));
				let newState: ITreeListProps = { nodes: nodes };
				if (DirID) {
					newState.parent = {
						type: TreeNodeType.DIR,
						id: DirID,
						title: "...",
					};
				}
				setState(newState);
				setNode({
					id: data.ID,
					name: data.Title,
					inherit_roles: false,
				});
			});
	}, [treeNodeID])
	return <Block className={"ui-tree-list-wrap"} header={
		node && <Link to={`/nodes/${node.id}`}>{node.name}</Link>
	}>
		<TreeList parent={state.parent} nodes={state.nodes} />
	</Block>;
};
