package worker

import (
	"container/list"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"os"
	"sync"
	"sync/atomic"
	"time"

	"a.yandex-team.ru/library/go/core/log"

	"a.yandex-team.ru/drive/runner/config"
)

// FileURL represents URL to loadable file.
type FileURL struct {
	Provider string
	Path     string
}

// FileProvider represents file provider.
type FileProvider interface {
	ReadFile(url FileURL) (FileReader, error)
}

type cachedFile struct {
	path    string
	expires time.Time
	usages  int32
	loaded  chan struct{}
	err     error
}

// FileStore represents cached file store.
type FileStore struct {
	dir       string
	providers map[string]FileProvider
	files     *list.List
	index     map[FileURL]*list.Element
	trash     []*list.Element
	mutex     sync.Mutex
	logger    log.Logger
}

// NewFileStore creates new instance of file store.
func NewFileStore(cfg *config.Worker, logger log.Logger) (*FileStore, error) {
	_ = os.RemoveAll(cfg.CacheDir)
	if err := os.MkdirAll(cfg.CacheDir, os.ModePerm); err != nil {
		return nil, err
	}
	providers := make(map[string]FileProvider)
	if cfg.Arcadia != nil {
		providers["arcadia"] = &httpOAuthFileProvider{
			baseURL: cfg.Arcadia.BaseURL,
			token:   cfg.Arcadia.Token.Secret(),
			cache:   -time.Hour,
			logger:  logger,
		}
	}
	if cfg.Sandbox != nil {
		providers["sandbox"] = &httpOAuthFileProvider{
			baseURL: cfg.Sandbox.BaseURL,
			token:   cfg.Sandbox.Token.Secret(),
			cache:   12 * time.Hour,
			logger:  logger,
		}
	}
	return &FileStore{
		dir:       cfg.CacheDir,
		providers: providers,
		files:     list.New(),
		index:     map[FileURL]*list.Element{},
		logger:    logger,
	}, nil
}

type FileReader interface {
	io.ReadCloser
	ExpireTime() time.Time
}

type uncachedFileError struct {
	FileReader
}

func (e uncachedFileError) Error() string {
	return "file does not need any cache"
}

func (s *FileStore) getCachedFile(
	url FileURL, forceCache bool,
) (*cachedFile, error) {
	provider, ok := s.providers[url.Provider]
	if !ok {
		return nil, fmt.Errorf("unsupported provider %q", url.Provider)
	}
	s.mutex.Lock()
	defer s.mutex.Unlock()
	if elem, ok := s.index[url]; ok {
		cache := elem.Value.(*cachedFile)
		select {
		case <-cache.loaded:
			if cache.expires.After(time.Now()) && cache.err == nil {
				atomic.AddInt32(&cache.usages, 1)
				return cache, nil
			}
			delete(s.index, url)
			s.trash = append(s.trash, elem)
		default:
			atomic.AddInt32(&cache.usages, 1)
			return cache, nil
		}
	}
	// Now we try to read file from provider.
	file, err := provider.ReadFile(url)
	if err != nil {
		return nil, err
	}
	// If file has too small expire time, we should not store
	// this file in cache, so we immediately return FileReader.
	if !forceCache && file.ExpireTime().Before(time.Now()) {
		return nil, uncachedFileError{file}
	}
	// Try to create temp path. If we fail, we should return
	// FileReader from previous step.
	temp, err := ioutil.TempFile(s.dir, url.Provider+"-")
	if err != nil {
		if forceCache {
			if err := file.Close(); err != nil {
				s.logger.Error("Error closing file", log.Error(err))
			}
			return nil, err
		}
		return nil, uncachedFileError{file}
	}
	// Create cache.
	cache := &cachedFile{
		path:    temp.Name(),
		expires: file.ExpireTime(),
		usages:  1,
		loaded:  make(chan struct{}),
	}
	s.index[url] = s.files.PushBack(cache)
	go s.loadCachedFile(cache, temp, file)
	return cache, nil
}

func (s *FileStore) loadCachedFile(
	cache *cachedFile, temp *os.File, file io.ReadCloser,
) {
	defer func() {
		if err := file.Close(); err != nil {
			if cache.err == nil {
				cache.err = err
			}
			s.logger.Error(
				"Error closing file",
				log.String("file", cache.path), log.Error(err),
			)
		}
		close(cache.loaded)
	}()
	_, cache.err = io.Copy(temp, file)
}

type cachedFileReader struct {
	io.ReadCloser
	expires time.Time
	cache   *cachedFile
}

func (r *cachedFileReader) ExpireTime() time.Time {
	return r.expires
}

func (r *cachedFileReader) Read(p []byte) (int, error) {
	return r.ReadCloser.Read(p)
}

func (r *cachedFileReader) Close() error {
	defer func() {
		if r.cache != nil {
			atomic.AddInt32(&r.cache.usages, -1)
			r.cache = nil
		}
	}()
	return r.ReadCloser.Close()
}

// File represents file in file store.
type File struct {
	expires time.Time
	cache   *cachedFile
}

// ExpireTime returns expire time of file.
func (f *File) ExpireTime() time.Time {
	return f.expires
}

// Name returns path to cached file.
func (f *File) Name() string {
	if f.cache == nil {
		panic("File already closed")
	}
	return f.cache.path
}

// Open opens file.
func (f *File) Open() (*os.File, error) {
	if f.cache == nil {
		return nil, fmt.Errorf("file already freed")
	}
	return os.Open(f.cache.path)
}

// Close releases file in file store.
func (f *File) Close() {
	if f.cache != nil {
		atomic.AddInt32(&f.cache.usages, -1)
		f.cache = nil
	}
}

// LoadFile loads file from specified url.
//
// This function can be called concurrently.
func (s *FileStore) LoadFile(url FileURL) (*File, error) {
	cache, err := s.getCachedFile(url, true)
	if err != nil {
		return nil, err
	}
	if <-cache.loaded; cache.err != nil {
		atomic.AddInt32(&cache.usages, -1)
		return nil, cache.err
	}
	return &File{cache.expires, cache}, nil
}

// ReadFile reads file from specified url.
//
// This function can be called concurrently.
func (s *FileStore) ReadFile(url FileURL) (FileReader, error) {
	cache, err := s.getCachedFile(url, false)
	if err != nil {
		if f, ok := err.(uncachedFileError); ok {
			return f.FileReader, nil
		}
		return nil, err
	}
	if <-cache.loaded; cache.err != nil {
		atomic.AddInt32(&cache.usages, -1)
		return nil, cache.err
	}
	file, err := os.Open(cache.path)
	if err != nil {
		atomic.AddInt32(&cache.usages, -1)
		return nil, err
	}
	return &cachedFileReader{file, cache.expires, cache}, nil
}

// Cleanup cleans unused files.
func (s *FileStore) Cleanup(force bool) {
	s.findExpiredFiles(force)
	s.removeFiles()
}

func (s *FileStore) findExpiredFiles(force bool) {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	for url, elem := range s.index {
		cache := elem.Value.(*cachedFile)
		if force || cache.expires.Before(time.Now()) || cache.err != nil {
			delete(s.index, url)
			s.trash = append(s.trash, elem)
		}
	}
}

func (s *FileStore) removeFiles() {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	newLen := 0
	for _, elem := range s.trash {
		s.trash[newLen] = elem
		cache := elem.Value.(*cachedFile)
		// Check that file is not used anywhere.
		if u := atomic.LoadInt32(&cache.usages); u > 0 {
			newLen++
			continue
		}
		// Try to remove this file.
		if err := os.RemoveAll(cache.path); err != nil {
			s.logger.Error(
				"Unable to remove file",
				log.String("file", cache.path), log.Error(err),
			)
			newLen++
			continue
		}
		s.files.Remove(elem)
	}
	s.trash = s.trash[:newLen]
}

type fileReaderWithExpire struct {
	io.ReadCloser
	expires time.Time
}

func (r fileReaderWithExpire) ExpireTime() time.Time {
	return r.expires
}

type httpOAuthFileProvider struct {
	baseURL string
	token   string
	cache   time.Duration
	client  http.Client
	logger  log.Logger
}

func (p *httpOAuthFileProvider) ReadFile(url FileURL) (FileReader, error) {
	req, err := http.NewRequest(
		http.MethodGet,
		fmt.Sprintf("%s%s", p.baseURL, url.Path),
		nil,
	)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Authorization", fmt.Sprintf("OAuth %s", p.token))
	resp, err := p.client.Do(req)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != http.StatusOK {
		if err := resp.Body.Close(); err != nil {
			p.logger.Error("Error while closing body", log.Error(err))
		}
		return nil, fmt.Errorf(
			"server returned error: %d", resp.StatusCode,
		)
	}
	return fileReaderWithExpire{resp.Body, time.Now().Add(p.cache)}, nil
}
