package worker

import (
	"context"
	"fmt"
	"math/rand"
	"time"

	"a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/library/go/core/log"
)

var (
	errLeaderLockReleased = fmt.Errorf("leader lock released")
	errLeaderDisabled     = fmt.Errorf("host can not be leader")
)

func (w *Worker) isLeaderActive(ctx context.Context) bool {
	return w.isRunnerActive(ctx) && w.core.Host().Config.EnableLeader
}

func (w *Worker) leaderLoop(ctx context.Context) error {
	defer func() {
		if r := recover(); r != nil {
			if err, ok := r.(error); ok && err != nil {
				w.logger.Error(
					"Leader panicked with error",
					log.Error(err),
				)
			} else {
				w.logger.Error(
					"Leader panicked without error",
					log.Any("recovered", r),
				)
			}
		}
	}()
	// First of all we should check that leader is active.
	if !w.isLeaderActive(ctx) {
		return nil
	}
	w.logger.Info("Leadership acquired")
	defer w.logger.Info("Leadership released")
	ticker := time.NewTicker(time.Second)
	defer ticker.Stop()
	for w.isLeaderActive(ctx) {
		select {
		case <-ctx.Done():
			return nil
		case <-ticker.C:
			w.leaderTick(ctx)
		}
	}
	return nil
}

func (w *Worker) leaderTick(ctx context.Context) {
	// First of all we should check that leader is active.
	if !w.isLeaderActive(ctx) {
		return
	}
	start := time.Now()
	defer func() {
		stop := time.Now()
		w.core.Signal("leader_tick.duration_last", nil).
			Set(float64(stop.Sub(start).Microseconds()))
	}()
	allNodes, err := w.core.Hosts.All()
	if err != nil {
		w.logger.Error("Unable to list nodes", log.Error(err))
		return
	}
	var nodes []models.Host
	for _, node := range allNodes {
		if node.HasRunningTask("runner") || node.HasRunningTask("worker") {
			nodes = append(nodes, node)
		} else if node.Offline() {
			if err := w.core.Tasks.KillByNode(node.ID); err != nil {
				w.logger.Error("Unable to kill tasks", log.Error(err))
			}
		}
	}
	if len(nodes) == 0 {
		w.logger.Error("No nodes with runner available")
		return
	}
	states, err := w.core.PlannerStates.All()
	if err != nil {
		w.logger.Error("Unable to get all planner states", log.Error(err))
	}
	statesMap := map[int]models.PlannerState{}
	for _, state := range states {
		statesMap[state.PlannerID] = state
	}
	// Get all planners
	planners, err := w.core.Planners.All()
	if err != nil {
		w.logger.Error("Unable to get all planners", log.Error(err))
	}
	plannersMap := map[int]models.Planner{}
	successListeners := map[int][]int{}
	failureListeners := map[int][]int{}
	for _, planner := range planners {
		if !w.isLeaderActive(ctx) {
			return
		}
		state, ok := statesMap[planner.ID]
		if !ok {
			state.PlannerID = planner.ID
		}
		if err := w.tryRunPlannerTask(&planner, &state); err != nil {
			w.logger.Error("Unable to run planner", log.Error(err))
		}
		plannersMap[planner.ID] = planner
		statesMap[planner.ID] = state
		for _, id := range planner.Settings.ListenSuccess {
			successListeners[id] = append(
				successListeners[id], planner.ID,
			)
		}
		for _, id := range planner.Settings.ListenFailure {
			failureListeners[id] = append(
				failureListeners[id], planner.ID,
			)
		}
	}
	// Find queued unassigned tasks
	tasks, err := w.core.Tasks.FindQueuedUnassigned()
	if err != nil {
		w.logger.Error("Unable to find queued tasks", log.Error(err))
		return
	}
	for _, task := range tasks {
		if !w.isLeaderActive(ctx) {
			return
		}
		if task.PlannerID != 0 {
			planner, ok := plannersMap[int(task.PlannerID)]
			if !ok {
				continue
			}
			flag, err := w.canAssignTask(&planner)
			if err != nil || !flag {
				if err != nil {
					w.logger.Error("Unable to assign task", log.Error(err))
				}
				continue
			}
		}
		// TODO(iudovin@): Implement resource usage.
		node := rand.Int() % len(nodes)
		task.NodeID = models.NInt(nodes[node].ID)
		err := w.core.Tasks.Update(&task)
		if err != nil {
			w.logger.Error(
				"Unable to assign task",
				log.Int64("task_id", task.ID),
				log.String("host_name", nodes[node].Name),
				log.Error(err),
			)
			continue
		}
		w.logger.Info(
			"Task assigned",
			log.Int64("task_id", task.ID),
			log.String("host_name", nodes[node].Name),
		)
	}
	// Find finished tasks that not processed by leader
	tasks, err = w.core.Tasks.FindFinishedAssigned()
	if err != nil {
		w.logger.Error("Unable to find finished tasks", log.Error(err))
		return
	}
	for _, task := range tasks {
		if !w.isLeaderActive(ctx) {
			return
		}
		if err := w.processFinishedTask(
			&task, plannersMap, statesMap, successListeners, failureListeners,
		); err != nil {
			w.logger.Error("Unable to process finished tasks", log.Error(err))
		}
	}
}
