package worker

import (
	"database/sql"
	"fmt"
	"html"
	"net/smtp"
	"time"

	"a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/library/go/core/log"
)

func (w *Worker) canAssignTask(planner *models.Planner) (bool, error) {
	if planner.Settings.AllowOverlaps {
		return true, nil
	}
	tasks, err := w.core.Tasks.GetAssignedByPlanner(planner.ID)
	if err != nil || len(tasks) > 0 {
		return false, err
	}
	return true, nil
}

func (w *Worker) hasQueuedTask(planner *models.Planner) (bool, error) {
	if planner.Settings.AllowOverlaps {
		return true, nil
	}
	tasks, err := w.core.Tasks.GetQueuedByPlanner(planner.ID)
	if err != nil || len(tasks) > 0 {
		return false, err
	}
	return true, nil
}

func formatEmailTaskLogs(
	logs []models.TaskLog,
) (system string, stdout string, stderr string) {
	for _, taskLog := range logs {
		switch taskLog.Type {
		case models.SystemTaskLog:
			system += taskLog.Lines
		case models.StdoutTaskLog:
			stdout += taskLog.Lines
		case models.StderrTaskLog:
			stderr += taskLog.Lines
		}
	}
	return
}

func (w *Worker) sendFailureEmail(
	task *models.Task, planner models.Planner, state models.PlannerState,
) error {
	if state.FailureCounter <= planner.Settings.NotifyFailureThreshold {
		return nil
	}
	var mails []string
	for _, login := range planner.Settings.FailureMails {
		mails = append(mails, fmt.Sprintf("%s@yandex-team.ru", login))
	}
	logs, err := w.core.TaskLogs.FindByTask(task.ID)
	if err != nil {
		return err
	}
	systemLog, stdoutLog, stderrLog := formatEmailTaskLogs(logs)
	message := []byte(fmt.Sprintf(
		"From: \"Drive.Runner\" <%s>\r\n"+
			"Subject: Task for Planner %q FAILED\r\n"+
			"Content-Type: text/html\r\n"+
			"\r\n"+
			"<p>See full information here:"+
			"<a href=\"https://%s/tasks/%d/\">Task #%d</a></p>"+
			"<h3>System log</h3>"+
			"<pre><code>%s</code></pre>"+
			"<h3>Standard error</h3>"+
			"<pre><code>%s</code></pre>"+
			"<h3>Standard output</h3>"+
			"<pre><code>%s</code></pre>",
		w.core.Config.Mail.From, planner.Title,
		w.core.Config.Domain, task.ID, task.ID,
		html.EscapeString(systemLog),
		html.EscapeString(stderrLog),
		html.EscapeString(stdoutLog),
	))
	return smtp.SendMail(
		fmt.Sprintf("%s:%d",
			w.core.Config.Mail.Host,
			w.core.Config.Mail.Port,
		),
		nil,
		w.core.Config.Mail.From,
		mails,
		message,
	)
}

// runPlannerTaskTx creates task for specified planner
func (w *Worker) runPlannerTaskTx(
	tx *sql.Tx, planner *models.Planner,
) (task models.Task, err error) {
	task = models.Task{
		ActionID:  planner.ActionID,
		Options:   planner.Options,
		PlannerID: models.NInt(planner.ID),
	}
	if err = w.core.Tasks.CreateTx(tx, &task); err != nil {
		w.logger.Error(
			"Unable to create task for planner",
			log.Int("planner_id", planner.ID),
		)
	}
	return
}

// tryRunPlannerTask checks that planner should run task and creates it
func (w *Worker) tryRunPlannerTask(planner *models.Planner, state *models.PlannerState) error {
	if !planner.Settings.Enabled {
		return nil
	}
	nowTime := time.Now().Unix()
	// Planner should wait next time.
	if state.NextTime == 0 || int64(state.NextTime) >= nowTime {
		return nil
	}
	if ok, err := w.canAssignTask(planner); !ok || err != nil {
		if err != nil {
			return err
		}
		stateCopy := *state
		// TODO(iudovin@): Add support of planner failure.
		stateCopy.NextTime = models.NInt64(planner.Next(nowTime))
		if err := w.core.PlannerStates.Upsert(stateCopy, "next_time"); err != nil {
			return err
		}
		*state = stateCopy
		return nil
	}
	tx, err := w.core.DB.Begin()
	if err != nil {
		return err
	}
	if _, err := w.runPlannerTaskTx(tx, planner); err != nil {
		if err := tx.Rollback(); err != nil {
			w.logger.Error("Rollback error", log.Error(err))
		}
		return err
	}
	// We can not modify planner before transaction is not committed
	stateCopy := *state
	stateCopy.NextTime = models.NInt64(planner.Next(int64(state.NextTime)))
	if int64(stateCopy.NextTime) < nowTime {
		stateCopy.NextTime = models.NInt64(planner.Next(nowTime))
	}
	if err := w.core.PlannerStates.UpsertTx(tx, stateCopy, "next_time"); err != nil {
		if err := tx.Rollback(); err != nil {
			w.logger.Error("Rollback error", log.Error(err))
		}
		return err
	}
	if err := tx.Commit(); err != nil {
		return err
	}
	// After successful commit we can update planner data
	*state = stateCopy
	return nil
}

func (w *Worker) runListenersTx(
	tx *sql.Tx, planners map[int]models.Planner, listeners []int,
) error {
	for _, id := range listeners {
		planner, ok := planners[id]
		if !ok {
			w.logger.Error(
				"Unable to find planner", log.Int("planner_id", id),
			)
			continue
		}
		if !planner.Settings.Enabled {
			continue
		}
		if ok, err := w.hasQueuedTask(&planner); !ok || err != nil {
			if err != nil {
				return err
			}
			continue
		}
		if _, err := w.runPlannerTaskTx(tx, &planner); err != nil {
			return err
		}
	}
	return nil
}

func (w *Worker) processFinishedTask(
	task *models.Task, planners map[int]models.Planner,
	states map[int]models.PlannerState,
	successListeners map[int][]int, failureListeners map[int][]int,
) error {
	tx, err := w.core.DB.Begin()
	if err != nil {
		return err
	}
	// Detach task from node
	task.NodeID = 0
	if err := w.core.Tasks.UpdateTx(tx, task, "node_id"); err != nil {
		if err := tx.Rollback(); err != nil {
			w.logger.Error("Rollback error", log.Error(err))
		}
		return err
	}
	w.logger.Info("Task unassigned from node", log.Int64("task_id", task.ID))
	if task.PlannerID == 0 {
		return tx.Commit()
	}
	planner, ok := planners[int(task.PlannerID)]
	if !ok {
		if err := tx.Rollback(); err != nil {
			w.logger.Error("Rollback error", log.Error(err))
		}
		return fmt.Errorf(
			"unable to find planner #%d for task #%d",
			task.PlannerID, task.ID,
		)
	}
	state, ok := states[planner.ID]
	if !ok {
		state.PlannerID = planner.ID
	}
	if task.Status == models.SucceededTask {
		state.FailureCounter = 0
		if err := w.core.PlannerStates.UpsertTx(
			tx, state, "failure_counter",
		); err != nil {
			if err := tx.Rollback(); err != nil {
				w.logger.Error("Rollback error", log.Error(err))
			}
			return err
		}
		if err := w.runListenersTx(
			tx, planners, successListeners[planner.ID],
		); err != nil {
			if err := tx.Rollback(); err != nil {
				w.logger.Error("Rollback error", log.Error(err))
			}
			return err
		}
	} else if task.Status == models.FailedTask {
		state.FailureCounter++
		if planner.Settings.FailureDelay != nil {
			nextTime := task.UpdateTime + int64(*planner.Settings.FailureDelay)
			if nextTime < int64(state.NextTime) {
				state.NextTime = models.NInt64(nextTime)
			}
		}
		if err := w.core.PlannerStates.UpsertTx(
			tx, state, "next_time", "failure_counter",
		); err != nil {
			if err := tx.Rollback(); err != nil {
				w.logger.Error("Rollback error", log.Error(err))
			}
			return err
		}
		if err := w.sendFailureEmail(task, planner, state); err != nil {
			w.logger.Warn("Error", log.Error(err))
		}
		if err := w.runListenersTx(
			tx, planners, failureListeners[planner.ID],
		); err != nil {
			if err := tx.Rollback(); err != nil {
				w.logger.Error("Rollback error", log.Error(err))
			}
			return err
		}
	}
	return tx.Commit()
}
