package worker

import (
	"encoding/json"
	"fmt"
	"os"
	"strconv"
	"strings"
	"text/template"
	"unicode"
)

const (
	layersOption      = "@layers"
	workDirOption     = "@work_dir"
	environOption     = "@environ"
	commandOption     = "@command"
	filesOption       = "@files"
	memoryLimitOption = "@memory_limit"
	cpuLimitOption    = "@cpu_limit"
	// Deprecated.
	layersOldOption  = "@Layers"
	workDirOldOption = "@WorkDir"
	environOldOption = "@Environ"
	commandOldOption = "@Command"
	filesOldOption   = "@Files"
)

type localFileURL struct {
	FileURL
	Path string
	Mode os.FileMode
}

type parsedOptions struct {
	Layers      []FileURL
	WorkDir     string
	Environ     []string
	Command     string
	Files       []localFileURL
	MemoryLimit int64
	Options     TaskOptions
}

func (o parsedOptions) environ() string {
	escaper := strings.NewReplacer("\\", "\\\\", ";", "\\;")
	var environ strings.Builder
	for _, variable := range o.Environ {
		_, _ = escaper.WriteString(&environ, variable)
		_, _ = environ.WriteRune(';')
	}
	return environ.String()
}

func parseOptions(taskOpts TaskOptions) (opts parsedOptions, err error) {
	opts.Layers, err = extractLayersOption(taskOpts)
	if err != nil {
		return
	}
	opts.WorkDir, err = extractWorkDirOption(taskOpts)
	if err != nil {
		return
	}
	opts.Environ, err = extractEnvironOption(taskOpts)
	if err != nil {
		return
	}
	opts.Command, err = extractCommandOption(taskOpts)
	if err != nil {
		return
	}
	opts.Files, err = extractFilesOption(taskOpts)
	if err != nil {
		return
	}
	opts.MemoryLimit, err = extractMemoryLimitOption(taskOpts)
	if err != nil {
		return
	}
	opts.Options = taskOpts
	return
}

func renderValue(option string, opts TaskOptions) (string, error) {
	funcs := template.FuncMap{
		"option": func(name string) string {
			return opts[name]
		},
		"json": func(name string) (string, error) {
			data, err := json.Marshal(name)
			return string(data), err
		},
		// Deprecated.
		"Option": func(name string) string {
			return opts[name]
		},
	}
	tmpl, err := template.New("Task").Funcs(funcs).Parse(option)
	if err != nil {
		return "", err
	}
	var buffer strings.Builder
	if err = tmpl.Execute(&buffer, nil); err != nil {
		return "", err
	}
	return buffer.String(), nil
}

func parseFileURL(path string) (FileURL, error) {
	parts := strings.SplitN(path, ":", 2)
	if len(parts) != 2 {
		return FileURL{}, fmt.Errorf(
			"URL %q is not valid", path,
		)
	}
	result := FileURL{
		Provider: parts[0],
		Path:     parts[1],
	}
	if len(result.Provider) == 0 {
		return FileURL{}, fmt.Errorf(
			"URL %q has invalid schema", path,
		)
	}
	result.Path = strings.TrimPrefix(result.Path, "//")
	if len(result.Path) == 0 {
		return FileURL{}, fmt.Errorf(
			"URL %q has invalid path", path,
		)
	}
	return result, nil
}

func parsePath(path string) (string, os.FileMode, error) {
	parts := strings.SplitN(path, ":", 2)
	mode := os.FileMode(0666)
	if len(parts) > 1 && parts[1] == "+x" {
		mode |= 0111
	}
	return parts[0], mode, nil
}

func extractLayersOption(options TaskOptions) ([]FileURL, error) {
	value, ok := options[layersOption]
	if !ok {
		value, ok = options[layersOldOption]
	}
	if !ok {
		return nil, nil
	}
	value, err := renderValue(value, options)
	if err != nil {
		return nil, fmt.Errorf(
			"option %q render error: %v",
			layersOption, err,
		)
	}
	lines := strings.FieldsFunc(value, splitLines)
	var result []FileURL
	for _, line := range lines {
		line = strings.TrimFunc(line, unicode.IsSpace)
		path, err := parseFileURL(line)
		if err != nil {
			return nil, err
		}
		result = append(result, path)
	}
	return result, nil
}

const defaultWorkDir = "/root"

func extractWorkDirOption(options TaskOptions) (string, error) {
	value, ok := options[workDirOption]
	if !ok {
		value, ok = options[workDirOldOption]
	}
	if !ok {
		if _, ok := options[layersOption]; !ok {
			return "/", nil
		}
		return defaultWorkDir, nil
	}
	value, err := renderValue(value, options)
	if err != nil {
		return "", fmt.Errorf(
			"option %q render error: %v",
			workDirOption, err,
		)
	}
	// Treat empty string as default value.
	if value != "" {
		if !strings.HasPrefix(value, "/") {
			return "", fmt.Errorf(
				"option %q has invalid value: %q",
				workDirOption, value,
			)
		}
		return value, nil
	}
	return defaultWorkDir, nil
}

func extractEnvironOption(options TaskOptions) ([]string, error) {
	value, ok := options[environOption]
	if !ok {
		value, ok = options[environOldOption]
	}
	if !ok {
		return nil, fmt.Errorf("option %q does not exist", environOption)
	}
	value, err := renderValue(value, options)
	if err != nil {
		return nil, fmt.Errorf(
			"option %q render error: %v",
			environOption, err,
		)
	}
	return strings.FieldsFunc(value, splitLines), nil
}

func extractFilesOption(options TaskOptions) ([]localFileURL, error) {
	value, ok := options[filesOption]
	if !ok {
		value, ok = options[filesOldOption]
	}
	if !ok {
		return nil, nil
	}
	value, err := renderValue(value, options)
	if err != nil {
		return nil, fmt.Errorf(
			"option %q render error: %v",
			filesOption, err,
		)
	}
	lines := strings.FieldsFunc(value, splitLines)
	var result []localFileURL
	for _, line := range lines {
		parts := strings.SplitN(line, "=>", 2)
		if len(parts) != 2 {
			return nil, fmt.Errorf(
				"option %q has invalid line: %q",
				filesOption, line,
			)
		}
		parts[0] = strings.TrimFunc(parts[0], unicode.IsSpace)
		parts[1] = strings.TrimFunc(parts[1], unicode.IsSpace)
		url, err := parseFileURL(parts[0])
		if err != nil {
			return nil, err
		}
		if len(parts[1]) == 0 {
			return nil, fmt.Errorf(
				"option %q has invalid line: %q",
				filesOption, line,
			)
		}
		path, mode, err := parsePath(parts[1])
		if err != nil {
			return nil, err
		}
		result = append(result, localFileURL{url, path, mode})
	}
	return result, nil
}

func extractCommandOption(options TaskOptions) (string, error) {
	value, ok := options[commandOption]
	if !ok {
		value, ok = options[commandOldOption]
	}
	if !ok {
		return "", fmt.Errorf("option %q does not exist", commandOption)
	}
	value, err := renderValue(value, options)
	if err != nil {
		return "", fmt.Errorf(
			"option %q render error: %v",
			commandOption, err,
		)
	}
	if value != "" {
		return value, nil
	}
	return "", fmt.Errorf(
		"option %q is empty", commandOption,
	)
}

func extractMemoryLimitOption(options TaskOptions) (int64, error) {
	value, ok := options[memoryLimitOption]
	if !ok {
		return 0, nil
	}
	value, err := renderValue(value, options)
	if err != nil {
		return 0, fmt.Errorf(
			"option %q render error: %v",
			memoryLimitOption, err,
		)
	}
	intValue, err := strconv.ParseInt(value, 0, 64)
	if err != nil {
		return 0, fmt.Errorf(
			"option %q has invalid value: %q",
			memoryLimitOption, value,
		)
	}
	return intValue, nil
}

func splitLines(c rune) bool {
	return c == '\n' || c == '\r'
}
