package worker

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/library/go/core/log"
)

func (w *Worker) cleanupLoop(ctx context.Context) error {
	w.logger.Info("Cleanup started")
	defer w.logger.Info("Cleanup stopped")
	ticker := time.NewTicker(time.Minute)
	defer ticker.Stop()
	for {
		select {
		case <-ctx.Done():
			return nil
		case <-ticker.C:
			w.porto.Cleanup()
			w.files.Cleanup(false)
		}
	}
}

func (w *Worker) isRunnerActive(ctx context.Context) bool {
	select {
	case <-ctx.Done():
		return false
	default:
		return w.core.Host().Active()
	}
}

func (w *Worker) runnerLoop(ctx context.Context) error {
	defer func() {
		if r := recover(); r != nil {
			if err, ok := r.(error); ok && err != nil {
				w.logger.Error(
					"Runner panicked with error",
					log.Error(err),
				)
			} else {
				w.logger.Error(
					"Runner panicked without error",
					log.Any("recovered", r),
				)
			}
		}
	}()
	// First of all we should check that runner is running.
	if !w.isRunnerActive(ctx) {
		return nil
	}
	w.logger.Info("Runner started")
	defer w.logger.Info("Runner stopped")
	if err := w.core.Tasks.KillByNode(w.node().ID); err != nil {
		return err
	}
	ticker := time.NewTicker(time.Second)
	defer ticker.Stop()
	for w.isRunnerActive(ctx) {
		select {
		case <-ctx.Done():
			return nil
		case <-ticker.C:
			w.runnerTick(ctx)
		}
	}
	return nil
}

func (w *Worker) runnerTick(ctx context.Context) {
	// First of all we should check that runner is running.
	if !w.isRunnerActive(ctx) {
		return
	}
	tasks, err := w.core.Tasks.FindQueuedByNode(w.core.Host().ID)
	if err != nil {
		w.logger.Error("Unable to find queued tasks", log.Error(err))
		return
	}
	for _, task := range tasks {
		// Before any run we should check that task is running.
		if !w.isRunnerActive(ctx) {
			return
		}
		task := task
		w.logger.Info("Found task", log.Int64("task_id", task.ID))
		task.Status = models.StartingTask
		if err := w.core.Tasks.Update(&task); err != nil {
			w.logger.Error(
				"Error starting task",
				log.Int64("task_id", task.ID), log.Error(err),
			)
			continue
		}
		w.core.StartTask(
			fmt.Sprintf("task.%d", task.ID),
			func(ctx context.Context) {
				if err := w.runTask(task); err != nil {
					w.logger.Error(
						"Error running task",
						log.Int64("task_id", task.ID),
						log.Error(err),
					)
				}
			},
		)
	}
}
