package worker

import (
	"bufio"
	"context"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"net"
	"net/http"
	"os"
	"os/exec"
	"path"
	"path/filepath"
	"strings"
	"sync"
	"syscall"
	"text/template"
	"time"
	"unicode"

	"github.com/jmoiron/sqlx/types"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/runner/api"
	"a.yandex-team.ru/drive/runner/core"
	"a.yandex-team.ru/drive/runner/models"
	"a.yandex-team.ru/library/go/core/log"
)

type Worker struct {
	mutex  sync.RWMutex
	core   *core.Core
	closer chan struct{}
	waiter sync.WaitGroup
	porto  *portoProcessor
	files  *FileStore
	// logger contains worker logger.
	logger log.Logger
}

func cleanupOldRoots(path string, logger log.Logger) {
	files, err := ioutil.ReadDir(path)
	if err != nil {
		logger.Error("Unable to read dir", log.String("path", path))
		return
	}
	for _, file := range files {
		if !strings.HasPrefix(file.Name(), "dajr-") {
			continue
		}
		filePath := filepath.Join(path, file.Name())
		logger.Debug("Removing old root", log.String("path", filePath))
		if err := os.RemoveAll(filePath); err != nil {
			logger.Error(
				"Unable to remove path", log.String("path", filePath),
			)
		}
	}
}

func NewWorker(c *core.Core) (*Worker, error) {
	files, err := NewFileStore(c.Config.Worker, c.Logger("file_store"))
	if err != nil {
		return nil, err
	}
	processor, err := newPortoProcessor(c.Config, files)
	if err != nil {
		return nil, err
	}
	logger := c.Logger("worker")
	cleanupOldRoots(c.Config.SystemDir, logger)
	return &Worker{
		core:   c,
		porto:  processor,
		files:  files,
		logger: logger,
	}, nil
}

func (w *Worker) node() models.Host {
	return w.core.Host()
}

func mergeOptions(
	a models.ActionOptions,
	t models.TaskOptions,
) (models.TaskOptions, error) {
	m := make(models.TaskOptions)
	// First step validates that passed task options are correct
	for name, option := range t {
		o, ok := a[name]
		if !ok {
			return nil, fmt.Errorf("unknown option %q", name)
		}
		if o.Type != option.Type {
			return nil, fmt.Errorf("incorrect type of option %q", name)
		}
		if o.Required && option.Value == nil {
			return nil, fmt.Errorf("required option %q is empty", name)
		}
		m[name] = option
	}
	// Second step validates that default values are also correct
	for name, option := range a {
		_, ok := t[name]
		if ok {
			continue
		}
		if option.Required && option.Value == nil {
			return nil, fmt.Errorf("required option %q is empty", name)
		}
		m[name] = models.TaskOption{Type: option.Type, Value: option.Value}
	}
	return m, nil
}

func (w *Worker) getOptionValueMap(options models.TaskOptions) map[string]string {
	values := make(map[string]string)
	for name, option := range options {
		if option.Value == nil {
			values[name] = ""
			continue
		}
		switch option.Type {
		case models.StringOption:
			values[name] = option.Value.(string)
		case models.IntegerOption:
			value := option.Value.(int64)
			values[name] = fmt.Sprintf("%d", value)
		case models.FloatOption:
			value := option.Value.(float64)
			values[name] = fmt.Sprintf("%f", value)
		case models.ConfigOption:
			value := option.Value.(int64)
			cfg, err := w.core.Configs.Get(int(value))
			if err != nil {
				w.logger.Error("Unable to get config", log.Error(err))
				continue
			}
			values[name] = cfg.Data
		case models.SecretOption:
			switch value := option.Value.(type) {
			// Kostyl for old styled secrets.
			case models.SecretValue:
				secret, ok := w.core.Config.Secrets[value.Name]
				if ok {
					values[name] = secret.Secret()
				}
			case int64:
				secret, err := w.core.Secrets.Get(int(value))
				if err != nil {
					w.logger.Error("Unable to get secret", log.Error(err))
					continue
				}
				secretValue, err := w.core.Secrets.SecretValue(secret)
				if err != nil {
					w.logger.Error(
						"Unable to get secret value",
						log.Error(err),
					)
					continue
				}
				values[name] = secretValue
			}
		case models.JSONQuery:
			value := option.Value.(types.JSONText)
			values[name] = value.String()
		}
	}
	return values
}

var templateFuncs = template.FuncMap{
	"Env": func(s string) string {
		s = strings.ReplaceAll(s, "\n", "")
		s = strings.ReplaceAll(s, "\r", "")
		return s
	},
}

func (w *Worker) getTemplateValue(text string, values map[string]string) (string, error) {
	tmpl, err := template.New("Option").Funcs(templateFuncs).Parse(text)
	if err != nil {
		return "", err
	}
	var buffer strings.Builder
	err = tmpl.Execute(&buffer, values)
	if err != nil {
		return "", err
	}
	return buffer.String(), nil
}

func parseCommandLine(command string) ([]string, error) {
	var arguments []string
	var argument strings.Builder
	var escape bool
	var open rune
	for _, c := range command {
		if escape {
			argument.WriteRune(c)
			escape = false
		} else if c == '\\' {
			escape = true
		} else if open == '\'' || open == '"' {
			if c == open {
				arguments = append(arguments, argument.String())
				argument.Reset()
				open = 0
			} else {
				argument.WriteRune(c)
			}
		} else if c == '\'' || c == '"' {
			open = c
		} else if unicode.IsSpace(c) {
			if argument.Len() > 0 {
				arguments = append(arguments, argument.String())
				argument.Reset()
			}
		} else {
			argument.WriteRune(c)
		}
	}
	if open == '\'' || open == '"' {
		return nil, errors.New("unclosed quote in command line")
	}
	if argument.Len() > 0 {
		arguments = append(arguments, argument.String())
	}
	return arguments, nil
}

func (w *Worker) getCommand(options map[string]string) (*exec.Cmd, error) {
	command, ok := options["__Command"]
	if !ok {
		return nil, errors.New("option '__Command' does not exists")
	}
	command, err := w.getTemplateValue(command, options)
	if err != nil {
		return nil, err
	}
	args, err := parseCommandLine(command)
	if err != nil {
		return nil, err
	}
	if len(args) < 1 {
		return nil, errors.New("too view arguments")
	}
	variables, ok := options["__Variables"]
	if !ok {
		return nil, errors.New("option '__Variables' does not exists")
	}
	variables, err = w.getTemplateValue(variables, options)
	if err != nil {
		return nil, err
	}
	cmd := exec.Command(args[0], args[1:]...)
	cmd.Env = strings.FieldsFunc(variables, splitLines)
	// We use this for kill all processes in the same process group
	cmd.SysProcAttr = &syscall.SysProcAttr{Setpgid: true}
	return cmd, nil
}

func (w *Worker) prepareRootFS(root string, options map[string]string) error {
	files, ok := options["__Files"]
	if !ok {
		return errors.New("option '__Files' does not exists")
	}
	files, err := w.getTemplateValue(files, options)
	if err != nil {
		return err
	}
	fileLines := strings.FieldsFunc(files, splitLines)
	for _, line := range fileLines {
		parts := strings.SplitN(line, "=>", 2)
		if len(parts) < 2 {
			return errors.New("invalid value of option '__Files'")
		}
		source, target := parts[0], parts[1]
		source = strings.TrimFunc(source, unicode.IsSpace)
		target = strings.TrimFunc(target, unicode.IsSpace)
		if len(source) == 0 || len(target) == 0 {
			return errors.New("source or target is empty")
		}
		if strings.HasPrefix(target, "/") {
			return errors.New("target should not start with '/'")
		}
		// Make target absolute path
		target = path.Join(root, target)
		if strings.HasPrefix(source, "option:") {
			optionName := source[len("option:"):]
			if err := ioutil.WriteFile(
				target, []byte(options[optionName]), os.ModePerm,
			); err != nil {
				return err
			}
			continue
		}
		fileURL, err := parseFileURL(source)
		if err != nil {
			return err
		}
		if err := func() error {
			file, err := w.files.ReadFile(fileURL)
			if err != nil {
				return err
			}
			defer func() {
				if err := file.Close(); err != nil {
					w.logger.Error("Unable to close file", log.Error(err))
				}
			}()
			temp, err := os.OpenFile(
				target, os.O_RDWR|os.O_CREATE|os.O_TRUNC, os.ModePerm,
			)
			if err != nil {
				return err
			}
			defer func() {
				_ = temp.Close()
			}()
			_, err = io.Copy(temp, file)
			return err
		}(); err != nil {
			return err
		}
	}
	return nil
}

func (w *Worker) pushStreamChannel(reader io.Reader, lines chan<- string) {
	scanner := bufio.NewScanner(reader)
	for scanner.Scan() {
		lines <- scanner.Text()
	}
	close(lines)
}

func (w *Worker) saveTaskLog(
	task models.Task, buffer []string, logType models.TaskLogType,
) error {
	lines := strings.Join(buffer, "\n") + "\n"
	taskLog := models.TaskLog{TaskID: task.ID, Type: logType, Lines: lines}
	return w.core.TaskLogs.Create(&taskLog)
}

func (w *Worker) saveRunningTaskBuffer(
	task models.Task, required bool, buffer *[]string,
	logType models.TaskLogType,
) {
	err := w.saveTaskLog(task, *buffer, logType)
	if required {
		for i := 0; i < 12 && err != nil; i++ {
			time.Sleep(15 * time.Second)
			w.logger.Warn(
				"Unable to save logs",
				log.String("log_type", string(logType)),
				log.Int64("task_id", task.ID),
				log.Error(err),
			)
			err = w.saveTaskLog(task, *buffer, logType)
		}
		if err != nil {
			w.logger.Error(
				"Unable to save logs",
				log.String("log_type", string(logType)),
				log.Int64("task_id", task.ID),
				log.Error(err),
			)
			return
		}
	}
	if err == nil {
		// Clear buffer
		*buffer = nil
	} else {
		w.logger.Warn(
			"Unable to save logs",
			log.String("log_type", string(logType)),
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
	}
}

func setTaskStatus(
	store *models.TaskStore, task *models.Task, status models.TaskStatus,
	logger log.Logger,
) {
	task.Status = status
	err := store.Update(task)
	for i := 0; i < 120 && err != nil; i++ {
		time.Sleep(15 * time.Second)
		logger.Warn(
			"Unable to change task status",
			log.String("task_status", string(status)),
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		err = store.Update(task)
	}
	if err != nil {
		logger.Error(
			"Unable to change task status",
			log.String("task_status", string(status)),
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		panic(err)
	}
}

type taskLog struct {
	worker  *Worker
	task    models.Task
	logType models.TaskLogType
	buffer  []string
	mutex   sync.Mutex
}

func (l *taskLog) WriteLine(line string) {
	l.mutex.Lock()
	defer l.mutex.Unlock()
	l.buffer = append(l.buffer, line)
}

func (l *taskLog) Save(force bool) {
	l.mutex.Lock()
	defer l.mutex.Unlock()
	if len(l.buffer) > 0 {
		l.worker.saveRunningTaskBuffer(l.task, force, &l.buffer, l.logType)
	}
}

func (w *Worker) newTaskLog(
	task models.Task, logType models.TaskLogType,
) *taskLog {
	return &taskLog{worker: w, task: task, logType: logType}
}

func readLogLines(waiter *sync.WaitGroup, reader io.Reader, task *taskLog) {
	defer waiter.Done()
	scanner := bufio.NewScanner(reader)
	for scanner.Scan() {
		task.WriteLine(scanner.Text())
	}
}

func (w *Worker) buildTaskOptions(options models.TaskOptions) TaskOptions {
	taskOptions := TaskOptions{}
	for name, option := range options {
		if option.Value == nil {
			taskOptions[name] = ""
			continue
		}
		switch option.Type {
		case models.StringOption:
			taskOptions[name] = option.Value.(string)
		case models.IntegerOption:
			value := option.Value.(int64)
			taskOptions[name] = fmt.Sprintf("%d", value)
		case models.FloatOption:
			value := option.Value.(float64)
			taskOptions[name] = fmt.Sprintf("%f", value)
		case models.ConfigOption:
			value := option.Value.(int64)
			cfg, err := w.core.Configs.Get(int(value))
			if err != nil {
				w.logger.Error("Unable to get config", log.Error(err))
				continue
			}
			taskOptions[name] = cfg.Data
		case models.SecretOption:
			value := option.Value.(int64)
			secret, err := w.core.Secrets.Get(int(value))
			if err != nil {
				w.logger.Error("Unable to get secret", log.Error(err))
				continue
			}
			secretValue, err := w.core.Secrets.SecretValue(secret)
			if err != nil {
				w.logger.Error("Unable to get secret value", log.Error(err))
				continue
			}
			taskOptions[name] = secretValue
		}
	}
	return taskOptions
}

func (w *Worker) runTaskPorto(action models.Action, task models.Task) error {
	w.core.Signal("worker.task_accepted_sum", nil).Add(1)
	systemLog := w.newTaskLog(task, models.SystemTaskLog)
	defer systemLog.Save(true)
	stderrLog := w.newTaskLog(task, models.StderrTaskLog)
	defer stderrLog.Save(true)
	stdoutLog := w.newTaskLog(task, models.StdoutTaskLog)
	defer stdoutLog.Save(true)
	options, err := mergeOptions(action.Options, task.Options)
	if err != nil {
		systemLog.WriteLine(fmt.Sprintf(
			"Task runned with invalid options: %s", err,
		))
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	process, err := w.porto.Create(w.buildTaskOptions(options))
	if err != nil {
		systemLog.WriteLine(fmt.Sprintf(
			"Unable to create task process: %s", err,
		))
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	defer func() {
		if err := process.Destroy(); err != nil {
			w.logger.Error("Unable to destroy process", log.Error(err))
		}
	}()
	if err := process.Start(); err != nil {
		systemLog.WriteLine(fmt.Sprintf(
			"Unable to start task process: %s", err,
		))
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	setTaskStatus(w.core.Tasks, &task, models.RunningTask, w.logger)
	closer := make(chan struct{})
	var waiter sync.WaitGroup
	waiter.Add(3)
	go readLogLines(&waiter, process.Stderr(), stderrLog)
	go readLogLines(&waiter, process.Stdout(), stdoutLog)
	go func() {
		defer waiter.Done()
		ticker := time.NewTicker(10 * time.Second)
		defer ticker.Stop()
		for {
			select {
			case <-closer:
				return
			case <-ticker.C:
				taskCopy := task.Clone()
				if err := w.core.Tasks.Reload(&taskCopy); err != nil {
					w.logger.Warn("Unable to reload task", log.Error(err))
					continue
				} else {
					task.Status = taskCopy.Status
					if task.Status != models.RunningTask {
						w.logger.Info(
							"Kill task due to not running status",
							log.Int64("task_id", task.ID),
							log.String("task_status", string(task.Status)),
						)
						if err := process.Signal(syscall.SIGKILL); err != nil {
							w.logger.Error(
								"Unable to kill process",
								log.Error(err),
							)
						}
					}
				}
				stderrLog.Save(false)
				stdoutLog.Save(false)
			}
		}
	}()
	if err := process.Wait(); err != nil {
		if errExit, ok := err.(*ExitError); !ok {
			systemLog.WriteLine(fmt.Sprintf(
				"Unable to wait task process: %s", err,
			))
			close(closer)
			waiter.Wait()
			setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
			return err
		} else {
			if task.Status == models.AbortingTask {
				close(closer)
				waiter.Wait()
				setTaskStatus(w.core.Tasks, &task, models.AbortedTask, w.logger)
				return nil
			} else {
				systemLog.WriteLine(fmt.Sprintf("Exit code: %d", errExit.ExitCode()))
				close(closer)
				waiter.Wait()
				setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
				return nil
			}
		}
	}
	close(closer)
	waiter.Wait()
	setTaskStatus(w.core.Tasks, &task, models.SucceededTask, w.logger)
	return nil
}

func (w *Worker) runTask(task models.Task) error {
	// Store lines for push in memory
	var systemBuffer []string
	var stdoutBuffer []string
	var stderrBuffer []string
	// Save buffers.
	saveBuffers := func(required bool) {
		if len(systemBuffer) > 0 {
			w.saveRunningTaskBuffer(
				task, required, &systemBuffer, models.SystemTaskLog,
			)
		}
		if len(stdoutBuffer) > 0 {
			w.saveRunningTaskBuffer(
				task, required, &stdoutBuffer, models.StdoutTaskLog,
			)
		}
		if len(stderrBuffer) > 0 {
			w.saveRunningTaskBuffer(
				task, required, &stderrBuffer, models.StderrTaskLog,
			)
		}
	}
	defer saveBuffers(true)
	w.logger.Info("Starting task", log.Int64("task_id", task.ID))
	action, err := w.core.Actions.Get(task.ActionID)
	if err != nil {
		w.logger.Error(
			"Unable to start task",
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		systemBuffer = append(
			systemBuffer,
			fmt.Sprintf(
				"Unable to find Action with ID = %d: %s",
				task.ActionID, err,
			),
		)
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	// TODO(iudovin): Replace all code with porto containers
	if _, ok := action.Options[commandOption]; ok {
		return w.runTaskPorto(action, task)
	}
	if _, ok := action.Options[commandOldOption]; ok {
		return w.runTaskPorto(action, task)
	}
	w.core.Signal("worker.legacy_task_accepted_sum", nil).Add(1)
	options, err := mergeOptions(action.Options, task.Options)
	if err != nil {
		w.logger.Error(
			"Unable to merge options",
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		systemBuffer = append(
			systemBuffer,
			fmt.Sprintf(
				"Task runned with invalid options: %s",
				err,
			),
		)
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	optionMap := w.getOptionValueMap(options)
	cmd, err := w.getCommand(optionMap)
	if err != nil {
		w.logger.Error(
			"Unable to parse command",
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		systemBuffer = append(
			systemBuffer,
			fmt.Sprintf(
				"Unable to create command: %s",
				err,
			),
		)
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	// Prepare root FS
	rootFS, err := ioutil.TempDir(w.core.Config.SystemDir, "dajr-")
	if err != nil {
		w.logger.Error(
			"Unable to create root directory",
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		systemBuffer = append(
			systemBuffer,
			fmt.Sprintf(
				"Unable to create temp directory: %s",
				err,
			),
		)
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	defer func() {
		err := os.RemoveAll(rootFS)
		if err != nil {
			w.logger.Error(
				"Unable to remove root directory",
				log.Error(err),
			)
		}
	}()
	err = w.prepareRootFS(rootFS, optionMap)
	if err != nil {
		w.logger.Error(
			"Unable to prepare root directory",
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		systemBuffer = append(
			systemBuffer,
			fmt.Sprintf(
				"Unable to prepare task directory: %s",
				err,
			),
		)
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	cmd.Dir = rootFS
	// Try to setup task socket
	socketPath, err := w.getTemplateValue(
		optionMap["__SocketPath"], optionMap,
	)
	if err != nil {
		return err
	}
	if socketPath != "" {
		socketPath = path.Join(rootFS, socketPath)
		server := echo.New()
		listener, err := net.Listen("unix", socketPath)
		if err != nil {
			w.logger.Error(
				"Unable to prepare API listener",
				log.Int64("task_id", task.ID),
				log.Error(err),
			)
			systemBuffer = append(
				systemBuffer,
				fmt.Sprintf(
					"Unable to setup socket: %s",
					err,
				),
			)
			setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
			return err
		}
		server.HideBanner = true
		server.HidePort = true
		server.Listener = listener
		go func() {
			err := server.Start("")
			if err != nil && err != http.ErrServerClosed {
				w.logger.Error(
					"API server failed to start",
					log.Int64("task_id", task.ID),
					log.Error(err),
				)
			}
		}()
		defer func() {
			err := server.Shutdown(context.Background())
			if err != nil {
				w.logger.Error(
					"API server failed to shutdown",
					log.Int64("task_id", task.ID),
					log.Error(err),
				)
			}
		}()
		view := api.NewView(w.core)
		view.RegisterTask(task, server.Group(""))
	}
	// Setup stdout pipe
	stdoutPipe, err := cmd.StdoutPipe()
	if err != nil {
		w.logger.Error(
			"Unable to open stdout",
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		systemBuffer = append(
			systemBuffer,
			fmt.Sprintf(
				"Unable to prepare 'stdout' pipe: %s",
				err,
			),
		)
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	// Setup stderr pipe
	stderrPipe, err := cmd.StderrPipe()
	if err != nil {
		w.logger.Error(
			"Unable to open stderr",
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		systemBuffer = append(
			systemBuffer,
			fmt.Sprintf(
				"Unable to prepare 'stderr' pipe: %s",
				err,
			),
		)
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	// Try to start process
	if err := cmd.Start(); err != nil {
		w.logger.Error(
			"Unable to start command",
			log.Int64("task_id", task.ID),
			log.Error(err),
		)
		systemBuffer = append(
			systemBuffer,
			fmt.Sprintf(
				"Unable to start task: %s",
				err,
			),
		)
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
		return err
	}
	// Mark task as running
	setTaskStatus(w.core.Tasks, &task, models.RunningTask, w.logger)
	w.logger.Info(
		"Task is running",
		log.Int64("task_id", task.ID),
	)
	// Prepare channels for standard I/O lines
	stdoutLines := make(chan string, 256)
	stderrLines := make(chan string, 256)
	// Read lines in blocking mode, so do it in separate goroutines
	go w.pushStreamChannel(stdoutPipe, stdoutLines)
	go w.pushStreamChannel(stderrPipe, stderrLines)
	ticker := time.NewTicker(5 * time.Second)
	running := true
	for running && (stdoutLines != nil || stderrLines != nil) {
		select {
		case line, ok := <-stdoutLines:
			if !ok {
				stdoutLines = nil
				continue
			}
			stdoutBuffer = append(stdoutBuffer, line)
		case line, ok := <-stderrLines:
			if !ok {
				stderrLines = nil
				continue
			}
			stderrBuffer = append(stderrBuffer, line)
		case <-ticker.C:
			taskCopy := task.Clone()
			if err := w.core.Tasks.Reload(&taskCopy); err != nil {
				w.logger.Error(
					"Unable reload task",
					log.Int64("task_id", task.ID),
					log.Error(err),
				)
			} else {
				task.Status = taskCopy.Status
				if task.Status != models.RunningTask {
					w.logger.Info(
						"Kill task due to not running status",
						log.Int64("task_id", task.ID),
						log.String("task_status", string(task.Status)),
					)
					pgid, err := syscall.Getpgid(cmd.Process.Pid)
					if err != nil {
						w.logger.Error(
							"Unable to get process group",
							log.Int64("task_id", task.ID),
							log.Error(err),
						)
						continue
					}
					// Kill process group.
					if err := syscall.Kill(-pgid, syscall.SIGKILL); err != nil {
						w.logger.Error(
							"Unable to kill process",
							log.Int64("task_id", task.ID),
							log.Error(err),
						)
					}
					running = false
				}
			}
			saveBuffers(false)
		}
	}
	ticker.Stop()
	timer := time.AfterFunc(5*time.Second, func() {
		pgid, err := syscall.Getpgid(cmd.Process.Pid)
		if err != nil {
			w.logger.Error(
				"Unable to get process group",
				log.Int64("task_id", task.ID),
				log.Error(err),
			)
			return
		}
		// Kill process group.
		if err := syscall.Kill(-pgid, syscall.SIGKILL); err != nil {
			w.logger.Error(
				"Unable to kill process",
				log.Int64("task_id", task.ID),
				log.Error(err),
			)
		}
	})
	if err := cmd.Wait(); err != nil {
		if _, ok := err.(*exec.ExitError); !ok {
			systemBuffer = append(
				systemBuffer, fmt.Sprintln("Wait error:", err),
			)
			setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
			return err
		}
	}
	timer.Stop()
	if task.Status == models.AbortingTask {
		setTaskStatus(w.core.Tasks, &task, models.AbortedTask, w.logger)
	} else if cmd.ProcessState.Success() {
		setTaskStatus(w.core.Tasks, &task, models.SucceededTask, w.logger)
	} else {
		systemBuffer = append(systemBuffer, fmt.Sprintf(
			"Exit code: %d", cmd.ProcessState.ExitCode(),
		))
		setTaskStatus(w.core.Tasks, &task, models.FailedTask, w.logger)
	}
	return nil
}

// Start starts worker.
func (w *Worker) Start() {
	w.core.StartDaemon("runner", w.runnerLoop)
	w.core.StartDaemon("cleanup", w.cleanupLoop)
	w.core.StartClusterDaemon(models.LeaderLockName, w.leaderLoop)
}
