#include <drive/telematics/api/server/client.h>
#include <drive/telematics/api/server/k8s/service_discovery.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/getopt/small/modchooser.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/cast.h>

int main_api_command(int argc, const char** argv) {
    NLastGetopt::TOpts options = NLastGetopt::TOpts::Default();
    options.AddLongOption("k8s-sd-endpoint", "K8s ServiceDiscovery endpoint").RequiredArgument("URL").DefaultValue(
        "http://drivematics-k8s-discovery-service/?app=drivematics-telematics-server"
    );
    options.AddLongOption('i', "imei", "Telematics client IMEI").RequiredArgument("IMEI").Optional();
    options.SetFreeArgsNum(1);
    options.SetFreeArgDefaultTitle("COMMAND", "command to execute");
    NLastGetopt::TOptsParseResult res(&options, argc, argv);

    const TString& imei = res.Get('i');
    const TString cmd = res.GetFreeArgs().at(0);
    const auto command = NDrive::ParseCommand(NJson::ToJson(NJson::JsonString(cmd)));

    NDrive::IServiceDiscovery::TPtr serviceDiscovery;
    {
        NDrive::NK8s::TServiceDiscoveryOptions serviceDiscoveryOptions;
        serviceDiscoveryOptions.Endpoint = res.Get("k8s-sd-endpoint");
        serviceDiscovery = MakeAtomicShared<NDrive::NK8s::TServiceDiscovery>(serviceDiscoveryOptions);
    }
    NDrive::TTelematicsClient::TOptions clientOptions;
    NDrive::TTelematicsClient client(serviceDiscovery, clientOptions);

    auto start = Now();
    auto handle = client.Command(imei, command.GetValue());
    handle.GetFuture().Wait();
    auto finish = Now();

    NJson::TJsonValue report = NJson::TMapBuilder
        ("id", handle.GetId())
        ("duration", NJson::ToJson(finish - start))
        ("message", handle.GetMessage())
        ("shard", handle.GetShard())
        ("status", ToString(handle.GetStatus()))
    ;
    Cout << report.GetStringRobust() << Endl;

    return EXIT_SUCCESS;
}

int main_api(int argc, const char** argv) {
    TModChooser modChooser;
    modChooser.AddMode("command", main_api_command, "Set a command to a device");
    return modChooser.Run(argc, argv);
}

int main(int argc, const char** argv) {
    DoInitGlobalLog("cerr", TLOG_INFO, false, false);

    TModChooser modChooser;
    modChooser.AddMode("api", main_api, "API mode");
    try {
        return modChooser.Run(argc, argv);
    } catch (const std::exception& e) {
        Cerr << "An exception has occurred: " << FormatExc(e) << Endl;
        return EXIT_FAILURE;
    }
}
