#include "processor.hpp"

#include <fmt/format.h>

#include <userver/components/component_context.hpp>
#include <userver/storages/redis/component.hpp>

#include <string>


drive::handlers::LPMProcessor::LPMProcessor(const components::ComponentConfig& config, const components::ComponentContext& context)
  : server::handlers::HttpHandlerBase(config, context)
  , redis_client_{ context.FindComponent<components::Redis>().GetClient("lpm-cache-api") }
{
}

std::string drive::handlers::LPMProcessor::HandleRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& /* context */) const {
  const auto& key = request.GetPathArg("key");
  if (key.empty()) {
    throw server::handlers::ClientError(server::handlers::ExternalBody{"No 'key' query argument"});
  }
  switch (request.GetMethod()) {
    case server::http::HttpMethod::kGet:
      return GetValue(key, request);
    case server::http::HttpMethod::kPost:
      return UpsertValue(key, request);
    case server::http::HttpMethod::kDelete:
      return DeleteValue(key);
    default:
      throw server::handlers::ClientError(server::handlers::ExternalBody{ fmt::format("Unsupported method {}", request.GetMethod()) });
  }
}

std::string drive::handlers::LPMProcessor::GetValue(std::string_view key, const server::http::HttpRequest& request) const {
  const auto result = redis_client_->Get(std::string{key}, redis_cc_).Get();
  if (!result) {
    request.SetResponseStatus(server::http::HttpStatus::kNotFound);
    return {};
  }
  return *result;
}

std::string drive::handlers::LPMProcessor::UpsertValue(std::string_view key, const server::http::HttpRequest& request) const {
  const auto& value = request.RequestBody();
  const auto& seconds = request.GetArg("ttl_seconds");
  if (!seconds.empty()) {
    auto ttl = std::chrono::seconds(std::stoi(seconds));
    redis_client_->Set(std::string{key}, value, ttl, redis_cc_).Get();
  } else {
    redis_client_->Set(std::string{key}, value, redis_cc_).Get();
  }
  request.SetResponseStatus(server::http::HttpStatus::kCreated);
  return std::string{value};
}

std::string drive::handlers::LPMProcessor::DeleteValue(std::string_view key) const {
  const auto result = redis_client_->Del(std::string{key}, redis_cc_).Get();
  return std::to_string(result);
}

