#include "processor.hpp"

#include <fmt/format.h>

#include <userver/components/component_context.hpp>
#include <userver/formats/json/value.hpp>
#include <userver/storages/redis/component.hpp>

#include <string>


drive::handlers::StoreOffersProcessor::StoreOffersProcessor(const components::ComponentConfig& config, const components::ComponentContext& context)
  : server::handlers::HttpHandlerBase(config, context)
  , redis_client_{ context.FindComponent<components::Redis>().GetClient("offers-cache-api") }
{
}

std::string MakeUserKey(const std::string& user_id, const std::string& offer_id) {
  return fmt::format("{{{}}}:{}", user_id, offer_id);
}

std::string drive::handlers::StoreOffersProcessor::HandleRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& /* context */) const {
  if (request.GetMethod() != server::http::HttpMethod::kPost) {
      throw server::handlers::ClientError(server::handlers::ExternalBody{ fmt::format("Unsupported method {}", request.GetMethod()) });
  }
  const auto json = formats::json::FromString(request.RequestBody());
  json.CheckArrayOrNull();
  for (const auto& item : json) {
    item.CheckObject();
    std::chrono::milliseconds ttl = std::chrono::minutes(30);
    const auto deadline = std::chrono::seconds(item["deadline"].As<uint64_t>());
    const auto now_value = std::chrono::system_clock::now().time_since_epoch();
    if (deadline > now_value) {
      ttl = std::chrono::duration_cast<std::chrono::milliseconds>(deadline - now_value);
    }
    std::string offer_key = item["id"].As<std::string>();
    std::string user_key = MakeUserKey(item["user_id"].As<std::string>(), offer_key);
    redis_client_->Set(std::move(offer_key), formats::json::ToString(item), ttl, redis_cc_).Get();
    redis_client_->Set(std::move(user_key), formats::json::ToString(item), ttl, redis_cc_).Get();
  }
  request.SetResponseStatus(server::http::HttpStatus::kCreated);
  return "";
}

drive::handlers::RestoreOfferProcessor::RestoreOfferProcessor(const components::ComponentConfig& config, const components::ComponentContext& context)
  : server::handlers::HttpHandlerBase(config, context)
  , redis_client_{ context.FindComponent<components::Redis>().GetClient("offers-cache-api") }
{
}

std::string drive::handlers::RestoreOfferProcessor::HandleRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& /* context */) const {
  if (request.GetMethod() != server::http::HttpMethod::kGet) {
      throw server::handlers::ClientError(server::handlers::ExternalBody{ fmt::format("Unsupported method {}", request.GetMethod()) });
  }
  auto& offer_id = request.GetPathArg("offer_id");
  if (offer_id.empty()) {
    throw server::handlers::ClientError(server::handlers::ExternalBody{"No 'offer_id' query argument"});
  }
  const auto result = redis_client_->Get(std::move(offer_id), redis_cc_).Get();
  if (!result) {
    request.SetResponseStatus(server::http::HttpStatus::kNotFound);
    return {};
  }
  return *result;
}

drive::handlers::HandleOfferProcessor::HandleOfferProcessor(const components::ComponentConfig& config, const components::ComponentContext& context)
  : server::handlers::HttpHandlerBase(config, context)
  , redis_client_{ context.FindComponent<components::Redis>().GetClient("offers-cache-api") }
{
}

std::string drive::handlers::HandleOfferProcessor::HandleRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& /* context */) const {
  const auto& offer_id = request.GetPathArg("offer_id");
  const auto& user_id = request.GetPathArg("user_id");
  if (offer_id.empty() || user_id.empty()) {
    throw server::handlers::ClientError(server::handlers::ExternalBody{"No 'offer_id' or 'user_id' query argument"});
  }
  std::string user_key = MakeUserKey(user_id, offer_id);
  switch (request.GetMethod()) {
    case server::http::HttpMethod::kGet:
      {
        const auto result = redis_client_->Get(std::move(user_key), redis_cc_).Get();
        if (!result) {
          request.SetResponseStatus(server::http::HttpStatus::kNotFound);
          return {};
        }
        return *result;
      }
    case server::http::HttpMethod::kDelete:
      {
        const auto result = redis_client_->Del({offer_id, user_key}, redis_cc_).Get();
        return std::to_string(result);
      }
    default:
      throw server::handlers::ClientError(server::handlers::ExternalBody{ fmt::format("Unsupported method {}", request.GetMethod()) });
  }
}

drive::handlers::CountOffersProcessor::CountOffersProcessor(const components::ComponentConfig& config, const components::ComponentContext& context)
  : server::handlers::HttpHandlerBase(config, context)
  , redis_client_{ context.FindComponent<components::Redis>().GetClient("offers-cache-api") }
{
}

std::string drive::handlers::CountOffersProcessor::HandleRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& /* context */) const {
  if (request.GetMethod() != server::http::HttpMethod::kGet) {
      throw server::handlers::ClientError(server::handlers::ExternalBody{ fmt::format("Unsupported method {}", request.GetMethod()) });
  }
  const auto& user_id = request.GetPathArg("user_id");
  const auto& object_id = request.GetPathArg("object_id");
  if (object_id.empty() || user_id.empty()) {
    throw server::handlers::ClientError(server::handlers::ExternalBody{"No 'object_id' or 'user_id' query argument"});
  }
  std::map<std::string, size_t> counts;
  storages::redis::ScanOptions opt{
    storages::redis::ScanOptions::Match(fmt::format("{{{}}}:*", user_id))
  };
  for (size_t i(0); i < redis_client_->ShardsCount(); ++i) {
    auto keys = redis_client_->Scan(i, opt, redis_cc_).GetAll();
    const auto offers = redis_client_->Mget(std::move(keys), redis_cc_).Get();
    for (const auto& offer : offers) {
      if (!offer) {
        continue;
      }
      const auto json = formats::json::FromString(*offer);
      if (json.HasMember("object_id") && json["object_id"].As<std::string>() == object_id) {
        auto constructor_id = json["constructor_id"].As<std::string>();
        if (auto it = counts.find(constructor_id); it != counts.end()) {
          it->second += 1;
        } else {
          counts[std::move(constructor_id)] += 1;
        }
      }
    }
  }
  size_t result = 0;
  for (const auto& [_, count] : counts) {
    result = std::max(result, count);
  }
  return std::to_string(result);
}
