#include "processor.hpp"

#include <drive/services/telematics_cache_api/src/models/telematics_data_types.hpp>

#include <drive/services/userver_libraries/redis_utils/utils.hpp>

#include <fmt/format.h>

#include <userver/components/component_context.hpp>
#include <userver/crypto/base64.hpp>
#include <userver/dynamic_config/value.hpp>
#include <userver/formats/json/exception.hpp>
#include <userver/formats/json/value.hpp>
#include <userver/logging/level.hpp>
#include <userver/logging/log.hpp>
#include <userver/logging/log_filepath.hpp>
#include <userver/logging/log_helper.hpp>
#include <userver/logging/logger.hpp>
#include <userver/tracing/span.hpp>
#include <userver/utils/datetime.hpp>

#include <chrono>
#include <map>
#include <stdexcept>
#include <vector>

namespace {

using DataType = drive::models::tca::telematics_data_types::DataType;
using SensorId = drive::models::tca::telematics_data_types::SensorId;

int64_t ParseBatchSizeCfg(const dynamic_config::DocsMap& docs_map) {
  return docs_map.Get("TELEMATICS_CACHE_API_REDIS_BULK_STORE_BATCH_SIZE").As<int64_t>();
}
int ParseFutureTimestampDeltaSecondsCfg(const dynamic_config::DocsMap& docs_map) {
  return docs_map.Get("TELEMATICS_CACHE_API_FUTURE_TIMESTAMP_DELTA_SECONDS").As<int>();
}
constexpr dynamic_config::Key<ParseBatchSizeCfg> kRedisBatchSizeConfig{};
constexpr dynamic_config::Key<ParseFutureTimestampDeltaSecondsCfg> kFutureTimestampDeltaSecondsConfig{};

constexpr auto kRedisMGetSettings = userver_libraries::redis_utils::BuildRedisCc(/*force_request_to_master = */ false);
constexpr auto kRedisMSetSettings = userver_libraries::redis_utils::BuildRedisCc(/*force_request_to_master = */ true);

struct RequestBodyItem {
  uint32_t unix_timestamp;
  std::string data_proto;
  std::string key;
};

DataType ParseDataTypeStr(const std::string& data_type_str) {
    if (data_type_str == "sensor") {
      return DataType::kSensor;
    } else if (data_type_str == "location") {
      return DataType::kLocation;
    } else if (data_type_str == "heartbeat") {
      return DataType::kHeartbeat;
    } else if (data_type_str == "handler") {
      return DataType::kHandler;
    } else {
      throw std::invalid_argument("data_type must be one of {'sensor', 'location', 'heartbeat', 'handler'}");
    }
}

RequestBodyItem Parse(const formats::json::Value& json,
                formats::parse::To<RequestBodyItem>) {
  try {
    const auto data_type = ParseDataTypeStr(json["data_type"].As<std::string>());
    std::string key;
    switch (data_type) {
      case DataType::kSensor:
        key = drive::models::tca::telematics_data_types::GetSensorUrl(json["imei"].As<std::string>(), json["sensor_id"].As<SensorId>());
        break;
      case DataType::kHeartbeat:
        key = drive::models::tca::telematics_data_types::GetHeartbeatUrl(json["imei"].As<std::string>(), json["heartbeat_name"].As<std::string>(""));
        break;
      case DataType::kLocation:
        key = drive::models::tca::telematics_data_types::GetLocationUrl(json["imei"].As<std::string>(), json["location_name"].As<std::string>());
        break;
      case DataType::kHandler:
        key = drive::models::tca::telematics_data_types::GetHandlerUrl(json["handler_id"].As<std::string>());
        break;
    }
    return RequestBodyItem{
        json["unix_timestamp"].As<uint32_t>(),
        json["data_proto"].As<std::string>(""),
        std::move(key)
    };
  }
  catch (const std::exception& ex) {
    throw server::handlers::ClientError(server::handlers::ExternalBody{
        fmt::format("Failed to parse request_body_item: {}", ex.what())});
  }
}

struct TimestampAndData {
  uint32_t unix_timestamp;
  std::string data_proto;

  TimestampAndData() = default;
  TimestampAndData(uint32_t unix_timestamp, std::string&& data_proto) : unix_timestamp(unix_timestamp), data_proto(std::move(data_proto)) {}
};

formats::json::Value ParseToJsonArray(const server::http::HttpRequest& request) {
  const auto& request_body_raw = request.RequestBody();
  try {
    const auto request_body_json = formats::json::FromString(request_body_raw);
    request_body_json.CheckNotMissing();
    request_body_json.CheckObject();
    request_body_json["data_array"].CheckArrayOrNull();
    return request_body_json["data_array"];
  }
  catch (const std::exception& ex) {
    throw server::handlers::ClientError(server::handlers::ExternalBody{
        fmt::format("Failed to parse request_body to json array: {}", ex.what())});
  }
}

bool IsFutureTimestamp(const uint32_t unix_timestamp, const uint32_t delta) {
  const auto now = utils::datetime::Now();
  const auto now_unix_timestamp = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count();
  if (unix_timestamp > now_unix_timestamp + delta) {
    LOG_ERROR() << fmt::format("Future timestamp. Now = {}, request unix_timestamp = {}, delta = {}",
      now_unix_timestamp, unix_timestamp, delta);
    return true;
  }
  return false;
}

bool NeedUpdateData(const uint32_t request_unix_timestamp, const std::string& db_data) {
  const auto pos = db_data.find(userver_libraries::redis_utils::symbol_not_used_in_base64);
  if (pos == std::string::npos) {
    return true; // found old format value
  }
  const uint32_t db_unix_timestamp = std::stoi(db_data.substr(0, pos));
  return request_unix_timestamp > db_unix_timestamp;
}

}  // namespace

namespace drive::handlers::api_telematics_cache_api_v1_data_bulk_store {

Processor::Processor(const components::ComponentConfig& config, const components::ComponentContext& context)
    : server::handlers::HttpHandlerBase(config, context)
    , redis_client_{ context.FindComponent<components::Redis>().GetClient("telematics-cache-api") }
    , config_{ context.FindComponent<components::DynamicConfig>().GetSource() }
{
}

std::string Processor::HandlePostRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& /*context*/) const {
  tracing::Span span("HandlePostRequestThrow (ms)");

  const auto runtime_config = config_.GetSnapshot();
  const uint32_t delta = runtime_config[kFutureTimestampDeltaSecondsConfig];
  const size_t batch_size_config = runtime_config[kRedisBatchSizeConfig];

  size_t upsert_counter = 0;
  const auto json_array = ParseToJsonArray(request);

  const size_t batch_size = std::min(json_array.GetSize(), batch_size_config);
  std::map<std::string, TimestampAndData> keys_to_data;
  std::vector<std::string> keys_batch;
  keys_batch.reserve(batch_size);
  std::vector<std::pair<std::string, std::string>> key_values_to_upsert_buffer;
  key_values_to_upsert_buffer.reserve(batch_size);
  for (size_t i = 0; i < json_array.GetSize(); ++i) {
    auto request_body_item = json_array[i].As<RequestBodyItem>();
    if (!IsFutureTimestamp(request_body_item.unix_timestamp, delta)) {
      keys_batch.push_back(request_body_item.key);
      keys_to_data.emplace(std::piecewise_construct,
                          std::forward_as_tuple(std::move(request_body_item.key)),
                          std::forward_as_tuple(request_body_item.unix_timestamp,
                                                std::move(request_body_item.data_proto)));
    }
    if (keys_batch.size() == batch_size || i == json_array.GetSize() - 1) {
      std::vector<std::optional<std::string>> redis_optional_values;
      {
        tracing::Span span("Redis MGET request (ms)");
        redis_optional_values = redis_client_->Mget(keys_batch, kRedisMGetSettings).Get();
      }
      if (keys_batch.size() != redis_optional_values.size()) {
        throw std::runtime_error("Redis MGET array's size is not equal to keys' size");
      }
      for (size_t j = 0; j < keys_batch.size(); ++j) {
        const auto& timestamp_and_data = keys_to_data[keys_batch[j]];
        if (!redis_optional_values[j] || NeedUpdateData(timestamp_and_data.unix_timestamp, *redis_optional_values[j])) {
          auto new_value = fmt::format("{}{}{}",
                                       timestamp_and_data.unix_timestamp,
                                       userver_libraries::redis_utils::symbol_not_used_in_base64,
                                       crypto::base64::Base64Encode(timestamp_and_data.data_proto));
          key_values_to_upsert_buffer.emplace_back(std::move(keys_batch[j]), std::move(new_value));
        }
        if (key_values_to_upsert_buffer.size() == batch_size) {
          tracing::Span span("Redis MSET request (ms)");
          upsert_counter += key_values_to_upsert_buffer.size();
          redis_client_->Mset(std::move(key_values_to_upsert_buffer), kRedisMSetSettings).IgnoreResult();
        }
      }
      keys_batch.clear();
      keys_to_data.clear();
    }

  }
  if (!key_values_to_upsert_buffer.empty()) {
    tracing::Span span("Redis MSET request (ms)");
    upsert_counter += key_values_to_upsert_buffer.size();
    redis_client_->Mset(std::move(key_values_to_upsert_buffer), kRedisMSetSettings).IgnoreResult();
  }

  request.SetResponseStatus(server::http::HttpStatus::kOk);
  return fmt::format("{{\"written\": {}}}", upsert_counter);
}

std::string Processor::HandleRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& context) const {
  switch (request.GetMethod()) {
    case server::http::HttpMethod::kPost:
      return HandlePostRequestThrow(request, context);
    default:
      throw server::handlers::ClientError(server::handlers::ExternalBody{
          fmt::format("Unsupported method {}", request.GetMethod())});
  }
}

}  // namespace drive::handlers::api_telematics_cache_api_v1_data_bulk_store
