#include "processor.hpp"

#include <drive/services/userver_libraries/redis_utils/utils.hpp>

#include <fmt/format.h>

#include <userver/components/component_context.hpp>
#include <userver/dynamic_config/value.hpp>
#include <userver/formats/json/value.hpp>
#include <userver/utils/datetime.hpp>

#include <stdexcept>
#include <vector>

namespace {

int ParseRuntimeCfg(const dynamic_config::DocsMap& docs_map) {
  return docs_map.Get("TELEMATICS_CACHE_API_FUTURE_TIMESTAMP_DELTA_SECONDS").As<int>();
}
constexpr dynamic_config::Key<ParseRuntimeCfg> kFutureTimestampDeltaConfig{};

constexpr auto kRedisPutSettings = userver_libraries::redis_utils::BuildRedisCc(/*force_request_to_master = */ true);

const std::string kCheckTimestampAndUpsertKey = R"(
local key = KEYS[1];
local old_timestamp = tonumber(redis.call("HGET", key, "timestamp"));
local new_timestamp = tonumber(ARGV[1]);

if (not old_timestamp or new_timestamp > old_timestamp) then
  redis.call("HSET", key, "timestamp", new_timestamp, "data_proto",  ARGV[2]);
  return 1;
end

return 0;
)";

struct PutRequestBody {
  std::string data_proto;
  uint32_t unix_timestamp;
  std::string key;
};

PutRequestBody GetPutRequestBody(const server::http::HttpRequest& request) {
  const auto& request_body_raw = request.RequestBody();
  try {
    const auto request_body_json = formats::json::FromString(request_body_raw);
    request_body_json.CheckNotMissing();
    request_body_json.CheckObject();
    PutRequestBody request_body;

    auto data_proto = request_body_json["data_proto"].As<std::string>();
    if (data_proto.empty()) {
      throw std::runtime_error("Value of request_body['data_proto'] must be nonempty");
    }

    const auto unix_timestamp = request_body_json["unix_timestamp"].As<int64_t>();
    if (unix_timestamp < 0) {
      throw std::runtime_error("Value of request_body['unix_timestamp'] must be nonnegative");
    }

    auto key = request_body_json["key"].As<std::string>();
    if (key.empty()) {
      throw std::runtime_error("Value of request_body['key'] must be nonempty");
    }

    request_body.data_proto = std::move(data_proto);
    request_body.unix_timestamp = unix_timestamp;
    request_body.key = std::move(key);
    return request_body;
  } catch (const std::exception& ex) {
    throw server::handlers::ClientError(server::handlers::ExternalBody{
        fmt::format("Failed to parse request_body: {}", ex.what())});
  }
}

}  // namespace

namespace drive::handlers::api_telematics_cache_api_v1_data {

Processor::Processor(const components::ComponentConfig& config, const components::ComponentContext& context)
    : server::handlers::HttpHandlerBase(config, context)
    , redis_client_{ context.FindComponent<components::Redis>().GetClient("telematics-cache-api") }
    , config_{ context.FindComponent<components::DynamicConfig>().GetSource() }
{
}

std::string Processor::HandlePutRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& /*context*/) const {
  auto request_body = GetPutRequestBody(request);

  const auto runtime_config = config_.GetSnapshot();
  const uint32_t delta = runtime_config[kFutureTimestampDeltaConfig];

  const auto now = utils::datetime::Now();
  const auto now_unix_timestamp = std::chrono::duration_cast<std::chrono::seconds>(now.time_since_epoch()).count();
  if (request_body.unix_timestamp > now_unix_timestamp + delta) {
    throw server::handlers::ClientError(server::handlers::ExternalBody{
      fmt::format("Future timestamp. Now = {}, request_body.unix_timestamp = {}, delta = {}",
      now_unix_timestamp, request_body.unix_timestamp, delta)
    });
  }

  const auto is_written = redis_client_->Eval<size_t>(
    kCheckTimestampAndUpsertKey,
    std::vector{std::move(request_body.key)},
    std::vector{std::to_string(request_body.unix_timestamp), std::move(request_body.data_proto)},
    kRedisPutSettings
  ).Get();

  request.SetResponseStatus(server::http::HttpStatus::kOk);
  return fmt::format("{{\"written\": {}}}", is_written);
}

std::string Processor::HandleGetRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& /*context*/) const {
  const auto& key = request.GetArg("key");
  if (key.empty()) {
    throw server::handlers::ClientError(
        server::handlers::ExternalBody{"No 'key' query argument"});
  }
  const bool force_request_to_master = request.GetArg("force_request_to_master") == "true" ? true : false;

  auto data_proto_optional = redis_client_->Hget(key, "data_proto", userver_libraries::redis_utils::BuildRedisCc(force_request_to_master)).Get();
  if (!data_proto_optional) {
    throw server::handlers::ResourceNotFound(
        server::handlers::ExternalBody{fmt::format("Not found key {} in Redis", key)});
  }

  return fmt::format("{{\"data_proto\": \"{}\"}}", std::move(*data_proto_optional));
}

std::string Processor::HandleRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& context) const {
  switch (request.GetMethod()) {
    case server::http::HttpMethod::kGet:
      return HandleGetRequestThrow(request, context);
    case server::http::HttpMethod::kPut:
      return HandlePutRequestThrow(request, context);
    default:
      throw server::handlers::ClientError(server::handlers::ExternalBody{
          fmt::format("Unsupported method {}", request.GetMethod())});
  }
}

}  // namespace drive::handlers::api_telematics_cache_api_v1_data
