#include "processor.hpp"

#include <drive/services/telematics_cache_api/src/models/telematics_data_types.hpp>

#include <drive/services/userver_libraries/redis_utils/utils.hpp>

#include <fmt/format.h>

#include <userver/components/component_context.hpp>
#include <userver/crypto/base64.hpp>
#include <userver/dynamic_config/value.hpp>
#include <userver/formats/json/value.hpp>
#include <userver/formats/json/value_builder.hpp>
#include <userver/storages/redis/impl/reply.hpp>
#include <userver/tracing/span.hpp>

#include <algorithm>
#include <optional>
#include <stdexcept>
#include <vector>

#include <span>

namespace {

using DataType = drive::models::tca::telematics_data_types::DataType;
using SensorId = drive::models::tca::telematics_data_types::SensorId;

int64_t ParseBatchSizeCfg(const dynamic_config::DocsMap& docs_map) {
  return docs_map.Get("TELEMATICS_CACHE_API_REDIS_RETRIEVE_BATCH_SIZE").As<int64_t>();
}
constexpr dynamic_config::Key<ParseBatchSizeCfg> kRedisBatchSizeConfig{};

struct RedisResponse {
  storages::redis::RequestMget response_future;
  std::span<std::string> keys_batch_view;

  template<class... Args>
  RedisResponse(storages::redis::RequestMget&& response_future, Args&&... args) : response_future(std::move(response_future)), keys_batch_view(std::forward<Args>(args)...) {}
};

struct PostRequestBody {
  std::vector<std::string> imeis;
  std::vector<std::string> location_names;
  std::vector<std::string> heartbeat_names;
  std::vector<SensorId> sensor_ids;
  std::vector<std::string> handler_ids;
  DataType data_type;
};

PostRequestBody GetPostRequestBody(const server::http::HttpRequest& request) {
  tracing::Span span("GetPostRequestBody");

  const auto& request_body_raw = request.RequestBody();
  try {
    const auto request_body_json = formats::json::FromString(request_body_raw);
    request_body_json.CheckNotMissing();
    request_body_json.CheckObject();
    if (!request_body_json.HasMember("data_type")) {
      throw std::runtime_error("The request must contain field 'data_type'");
    }
    PostRequestBody request_body;
    const auto data_type_str = request_body_json["data_type"].As<std::string>();
    if (data_type_str == "sensor") {
      request_body_json["sensor_ids"].CheckArrayOrNull();
      if (!request_body_json["sensor_ids"].IsEmpty()) {
        request_body.sensor_ids = request_body_json["sensor_ids"].As<std::vector<SensorId>>();
      } else {
        throw std::runtime_error("Expected nonempty sensor_ids array for data_type 'sensor'");
      }
      request_body.data_type = DataType::kSensor;
    } else if (data_type_str == "location") {
      request_body_json["location_names"].CheckArrayOrNull();
      if (!request_body_json["location_names"].IsEmpty()) {
        request_body.location_names = request_body_json["location_names"].As<std::vector<std::string>>();
      } else {
        throw std::runtime_error("Expected nonempty location_names array for data_type 'location'");
      }
      request_body.data_type = DataType::kLocation;
    } else if (data_type_str == "heartbeat") {
      if (request_body_json.HasMember("heartbeat_names")) {
        request_body_json["heartbeat_names"].CheckArrayOrNull();
        request_body.heartbeat_names = request_body_json["heartbeat_names"].As<std::vector<std::string>>();
      }
      request_body.data_type = DataType::kHeartbeat;
    } else if (data_type_str == "handler") {
      request_body_json["handler_ids"].CheckArrayOrNull();
      request_body.handler_ids = request_body_json["handler_ids"].As<std::vector<std::string>>();
      request_body.data_type = DataType::kHandler;
    } else {
      throw std::invalid_argument("data_type must be one of {'sensor', 'location', 'heartbeat', 'handler'}");
    }
    if (request_body.data_type != DataType::kHandler) {
      request_body_json["imeis"].CheckArrayOrNull();
      if (!request_body_json["imeis"].IsEmpty()) {
        request_body.imeis = request_body_json["imeis"].As<std::vector<std::string>>();
      } else {
        throw std::runtime_error("Expected nonempty imeis array for data_type in {'sensor', 'location', 'heartbeat'}");
      }
    }
    return request_body;
  } catch (const std::exception& ex) {
    throw server::handlers::ClientError(server::handlers::ExternalBody{
        fmt::format("Failed to parse request_body: {}", ex.what())});
  }
}

std::vector<std::string> GetSensorDbKeys(const PostRequestBody& request_body) {
  std::vector<std::string> result;
  result.reserve(request_body.imeis.size() * request_body.sensor_ids.size());
  for (const auto& imei : request_body.imeis) {
    for (const auto& sensor_id : request_body.sensor_ids) {
      result.push_back(drive::models::tca::telematics_data_types::GetSensorUrl(imei, sensor_id));
    }
  }
  return result;
}

std::vector<std::string> GetLocationDbKeys(const PostRequestBody& request_body) {
  std::vector<std::string> result;
  result.reserve(request_body.imeis.size() * request_body.location_names.size());
  for (const auto& imei : request_body.imeis) {
    for (const auto& name : request_body.location_names) {
      result.push_back(drive::models::tca::telematics_data_types::GetLocationUrl(imei, name));
    }
  }
  return result;
}

std::vector<std::string> GetHeartbeatDbKeys(const PostRequestBody& request_body) {
  static constexpr const char* kDefaultHeartbeatName = "";

  std::vector<std::string> result;
  result.reserve(request_body.imeis.size() * std::max<size_t>(1, request_body.heartbeat_names.size()));
  if (request_body.heartbeat_names.empty()) {
    for (const auto& imei : request_body.imeis) {
      result.push_back(drive::models::tca::telematics_data_types::GetHeartbeatUrl(imei, kDefaultHeartbeatName));
    }
  } else {
    for (const auto& imei : request_body.imeis) {
      for (const auto& name : request_body.heartbeat_names) {
        result.push_back(drive::models::tca::telematics_data_types::GetHeartbeatUrl(imei, name));
      }
    }
  }
  return result;
}

std::vector<std::string> GetHandlerDbKeys(const PostRequestBody& request_body) {
  std::vector<std::string> result;
  result.reserve(request_body.handler_ids.size());
  for (const auto& handler_id : request_body.handler_ids) {
    result.push_back(drive::models::tca::telematics_data_types::GetHandlerUrl(handler_id));
  }
  return result;
}

}  // namespace

namespace drive::handlers::api_telematics_cache_api_v1_data_retrieve {

Processor::Processor(const components::ComponentConfig& config, const components::ComponentContext& context)
    : server::handlers::HttpHandlerBase(config, context)
    , redis_client_{ context.FindComponent<components::Redis>().GetClient("telematics-cache-api") }
    , config_{ context.FindComponent<components::DynamicConfig>().GetSource() }
{
}

std::string Processor::HandlePostRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& /*context*/) const {
  tracing::Span span("HandlePostRequestThrow (ms)");

  auto request_body = GetPostRequestBody(request);

  std::vector<std::string> db_keys;
  switch (request_body.data_type) {
    case DataType::kSensor:
      db_keys = GetSensorDbKeys(request_body);
      break;
    case DataType::kHeartbeat:
      db_keys = GetHeartbeatDbKeys(request_body);
      break;
    case DataType::kLocation:
      db_keys = GetLocationDbKeys(request_body);
      break;
    case DataType::kHandler:
      db_keys = GetHandlerDbKeys(request_body);
      break;
  }

  if (db_keys.empty()) {
    request.SetResponseStatus(server::http::HttpStatus::kOk);
    return "{\"data_proto_array\": []}";
  }

  const bool force_request_to_master = request.GetArg("force_request_to_master") == "true" ? true : false;

  const auto runtime_config = config_.GetSnapshot();
  const size_t batch_size = std::min<size_t>(db_keys.size(), runtime_config[kRedisBatchSizeConfig]);
  const auto redis_queries_number = static_cast<size_t>(std::ceil(1.0 * db_keys.size() / batch_size));

  std::vector<formats::json::Value> data_proto_array;
  data_proto_array.reserve(db_keys.size());
  std::vector<std::optional<std::string>> redis_optional_values;
  std::vector<RedisResponse> redis_responses;
  redis_responses.reserve(redis_queries_number);

  for (size_t i = 0; i < redis_queries_number; ++i) {
    auto begin = db_keys.begin() + i * batch_size;
    auto end = std::min(begin + batch_size, db_keys.end());
    auto db_future = redis_client_->Mget(std::vector<std::string>(begin, end), userver_libraries::redis_utils::BuildRedisCc(force_request_to_master));
    redis_responses.emplace_back(std::move(db_future), db_keys.data() + i * batch_size, end - begin);
  }

  for (auto& redis_response : redis_responses) {
    const auto redis_optional_values = redis_response.response_future.Get();
    if (redis_response.keys_batch_view.size() != redis_optional_values.size()) {
      throw std::runtime_error("Redis MGET array's size is not equal to keys' size");
    }
    for (size_t i = 0; i < redis_response.keys_batch_view.size(); ++i) {
      if (!redis_optional_values[i]) {
        continue;
      }
      const auto& redis_value = *redis_optional_values[i];
      const auto pos = redis_value.find(userver_libraries::redis_utils::symbol_not_used_in_base64);
      if (pos == std::string::npos || pos == redis_value.size() - 1) {
        continue;
      }
      formats::json::ValueBuilder item(formats::common::Type::kObject);
      if (auto optional_imei = drive::models::tca::telematics_data_types::GetImeiFromUrl(redis_response.keys_batch_view[i])) {
        item["imei"] = std::move(*optional_imei);
      }
      const std::string_view encoded_data_proto_view(redis_value.data() + pos + 1, redis_value.size() - pos - 1);
      item["data_proto"] = crypto::base64::Base64Decode(encoded_data_proto_view);
      data_proto_array.push_back(item.ExtractValue());
    }
  }

  formats::json::ValueBuilder data_proto_array_builder(formats::common::Type::kArray);
  data_proto_array_builder.Resize(data_proto_array.size());
  std::move(data_proto_array.begin(), data_proto_array.end(), data_proto_array_builder.begin());

  formats::json::ValueBuilder response_builder(formats::common::Type::kObject);
  response_builder["data_proto_array"] = data_proto_array_builder.ExtractValue();
  return ToString(response_builder.ExtractValue());
}

std::string Processor::HandleRequestThrow(const server::http::HttpRequest& request, server::request::RequestContext& context) const {
  switch (request.GetMethod()) {
    case server::http::HttpMethod::kPost:
      return HandlePostRequestThrow(request, context);
    default:
      throw server::handlers::ClientError(server::handlers::ExternalBody{
          fmt::format("Unsupported method {}", request.GetMethod())});
  }
}

}  // namespace drive::handlers::api_telematics_cache_api_v1_data_retrieve
