#include "client.h"

#include <drive/telematics/server/tasks/lite.h>

#include <drive/library/cpp/network/data/data.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/library/executor/executor.h>
#include <rtline/library/unistat/signals.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/string/builder.h>
#include <util/string/vector.h>
#include <util/system/hostname.h>

namespace {
    NJson::TJsonValue ViewTask(TStringBuf fqdn, TStringBuf id, TDuration timeout = TDuration::MilliSeconds(100)) {
        NJson::TJsonValue result;

        auto host = fqdn.Before(':');
        ui16 port = FromStringWithDefault(fqdn.After(':'), 80);
        result["host"] = host;
        result["port"] = port;
        result["id"] = id;
        if (!port) {
            result["error"] = "cannot determine port";
            return result;
        }

        const TString request = TStringBuilder() << "http://" << host << ':' << port << "/task/view/?id=" << id;
        const NNeh::TMessage message = NNeh::TMessage::FromString(request);
        const NNeh::THandleRef handle = NNeh::Request(message);
        const NNeh::TResponseRef response = handle ? handle->Wait(timeout) : nullptr;
        if (response && !response->IsError()) {
            if (!NJson::ReadJsonFastTree(response->Data, &result)) {
                result["data"] = response->Data;
            }
        } else {
            result["error"] = response ? response->GetErrorText() : "no response";
        }
        return result;
    }
}

class TCallbacksSelector: public IQueueViewSelector {
public:
    using IQueueViewSelector::IQueueViewSelector;

    virtual ~TCallbacksSelector() = default;

    bool IsAvailable(const TString& taskId) const override {
        return TFsPath(taskId).GetName().StartsWith("callback") || TFsPath(taskId).GetName() == "deprecated-cleaner";
    }
};

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::THandler::MakeError(const TString& error) {
    THandler result;
    result.Error = error;
    return result;
}

NDrive::TTelematicsApi::THandler::THandler() {
}

NDrive::TTelematicsApi::THandler::THandler(const TString& id)
    : Id(id)
{
}

NDrive::TTelematicsApi::THandler::THandler(THandler&& other)
    : Id(std::move(other.Id))
    , Data(std::move(other.Data))
{
}

NDrive::TTelematicsApi::THandler& NDrive::TTelematicsApi::THandler::operator=(THandler&& other) {
    Id = std::move(other.Id);
    Data = std::move(other.Data);
    return *this;
}

NDrive::TTelematicsApi::THandler::~THandler() {
}

NDrive::TTelematicsApi::TTelematicsApi(const TTaskExecutorConfig& config, IDistributedTaskContext* context)
    : Host(TStringBuf(NUtil::GetSlotFromFqdn(HostName()).first).Before('.'))
    , Executor(MakeHolder<TTaskExecutor>(config, context))
    , InterattemptPause(TDuration::Seconds(1))
{
    Executor->SetSelector(MakeHolder<TCallbacksSelector>(context));
    Executor->Start();
}

NDrive::TTelematicsApi::~TTelematicsApi() {
    CHECK_WITH_LOG(Executor);
    Executor->Stop();
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::Enqueue(const TCommonDistributedData& data, IDistributedTask& task, IDistributedTask::TPtr callbackTask) const {
    CHECK_WITH_LOG(Executor);
    if (!Executor->StoreData2(&data)) {
        return THandler::MakeError("StoreData2 failure");
    }
    if (!Executor->StoreTask2(&task)) {
        return THandler::MakeError("StoreTask2 failure");
    }
    if (!Executor->EnqueueTask(task.GetIdentifier(), data.GetIdentifier(), "", TInstant::Zero(), data.GetMetaInfo().GetDeadline())) {
        return THandler::MakeError("EnqueueTask failure");
    }
    if (callbackTask) {
        Executor->StoreLocalTask(callbackTask);
        IDTasksQueue::TRestoreDataMeta dataMeta(data.GetIdentifier());
        dataMeta.SetExistanceRequirement(EExistanceRequirement::Finished);
        if (!Executor->EnqueueTask(callbackTask->GetIdentifier(), dataMeta, task.GetIdentifier(), TInstant::Zero(), data.GetMetaInfo().GetCallbackDeadline())) {
            return THandler::MakeError("EnqueueCallbackTask failure");
        }
    }
    DEBUG_LOG << task.GetIdentifier() << ": enqueued telematics task" << Endl;
    return { data.GetIdentifier() };
}

THolder<NDrive::TCommonDistributedData> NDrive::TTelematicsApi::GetDataImpl(const TString& id) const {
    CHECK_WITH_LOG(Executor);
    auto d = Executor->RestoreDataInfo(id);
    Y_ENSURE(d, "incorrect id: " << id);

    auto data = dynamic_cast<TCommonDistributedData*>(d.Get());
    Y_ENSURE(data, "incorrect type: " << d->GetType());
    DEBUG_LOG << id << ": got data " << data->GetType() << " " << NProto::TTelematicsTask::EStatus_Name(data->GetStatus()) << Endl;
    Y_UNUSED(d.Release());
    return THolder(data);
}

THolder<NDrive::TCommonDistributedData> NDrive::TTelematicsApi::GetData(const TString& id) const try {
    return GetDataImpl(id);
} catch (const std::exception& e) {
    DEBUG_LOG << "cannot acquire data id " << id << FormatExc(e) << Endl;
    return nullptr;
}

template <class T>
T NDrive::TTelematicsApi::GetValueImpl(const THandler& handler) const {
    const NDrive::TCommonDistributedData* data = nullptr;
    THolder<NDrive::TCommonDistributedData> localData = nullptr;
    if (handler.Data) {
        data = handler.Data.Get();
    } else {
        localData = GetDataImpl(handler.Id);
        data = localData.Get();
    }
    CHECK_WITH_LOG(data);
    Y_ENSURE(GetStatusImpl(*data) == EStatus::Success, "task is not successful");

    if (auto d = dynamic_cast<const NDrive::TSendCommandDistributedData*>(data)) {
        switch (d->GetCommand()) {
        case NDrive::NVega::ECommandCode::GET_PARAM: {
            NDrive::TSensorRef referenc = Yensured(d->GetParameter())->GetValue();
            NDrive::TSensorValue value = NDrive::SensorValueFromRef(referenc);
            return std::get<T>(std::move(value));
        }
        default:
            ythrow yexception() << "Command " << d->GetCommand() << " does not contain Value";
        }
    }
    if (auto d = dynamic_cast<const NDrive::TInterfaceCommunicationDistributedData*>(data)) {
        if constexpr (std::is_same<T, TBuffer>::value) {
            return d->GetOutput();
        }
        if constexpr (std::is_same<T, TString>::value) {
            const TBuffer& buffer = d->GetOutput();
            return { buffer.Begin(), buffer.End() };
        }
        ythrow yexception() << "Interface output is not castable to " << TypeName<T>();
    }
    if (auto d = dynamic_cast<const NDrive::TDownloadFileDistributedData*>(data)) {
        if constexpr (std::is_same<T, TBuffer>::value) {
            return d->GetData();
        }
        if constexpr (std::is_same<T, TString>::value) {
            const TBuffer& buffer = d->GetData();
            return { buffer.data(), buffer.size() };
        }
        ythrow yexception() << data->GetType() << " output is not castable to " << TypeName<T>();
    }

    ythrow yexception() << data->GetType() << " does not contain Value";
}

template <>
ui64 NDrive::TTelematicsApi::GetValue<ui64>(const NDrive::TTelematicsApi::THandler& handler) const {
    return GetValueImpl<ui64>(handler);
}

template <>
double NDrive::TTelematicsApi::GetValue<double>(const NDrive::TTelematicsApi::THandler& handler) const {
    return GetValueImpl<double>(handler);
}

template <>
TString NDrive::TTelematicsApi::GetValue<TString>(const NDrive::TTelematicsApi::THandler& handler) const {
    return GetValueImpl<TString>(handler);
}

template <>
TBuffer NDrive::TTelematicsApi::GetValue<TBuffer>(const NDrive::TTelematicsApi::THandler& handler) const {
    return GetValueImpl<TBuffer>(handler);
}

NDrive::TSendCommandDistributedData NDrive::TTelematicsApi::BuildCommand(const TString& imei, NDrive::NVega::ECommandCode code, const TDuration taskTimeout, const TDuration dataTimeout, const TDuration callbackTimeout) const {
    NDrive::TCommonDistributedTaskMetaInfo meta(Host, imei, taskTimeout, (callbackTimeout == TDuration::Zero()) ? taskTimeout : callbackTimeout);
    NDrive::TSendCommandDistributedData data(meta);
    if (dataTimeout != TDuration::Max()) {
        data.SetDeadline(Now() + dataTimeout);
    }
    data.SetCommand(code);
    return data;
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::Command(const NDrive::TSendCommandDistributedData& command, IDistributedTask::TPtr callbackTask) const {
    NDrive::TSendCommandDistributedTaskLite task(command.GetMetaInfo());
    return Enqueue(command, task, callbackTask);
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::Command(const TString& imei, NDrive::NVega::ECommandCode code, const TDuration taskTimeout, const TDuration dataTimeout, const TDuration callbackTimeout, IDistributedTask::TPtr callbackTask) const {
    NDrive::TSendCommandDistributedData command = BuildCommand(imei, code, taskTimeout, dataTimeout, callbackTimeout);
    return Command(command, callbackTask);
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::GetParameter(const TString& imei, ui16 id, ui16 subid /*= 0*/, const TDuration taskTimeout, const TDuration dataTimeout) const {
    NDrive::TSendCommandDistributedData data = BuildCommand(imei, NVega::ECommandCode::GET_PARAM, taskTimeout, dataTimeout, TDuration::Zero());
    data.SetId(id);
    data.SetSubId(subid);

    return Command(data);
}

template <bool Polite, class T>
NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::SetParameterImpl(const TString& imei, ui16 id, ui16 subid, T value, const TDuration taskTimeout, const TDuration dataTimeout) const {
    NDrive::TSendCommandDistributedData data = BuildCommand(imei, Polite ? NVega::ECommandCode::SCENARIO_POLITE_SET_PARAM : NVega::ECommandCode::SET_PARAM, taskTimeout, dataTimeout, TDuration::Zero());
    data.SetId(id);
    data.SetSubId(subid);
    data.SetValue(value);

    return Command(data);
}

template <>
NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::SetParameter<ui64>(const TString& imei, ui16 id, ui16 subid, ui64 value, const TDuration taskTimeout, const TDuration dataTimeout) const {
    return SetParameterImpl<false>(imei, id, subid, value, taskTimeout, dataTimeout);
}

template <>
NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::SetParameter<double>(const TString& imei, ui16 id, ui16 subid, double value, const TDuration taskTimeout, const TDuration dataTimeout) const {
    return SetParameterImpl<false>(imei, id, subid, value, taskTimeout, dataTimeout);
}

template <>
NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::SetParameter<TString>(const TString& imei, ui16 id, ui16 subid, TString value, const TDuration taskTimeout, const TDuration dataTimeout) const {
    return SetParameterImpl<false>(imei, id, subid, value, taskTimeout, dataTimeout);
}

template <>
NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::SetParameter<TBuffer>(const TString& imei, ui16 id, ui16 subid, TBuffer value, const TDuration taskTimeout, const TDuration dataTimeout) const {
    return SetParameterImpl<false>(imei, id, subid, value, taskTimeout, dataTimeout);
}

template <>
NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::PoliteSetParameter<ui64>(const TString& imei, ui16 id, ui16 subid, ui64 value, const TDuration taskTimeout, const TDuration dataTimeout) const {
    return SetParameterImpl<true>(imei, id, subid, value, taskTimeout, dataTimeout);
}

template <>
NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::PoliteSetParameter<double>(const TString& imei, ui16 id, ui16 subid, double value, const TDuration taskTimeout, const TDuration dataTimeout) const {
    return SetParameterImpl<true>(imei, id, subid, value, taskTimeout, dataTimeout);
}

template <>
NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::PoliteSetParameter<TString>(const TString& imei, ui16 id, ui16 subid, TString value, const TDuration taskTimeout, const TDuration dataTimeout) const {
    return SetParameterImpl<true>(imei, id, subid, value, taskTimeout, dataTimeout);
}

template <>
NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::PoliteSetParameter<TBuffer>(const TString& imei, ui16 id, ui16 subid, TBuffer value, const TDuration taskTimeout, const TDuration dataTimeout) const {
    return SetParameterImpl<true>(imei, id, subid, value, taskTimeout, dataTimeout);
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::Ping(const TString& imei, TDuration timeout) const {
    NDrive::TCommonDistributedTaskMetaInfo meta(Host, imei, timeout);
    NDrive::TPingDistributedData data(meta);
    NDrive::TPingDistributedTaskLite task(meta);
    return Enqueue(data, task, nullptr);
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::Download(const TString& imei, const TString& filename, const TDuration taskTimeout /*= DefaultUploadTimeout*/, const TDuration dataTimeout /*= DefaultDataLifetime*/) const {
    NDrive::TCommonDistributedTaskMetaInfo meta(Host, imei, taskTimeout);
    NDrive::TDownloadFileDistributedData data(meta);
    data.SetName(filename);
    if (dataTimeout != TDuration::Max()) {
        data.SetDeadline(Now() + dataTimeout);
    }
    NDrive::TDownloadFileDistributedTaskLite task(meta);
    return Enqueue(data, task, nullptr);
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::Upload(const TString& imei, const TString& filename, TBuffer&& content, const TDuration taskTimeout /*= DefaultUploadTimeout*/, const TDuration dataTimeout /*= DefaultDataLifetime*/) const {
    NDrive::TCommonDistributedTaskMetaInfo meta(Host, imei, taskTimeout);
    NDrive::TUploadFileDistributedData data(meta);
    data.SetName(filename);
    data.SetData(std::move(content));
    if (dataTimeout != TDuration::Max()) {
        data.SetDeadline(Now() + dataTimeout);
    }
    NDrive::TUploadFileDistributedTaskLite task(meta);
    return Enqueue(data, task, nullptr);
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::Interface(const TString& imei, NDrive::NVega::TInterfaceData::EInterface interface_, TBuffer&& input, const TDuration taskTimeout /*= DefaultUploadTimeout*/, const TDuration dataTimeout /*= DefaultDataLifetime*/) const {
    NDrive::TCommonDistributedTaskMetaInfo meta(Host, imei, taskTimeout);
    NDrive::TInterfaceCommunicationDistributedData data(meta);
    data.SetInterface(interface_);
    data.SetInput(std::move(input));
    if (dataTimeout != TDuration::Max()) {
        data.SetDeadline(Now() + dataTimeout);
    }
    NDrive::TInterfaceCommunicationDistributedTaskLite task(meta);
    return Enqueue(data, task, nullptr);
}

NDrive::TCanRequestDistributedData NDrive::TTelematicsApi::BuildCanRequest(const TString& imei, ui32 canId, ui8 canIndex, TBuffer&& input, const TDuration taskTimeout /*= DefaultUploadTimeout*/, const TDuration dataTimeout /*= DefaultDataLifetime*/) const {
    NDrive::TCommonDistributedTaskMetaInfo meta(Host, imei, taskTimeout);
    NDrive::TCanRequestDistributedData data(meta);
    data.SetCanId(canId);
    data.SetCanIndex(canIndex);
    data.SetData(std::move(input));
    if (dataTimeout != TDuration::Max()) {
        data.SetDeadline(Now() + dataTimeout);
    }
    return data;
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::CanRequest(const TCanRequestDistributedData& data) const {
    NDrive::TCanRequestDistributedTaskLite task(data.GetMetaInfo());
    return Enqueue(data, task, nullptr);
}

NDrive::TTelematicsApi::THandler NDrive::TTelematicsApi::CanRequest(const TString& imei, ui32 canId, ui8 canIndex, TBuffer&& input, const TDuration taskTimeout, const TDuration dataTimeout) const {
    return CanRequest(BuildCanRequest(imei, canId, canIndex, std::move(input), taskTimeout, dataTimeout));
}

NDrive::TTelematicsApi::EStatus NDrive::TTelematicsApi::GetStatusImpl(const TCommonDistributedData& data) const {
    switch (data.GetStatus()) {
    case NDrive::NProto::TTelematicsTask::NEW:
    case NDrive::NProto::TTelematicsTask::PROCESSING: {
        TInstant now = Now();
        if (data.GetMetaInfo().GetDeadline() > now) {
            return EStatus::Processing;
        } else {
            DEBUG_LOG << data.GetIdentifier() << ": Timeouted " << data.GetMetaInfo().GetDeadline().MicroSeconds() << " " << now.MicroSeconds() << Endl;
            return EStatus::Timeouted;
        }
    }
    case NDrive::NProto::TTelematicsTask::SUCCESS:
        return EStatus::Success;
    default:
        return EStatus::Failure;
    }
}

NDrive::TTelematicsApi::EStatus NDrive::TTelematicsApi::GetStatus(const THandler& handler) const {
    if (handler.Data) {
        return GetStatusImpl(*handler.Data);
    } else {
        auto data = GetDataImpl(handler.Id);
        CHECK_WITH_LOG(data);
        DEBUG_LOG << handler.Id << " data: " << data->GetDataInfo().GetStringRobust() << Endl;
        return GetStatusImpl(*data);
    }
}

NDrive::TTelematicsApi::EStatus NDrive::TTelematicsApi::GetStatus(const THandlers& handlers) const {
    Y_ENSURE(!handlers.empty());
    return GetStatus(handlers.back());
}

TString NDrive::TTelematicsApi::GetMessageImpl(const TCommonDistributedData& data) const {
    TString result;
    if (!result) {
        result = data.GetMessage();
    }
    if (!result) {
        result = NDrive::NProto::TTelematicsTask_EStatus_Name(data.GetStatus());
    }
    return result;
}

TString NDrive::TTelematicsApi::GetMessage(const THandler& handler) const {
    if (handler.Data) {
        return GetMessageImpl(*handler.Data);
    } else {
        auto data = GetDataImpl(handler.Id);
        CHECK_WITH_LOG(data);
        return GetMessageImpl(*data);
    }
}

TString NDrive::TTelematicsApi::GetMessage(const THandlers& handlers) const {
    Y_ENSURE(!handlers.empty());
    return GetMessage(handlers.back());
}

const TTaskExecutor& NDrive::TTelematicsApi::GetExecutor() const {
    CHECK_WITH_LOG(Executor);
    return *Executor;
}

NJson::TJsonValue NDrive::TTelematicsApi::GetSubTasksInfo(const TCommonDistributedData& data) const {
    NJson::TJsonValue result(NJson::JSON_ARRAY);
    TStringBuf fqdn = data.GetHost();
    for (TStringBuf id : data.GetHandlers()) {
        result.AppendValue(ViewTask(fqdn, id));
    }
    return result;
}

NJson::TJsonValue NDrive::TTelematicsApi::GetSubTasksInfo(const THandler& handler) const {
    if (handler.Data) {
        return GetSubTasksInfo(*handler.Data);
    } else {
        auto data = GetDataImpl(handler.Id);
        CHECK_WITH_LOG(data);
        return GetSubTasksInfo(*data);
    }
}

bool NDrive::TTelematicsApi::Await(THandler& handler, bool throwing /*= true*/) const noexcept(false) {
    auto data = GetDataImpl(handler.Id);
    auto status = GetStatusImpl(*data);
    switch (status) {
    case EStatus::Processing:
        return false;
    case EStatus::Success:
        handler.Data = std::move(data);
        return true;
    default:
        handler.Data = std::move(data);
        if (throwing) {
            throw yexception() << Seconds() << " task " << handler.Id << " " << status << ": " << GetMessage(handler);
        } else {
            return true;
        }
    }
}

bool NDrive::TTelematicsApi::Await(THandlers& handlers, ui32 attempts, std::function<THandler()> action, std::function<bool(const THandler&)> check) const noexcept(false) {
    if (!handlers.empty()) {
        auto& handler = handlers.back();
        auto data = GetDataImpl(handler.Id);
        auto status = GetStatusImpl(*data);
        switch (status) {
        case EStatus::Processing:
            return false;
        case EStatus::Success:
            if (!check || check(handlers.back())) {
                handler.Data = std::move(data);
                return true;
            } else {
                DEBUG_LOG << handlers.back().Id << ": Await check failed" << Endl;
                // fallthrough
            }
        default:
            handler.Data = std::move(data);
            if (handlers.size() >= attempts) {
                throw yexception() << Seconds() << " task " << handlers.back().Id << " " << status << ": " << GetMessage(handlers);
            }
        }
    }

    handlers.push_back(action());
    return false;
}

bool NDrive::TTelematicsApi::Wait(THandler& handler) const {
    return Wait(handler, TInstant::Max());
}

bool NDrive::TTelematicsApi::Wait(THandler& handler, TDuration timeout) const {
    return Wait(handler, Now() + timeout);
}

bool NDrive::TTelematicsApi::Wait(THandler& handler, TInstant deadline) const {
    bool result = false;
    for (auto i = TInstant::Zero(); i < deadline; i = Now()) {
        if (result = Await(handler, /*throwing=*/false)) {
            break;
        } else {
            Sleep(std::min(InterattemptPause, deadline - i));
        }
    }
    return result;
}
