#pragma once

#include <rtline/library/executor/abstract/task.h>
#include <drive/telematics/api/sensor/client.h>
#include <drive/telematics/protocol/vega.h>

class IDistributedData;

class TTaskExecutor;
class TTaskExecutorConfig;
class IDistributedTaskContext;

namespace NRTLine {
    class TNehIndexingClient;
}

namespace NDrive {
    class TCommonDistributedData;
    class TCanRequestDistributedData;
    class TSendCommandDistributedData;

    constexpr TDuration DefaultApiTimeout = TDuration::Seconds(30);
    constexpr TDuration DefaultDataLifetime = TDuration::Minutes(20);
    constexpr TDuration DefaultUploadTimeout = TDuration::Minutes(10);

    class TTelematicsApi {
    public:
        enum class EStatus {
            Processing = 0,
            Success = 1,
            Failure = 2,
            Timeouted = 3,
        };

        struct THandler {
        public:
            TString Id;
            THolder<TCommonDistributedData> Data;

        public:
            static THandler MakeError(const TString& error);

        public:
            THandler();
            THandler(const TString& id);

            THandler(THandler&& other);
            THandler& operator=(THandler&& other);

            ~THandler();

            explicit operator bool() const {
                return !Id.empty();
            }

            const TString& GetError() const {
                return Error;
            }
            const TString& GetId() const {
                return Id;
            }

        private:
            TString Error;
        };
        using THandlers = TVector<THandler>;

    public:
        TTelematicsApi(const TTaskExecutorConfig& config, IDistributedTaskContext* context = nullptr);
        ~TTelematicsApi();

        TTelematicsApi(TTelematicsApi&& other) = default;
        TTelematicsApi& operator=(TTelematicsApi&& other) = default;

        TCanRequestDistributedData BuildCanRequest(const TString& imei, ui32 canId, ui8 canIndex, TBuffer&& input, const TDuration taskTimeout = DefaultUploadTimeout, const TDuration dataTimeout = DefaultDataLifetime) const;
        TSendCommandDistributedData BuildCommand(const TString& imei, NDrive::NVega::ECommandCode code, const TDuration taskTimeout, const TDuration dataTimeout, const TDuration callbackTimeout) const;

        THandler Command(const TSendCommandDistributedData& command, IDistributedTask::TPtr callbackTask = nullptr) const;
        THandler Command(const TString& imei, NDrive::NVega::ECommandCode code, const TDuration taskTimeout = DefaultApiTimeout, const TDuration dataTimeout = DefaultDataLifetime, const TDuration callbackTimeout = TDuration::Zero(), IDistributedTask::TPtr callbackTask = nullptr) const;
        THandler GetParameter(const TString& imei, ui16 id, ui16 subid = 0, const TDuration taskTimeout = DefaultApiTimeout, const TDuration dataTimeout = DefaultDataLifetime) const;
        template <class T>
        THandler SetParameter(const TString& imei, ui16 id, ui16 subid, T value, const TDuration taskTimeout = DefaultApiTimeout, const TDuration dataTimeout = DefaultDataLifetime) const;
        template <class T>
        THandler PoliteSetParameter(const TString& imei, ui16 id, ui16 subid, T value, const TDuration taskTimeout = DefaultApiTimeout, const TDuration dataTimeout = DefaultDataLifetime) const;
        THandler Ping(const TString& imei, TDuration timeout = DefaultApiTimeout) const;
        THandler Download(const TString& imei, const TString& filename, const TDuration taskTimeout = DefaultUploadTimeout, const TDuration dataTimeout = DefaultDataLifetime) const;
        THandler Upload(const TString& imei, const TString& filename, TBuffer&& content, const TDuration taskTimeout = DefaultUploadTimeout, const TDuration dataTimeout = DefaultDataLifetime) const;
        THandler Interface(const TString& imei, NDrive::NVega::TInterfaceData::EInterface interface_, TBuffer&& input = {}, const TDuration taskTimeout = DefaultUploadTimeout, const TDuration dataTimeout = DefaultDataLifetime) const;
        THandler CanRequest(const TCanRequestDistributedData& canRequest) const;
        THandler CanRequest(const TString& imei, ui32 canId, ui8 canIndex, TBuffer&& input, const TDuration taskTimeout = DefaultUploadTimeout, const TDuration dataTimeout = DefaultDataLifetime) const;

        THolder<TCommonDistributedData> GetData(const TString& id) const;
        NJson::TJsonValue GetSubTasksInfo(const TCommonDistributedData& data) const;
        NJson::TJsonValue GetSubTasksInfo(const THandler& handler) const;
        EStatus GetStatus(const THandler& handler) const;
        EStatus GetStatus(const THandlers& handlers) const;
        TString GetMessage(const THandler& handler) const;
        TString GetMessage(const THandlers& handlers) const;
        template <class T>
        T GetValue(const THandler& handler) const;
        template <class T>
        T GetValue(const THandlers& handlers) const {
            Y_ENSURE(!handlers.empty());
            return GetValue<T>(handlers.back());
        }

        bool Await(THandler& handler, bool throwing = true) const noexcept(false);
        bool Await(THandlers& handlers, ui32 attempts, std::function<THandler()> action, std::function<bool(const THandler&)> check = nullptr) const noexcept(false);

        bool Wait(THandler& handler) const;
        bool Wait(THandler& handler, TDuration timeout) const;
        bool Wait(THandler& handler, TInstant deadline) const;

        const TTaskExecutor& GetExecutor() const;

    private:
        THandler Enqueue(const TCommonDistributedData& data, IDistributedTask& task, IDistributedTask::TPtr callbackTask) const;
        THolder<TCommonDistributedData> GetDataImpl(const TString& id) const;

        EStatus GetStatusImpl(const TCommonDistributedData& data) const;
        TString GetMessageImpl(const TCommonDistributedData& data) const;
        template <class T>
        T GetValueImpl(const THandler& handler) const;
        template <bool Polite, class T>
        THandler SetParameterImpl(const TString& imei, ui16 id, ui16 subid, T value, const TDuration taskTimeout, const TDuration dataTimeout) const;

    private:
        TString Host;
        THolder<TTaskExecutor> Executor;
        TDuration InterattemptPause;
    };
}
