#include "client.h"

#include <drive/telematics/server/pusher/pusher.h>

#include <drive/telematics/api/sensor/history.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/string/vector.h>
#include <util/stream/output.h>

Y_UNIT_TEST_SUITE(SensorApiSuite) {
    Y_UNIT_TEST(Simple) {
        NRTLine::TNehSearchClient::TBalancerOptions balancerOptions;
        balancerOptions.BalancerTimeoutTable = {
            TDuration::MilliSeconds(250)
        };
        NRTLine::TNehSearchClient client(
            NRTLine::TNehSearchClient::TEndpoint("saas-searchproxy-maps-kv.yandex.net", 17000).SetService("drive_cache").SetBalancerOptions(balancerOptions)
        );
        NDrive::TSensorApi api("test", client);

        TString imei = "867962043095507";
        auto location = api.GetLocation(imei).ExtractValueSync();
        UNIT_ASSERT(location);

        auto sensor = api.GetSensor(imei, VEGA_LAT).ExtractValueSync();
        UNIT_ASSERT(sensor);
        UNIT_ASSERT(sensor->Timestamp > TInstant::Zero());
        UNIT_ASSERT_VALUES_EQUAL(sensor->Id, VEGA_LAT);
        UNIT_ASSERT_VALUES_EQUAL(sensor->SubId, 0);
        auto value = std::get<double>(sensor->Value);
        UNIT_ASSERT(value > 1);
    }

    Y_UNIT_TEST(Ages) {
        NRTLine::TNehSearchClient::TBalancerOptions balancerOptions;
        balancerOptions.BalancerTimeoutTable = {
            TDuration::MilliSeconds(250)
        };
        NRTLine::TNehSearchClient client(
            NRTLine::TNehSearchClient::TEndpoint("saas-searchproxy-maps-prestable.yandex.net", 17000).SetService("drive_cache").SetBalancerOptions(balancerOptions)
        );

        NDrive::TSensorApi::TOptions options;
        options.Timeout = TDuration::Seconds(2);
        NDrive::TSensorApi api("test", client, options);

        NDrive::TSensorApi::TLocations locations = api.GetLocations().ExtractValueSync();
        UNIT_ASSERT(!locations.empty());
        INFO_LOG << locations.size() << Endl;

        NDrive::TSensorApi::TSensors sensors = api.GetSensor(VEGA_MCU_FIRMWARE_VERSION).ExtractValueSync();
        UNIT_ASSERT(!sensors.empty());
        for (auto&& [imei, multisensors] : sensors) {
            UNIT_ASSERT(!imei.empty());
            UNIT_ASSERT(!multisensors.empty());
            for (auto&& sensor : multisensors) {
                UNIT_ASSERT_VALUES_EQUAL(sensor.Id, VEGA_MCU_FIRMWARE_VERSION);
                UNIT_ASSERT_VALUES_EQUAL(sensor.SubId, 0);
            }
        }

        NDrive::TSensorApi::THeartbeats heartbeats = api.GetHeartbeats().ExtractValueSync();
        UNIT_ASSERT(!heartbeats.empty());
        for (auto&& [imei, heartbeat] : heartbeats) {
            UNIT_ASSERT(!imei.empty());
            UNIT_ASSERT(heartbeat.Timestamp > TInstant::Zero());
        }
    }

    Y_UNIT_TEST(Batch) {
        NRTLine::TNehSearchClient::TBalancerOptions balancerOptions;
        balancerOptions.BalancerTimeoutTable = {
            TDuration::MilliSeconds(250)
        };
        NRTLine::TNehSearchClient client(
            NRTLine::TNehSearchClient::TEndpoint("saas-searchproxy-maps-prestable.yandex.net", 17000).SetService("drive_cache").SetBalancerOptions(balancerOptions)
        );

        NDrive::TPusherOptions pusherOptions;
        pusherOptions.Host = "saas-indexerproxy-maps-prestable.yandex.net";
        pusherOptions.Token = "ac6ac7f7a6544f336202c0b058104374";
        NDrive::TPusher pusher(pusherOptions);

        NDrive::TSensorApi::TOptions options;
        options.Timeout = TDuration::Seconds(2);
        NDrive::TSensorApi api("test", client, options);

        {
            NDrive::TSensorId sensorId = { CAN_FUEL_DISTANCE_KM };
            NDrive::TSensorApi::TSensors sensors = api.GetSensor(sensorId).ExtractValueSync();
            TVector<NThreading::TFuture<void>> waits;
            for (auto&&[imei, multisensor] : sensors) {
                UNIT_ASSERT(!imei.empty());
                for (auto&& sensor : multisensor) {
                    UNIT_ASSERT_VALUES_EQUAL(sensor, sensorId);
                    //waits.push_back(pusher.Remove(imei, sensor.Id, sensor.SubId, Now()));
                }
            }
            NThreading::WaitExceptionOrAll(waits).Wait();
        }
        {
            NDrive::TSensorId sensorId = { CAN_FUEL_DISTANCE_KM, CAN_ENGINE_IS_ON };
            NDrive::TSensorApi::TSensors sensors = api.GetSensor(sensorId).ExtractValueSync();
            TVector<NThreading::TFuture<void>> waits;
            for (auto&&[imei, multisensor] : sensors) {
                UNIT_ASSERT(!imei.empty());
                for (auto&& sensor : multisensor) {
                    UNIT_ASSERT_VALUES_EQUAL(sensor, sensorId);
                    //waits.push_back(pusher.Remove(imei, sensor.Id, sensor.SubId, Now()));
                }
            }
            NThreading::WaitExceptionOrAll(waits).Wait();
        }
    }

    Y_UNIT_TEST(Replication) {
        NRTLine::TNehSearchClient::TBalancerOptions balancerOptions;
        NRTLine::TNehSearchClient client(
            NRTLine::TNehSearchClient::TEndpoint("saas-searchproxy-maps-prestable.yandex.net", 17000).SetService("drive_cache").SetBalancerOptions(balancerOptions)
        );

        NDrive::TSensorApi::TOptions options;
        options.Timeout = TDuration::Seconds(2);
        NDrive::TSensorApi api("test", client, options);

        NDrive::TSensorId id(VEGA_MCU_FIRMWARE_VERSION);

        auto regular = api.GetSensor(id).ExtractValueSync();
        UNIT_ASSERT(!regular.empty());

        auto replicated = api.GetSensor(id, {}, 7).ExtractValueSync();
        UNIT_ASSERT(!replicated.empty());
    }
}
