#include "client.h"

#include <rtline/library/unistat/cache.h>
#include <rtline/library/unistat/signals.h>

#include <drive/telematics/server/pusher/common.h>
#include <drive/telematics/server/pusher/pusher.h>

class TSearchDocumentDataAccessor {
public:
    static const NMetaProtocol::TPairBytesBytes& GetAttribute(const NMetaProtocol::TDocument& document, ui32 index) {
        return document.GetArchiveInfo().GetGtaRelatedAttribute(index);
    }

    static ui32 GetAttributeSize(const NMetaProtocol::TDocument& document) {
        return document.GetArchiveInfo().GtaRelatedAttributeSize();
    }

    static const TString& GetKey(const NMetaProtocol::TPairBytesBytes& prop) {
        return prop.GetKey();
    }

    static const TString& GetValue(const NMetaProtocol::TPairBytesBytes& prop) {
        return prop.GetValue();
    }
};

NDrive::TSensorApi::TSensorApi(const TString& name, const NRTLine::TNehSearchClient& client, const TOptions& options /*= Default<TOptions>()*/)
    : TCommonMetadataClient(client, options)
    , Name(name)
{
}

NDrive::TSensorApi::TSensorApi(const TString& name, const NRTLine::TNehSearchClient& search, const NRTLine::TNehIndexingClient& indexing, const TOptions& options /*= Default<TOptions>()*/)
    : TCommonMetadataClient(search, options)
    , Pusher(MakeAtomicShared<TPusher>(indexing, Default<TPusherOptions>()))
    , Name(name)
{
}

NDrive::TSensorApi::TSensorApi(const TString& name, const NRTLine::TNehSearchClient& search, const NDrive::TPusherPtr pusher, const TOptions& options /*= Default<TOptions>()*/)
    : TCommonMetadataClient(search, options)
    , Pusher(std::move(pusher))
    , Name(name)
{
}

NDrive::TPusherPtr NDrive::TSensorApi::GetPusher() const {
    return Pusher;
}

NThreading::TFuture<NDrive::ISensorApi::THeartbeats> NDrive::TSensorApi::DoGetHeartbeats(TIMEIs imeis, THeartbeatsQueryOptions queryOptions) const {
    if (imeis.empty()) {
        return NThreading::MakeFuture(THeartbeats());
    }

    TVector<TString> keys;
    for (auto&& imei : imeis) {
        keys.push_back(GetHeartbeatUrl(imei, queryOptions.GetName()));
    }
    auto query = NRTLine::TQuery();
    if (queryOptions.GetFetchOptions().HasTimeout()) {
        query.SetTimeout(queryOptions.GetFetchOptions().GetTimeoutRef());
    }
    query.AddProperty("Host");
    query.AddProperty("Name");
    query.AddProperty("IMEI");
    query.AddProperty("Timestamp");
    query.AddProperty("Created");
    auto replies = Query(std::move(keys), std::move(query), queryOptions.GetFetchOptions().OptionalMaxKeysPerQuery());
    auto waiter = NThreading::WaitAll(replies);
    return waiter.Apply([
        this,
        queryOptions = std::move(queryOptions),
        replies = std::move(replies)
    ](const NThreading::TFuture<void>& w) {
        ProcessFuture(w);
        ProcessQuorum(replies, queryOptions.GetFetchOptions());
        THeartbeats heartbeats;
        for (auto&& reply : replies) {
            if (!reply.HasValue()) {
                continue;
            }
            FillHeartbeats(reply.GetValue(), heartbeats);
        }
        return heartbeats;
    });
}

NThreading::TFuture<NDrive::ISensorApi::THeartbeats> NDrive::TSensorApi::DoGetHeartbeats(THeartbeatsQueryOptions queryOptions) const {
    auto query = NRTLine::TQuery();
    if (queryOptions.GetFetchOptions().HasTimeout()) {
        query.SetTimeout(queryOptions.GetFetchOptions().GetTimeoutRef());
    }
    query.AddProperty("Host");
    query.AddProperty("Created");
    query.AddProperty("Timestamp");
    auto reply = HeavyQuery(GetHeartbeatKey(queryOptions.GetName()), "Sensor", std::move(query));
    return reply.Apply([this](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        ProcessFuture(r);
        THeartbeats heartbeats;
        FillHeartbeats(r.GetValue(), heartbeats);
        return heartbeats;
    });
}

void NDrive::TSensorApi::FillHeartbeats(const NRTLine::TSearchReply& reply, THeartbeats& heartbeats) const {
    reply.CheckSucceeded("heartbeats fetch has been unsuccessful");
    {
        reply.ScanDocs([&heartbeats] (const NMetaProtocol::TDocument& document) {
            TString imei;
            THeartbeat heartbeat;
            try {
            for (auto&& propertie : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
                const auto& key = propertie.GetKey();
                const auto& value = propertie.GetValue();
                if (heartbeat.Timestamp == TInstant::Zero() && key == "Timestamp") {
                    heartbeat.Timestamp = TInstant::Seconds(FromString<ui64>(value));
                    continue;
                }
                if (heartbeat.Created == TInstant::Zero() && key == "Created") {
                    heartbeat.Created = TInstant::Seconds(FromString<ui64>(value));
                    continue;
                }
                if (heartbeat.Host.empty() && key == "Host") {
                    heartbeat.Host = value;
                    continue;
                }
                if (heartbeat.Name.empty() && key == "Name") {
                    heartbeat.Name = value;
                    continue;
                }
                if (imei.empty() && (key == "IMEI" || key == "_SK_imei")) {
                    imei = value;
                    continue;
                }
            }
            if (imei) {
                heartbeats.emplace(imei, heartbeat);
            }
            } catch (const std::exception& e) {
                ERROR_LOG << "cannot parse heartbeat " << document.DebugString() << ": " << FormatExc(e) << Endl;
            }
        });
    }
}

namespace {
    class TSignalLocationType: public TEnumSignal<NDrive::TLocation::EType> {
    public:
        TSignalLocationType()
            : TEnumSignal<NDrive::TLocation::EType>({ "location-type" }, false)
        {
        }
    };

    class TSignalLocationsAge: public TUnistatSignal<double> {
    public:
        TSignalLocationsAge()
            : TUnistatSignal<double>({ "location-ages" }, NRTLineHistogramSignals::IntervalsTelematicSignals)
        {
        }
    };

    class TSignalGPSLocationsAge: public TUnistatSignal<double> {
    public:
        TSignalGPSLocationsAge()
            : TUnistatSignal<double>({ "location-ages-GPS" }, NRTLineHistogramSignals::IntervalsTelematicSignals)
        {
        }
    };

    class TSignalBeaconLocationsAge: public TUnistatSignal<double> {
    public:
        TSignalBeaconLocationsAge()
            : TUnistatSignal<double>({ "location-ages-Beacon" }, NRTLineHistogramSignals::IntervalsTelematicSignals)
        {
        }
    };

    class TSignalExternalLocationsAge: public TUnistatSignal<double> {
    public:
        TSignalExternalLocationsAge()
            : TUnistatSignal<double>({ "location-ages-External" }, NRTLineHistogramSignals::IntervalsTelematicSignals)
        {
        }
    };

    class TSignalLBSLocationsAge: public TUnistatSignal<double> {
    public:
        TSignalLBSLocationsAge()
            : TUnistatSignal<double>({ "location-ages-LBS" }, NRTLineHistogramSignals::IntervalsTelematicSignals)
        {
        }
    };

    class TSignalLinkedLocationsAge: public TUnistatSignal<double> {
    public:
        TSignalLinkedLocationsAge()
            : TUnistatSignal<double>({ "location-ages-Linked" }, NRTLineHistogramSignals::IntervalsTelematicSignals)
        {
        }
    };

    class TSignalProjectionLocationsAge: public TUnistatSignal<double> {
    public:
        TSignalProjectionLocationsAge()
            : TUnistatSignal<double>({ "location-ages-Projection" }, NRTLineHistogramSignals::IntervalsTelematicSignals)
        {
        }
    };

    static TSignalLocationsAge SignalLocationsAge;
    static TSignalLocationType SignalLocationsType;

    template <class TDocument, class TDocumentDataAccessor>
    std::tuple<NDrive::TSensor, TString> BuildSensorFromDocument(const TDocument& document) {
        TString imei;
        TString rawValue;
        NDrive::TSensor sensor;
        for (ui32 i = 0; i < TDocumentDataAccessor::GetAttributeSize(document); ++i) {
            const auto& propertie = TDocumentDataAccessor::GetAttribute(document, i);
            const auto& key = TDocumentDataAccessor::GetKey(propertie);
            const auto& value = TDocumentDataAccessor::GetValue(propertie);
            if (sensor.Id == 0 && key == "Id") {
                sensor.Id = FromString<ui16>(value);
                continue;
            }
            if (sensor.Id == 0 && key == "_SK_Sensor") {
                ui16 id = 0;
                if (TryFromString(value, id)) {
                    sensor.Id = id;
                }
                continue;
            }
            if (sensor.SubId == 0 && key == "SubId") {
                sensor.SubId = FromString<ui16>(value);
                continue;
            }
            if (sensor.Since == TInstant::Zero() && key == "Since") {
                sensor.Since = TInstant::Seconds(FromString<ui64>(value));
                continue;
            }
            if (sensor.Timestamp == TInstant::Zero() && key == "Timestamp") {
                sensor.Timestamp = TInstant::Seconds(FromString<ui64>(value));
            }
            if (rawValue.empty() && key == "Value") {
                rawValue = value;
                continue;
            }
            if (imei.empty() && (key == "IMEI" || key == "_SK_imei")) {
                imei = value;
                continue;
            }
        }
        if (sensor) {
            sensor.Value = NDrive::SensorValueFromString(rawValue, sensor);
        }

        return { sensor, imei };
    }

    TMap<NDrive::TLocation::EType, TAtomicSharedPtr<TUnistatSignal<double>>> LocationTypeToAgeSignal = {
        {NDrive::TLocation::EType::GPSCurrent,          MakeAtomicShared<TSignalGPSLocationsAge>()},
        {NDrive::TLocation::EType::GPSPrevious,         MakeAtomicShared<TSignalGPSLocationsAge>()},
        {NDrive::TLocation::EType::LBS,                 MakeAtomicShared<TSignalLBSLocationsAge>()},
        {NDrive::TLocation::EType::Linked,              MakeAtomicShared<TSignalLinkedLocationsAge>()},
        {NDrive::TLocation::EType::LinkedPrevious,      MakeAtomicShared<TSignalLinkedLocationsAge>()},
        {NDrive::TLocation::EType::External,            MakeAtomicShared<TSignalExternalLocationsAge>()},
        {NDrive::TLocation::EType::Beacon,              MakeAtomicShared<TSignalBeaconLocationsAge>()},
        {NDrive::TLocation::EType::Projection,          MakeAtomicShared<TSignalProjectionLocationsAge>()},
        {NDrive::TLocation::EType::ProjectionPrevious,  MakeAtomicShared<TSignalProjectionLocationsAge>()}
    };
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> NDrive::TSensorApi::DoGetLocations(NDrive::TSensorApi::TIMEIs imeis, TLocationsQueryOptions queryOptions) const {
    if (imeis.empty()) {
        return NThreading::MakeFuture(TLocations());
    }

    auto name = queryOptions.GetName();
    TVector<TString> keys;
    for (auto&& imei : imeis) {
        keys.push_back(GetLocationUrl(imei, name));
    }
    auto query = NRTLine::TQuery();
    if (queryOptions.GetFetchOptions().HasTimeout()) {
        query.SetTimeout(queryOptions.GetFetchOptions().GetTimeoutRef());
    }
    auto replies = Query(std::move(keys), std::move(query), queryOptions.GetFetchOptions().OptionalMaxKeysPerQuery());
    auto waiter = NThreading::WaitAll(replies);
    return waiter.Apply([
        this,
        name = std::move(name),
        queryOptions = std::move(queryOptions),
        replies = std::move(replies)
    ](const NThreading::TFuture<void>& w) {
        ProcessFuture(w);
        ProcessQuorum(replies, queryOptions.GetFetchOptions());
        TLocations locations;
        for (auto&& reply : replies) {
            if (!reply.HasValue()) {
                continue;
            }
            FillLocations(reply.GetValue(), locations, name);
        }
        return locations;
    });
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> NDrive::TSensorApi::DoGetLocations(TLocationsQueryOptions queryOptions) const {
    auto query = NRTLine::TQuery();
    query.AddProperty("BaseLatitude");
    query.AddProperty("BaseLongitude");
    query.AddProperty("BaseTimestamp");
    query.AddProperty("Content");
    query.AddProperty("Course");
    query.AddProperty("Latitude");
    query.AddProperty("Longitude");
    query.AddProperty("Name");
    query.AddProperty("Precision");
    query.AddProperty("Since");
    query.AddProperty("Timestamp");
    query.AddProperty("Type");
    if (queryOptions.GetFetchOptions().HasTimeout()) {
        query.SetTimeout(queryOptions.GetFetchOptions().GetTimeoutRef());
    }
    auto name = queryOptions.GetName();
    auto reply = HeavyQuery(GetLocationKey(name), "Sensor", std::move(query));
    return reply.Apply([this, name = std::move(name)](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        ProcessFuture(r);
        TLocations locations;
        FillLocations(r.GetValue(), locations, name);
        return locations;
    });
}

void NDrive::TSensorApi::FillLocations(const NRTLine::TSearchReply& reply, TLocations& locations, TMaybe<TString> expected) const {
    const TInstant nowSeconds = Now();
    auto parser = [&](const NMetaProtocol::TDocument& document) {
        try {
        TString imei;
        NDrive::TLocation location;
        for (auto&& propertie : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
            const auto& key = propertie.GetKey();
            const auto& value = propertie.GetValue();
            if (location.Type == TLocation::Unknown && key == "Type") {
                location.Type = FromString<NDrive::TLocation::EType>(value);
                SignalLocationsType.Signal(location.Type, 1);
                continue;
            }
            if (location.Timestamp == TInstant::Zero() && key == "Timestamp") {
                location.Timestamp = TInstant::Seconds(FromString<ui64>(value));
                if (location.Timestamp < nowSeconds) {
                    SignalLocationsAge.Signal((nowSeconds - location.Timestamp).MilliSeconds());
                }
                continue;
            }
            if (location.Since == TInstant::Zero() && key == "Since") {
                location.Since = TInstant::Seconds(FromString<ui64>(value));
            }
            if (key == "Latitude") {
                location.Latitude = FromString<double>(value);
                continue;
            }
            if (key == "Longitude") {
                location.Longitude = FromString<double>(value);
                continue;
            }
            if (key == "Precision") {
                location.Precision = FromString<double>(value);
                continue;
            }
            if (key == "Course") {
                location.Course = FromString<double>(value);
                continue;
            }
            if (imei.empty() && (key == "IMEI" || key == "_SK_imei")) {
                imei = value;
                continue;
            }
            if (key == "BaseLatitude") {
                location.MutableBase().Latitude = FromString<double>(value);
                continue;
            }
            if (key == "BaseLongitude") {
                location.MutableBase().Longitude = FromString<double>(value);
                continue;
            }
            if ((!location.Base || location.Base->Timestamp == TInstant::Zero()) && key == "BaseTimestamp") {
                location.MutableBase().Timestamp = TInstant::Seconds(FromString<ui64>(value));
                continue;
            }
            if (location.Content.empty() && key == "Content") {
                location.Content = value;
                continue;
            }
            if (location.Name.empty() && key == "Name") {
                location.Name = value;
                continue;
            }
        }
        if (location.Timestamp < nowSeconds) {
            auto it = LocationTypeToAgeSignal.find(location.Type);
            if (it != LocationTypeToAgeSignal.end()) {
                Yensured(it->second)->Signal((nowSeconds - location.Timestamp).MilliSeconds());
            }
        }
        if (expected && !(*expected == location.Name)) {
            DEBUG_LOG << "bad name: " << document.GetUrl() << Endl;
            return;
        }
        if (location.Type == TLocation::Unknown) {
            DEBUG_LOG << "bad type: " << document.GetUrl() << Endl;
            return;
        }
        if (imei) {
            locations.emplace(imei, location);
        }
        } catch (const std::exception& e) {
            ERROR_LOG << "cannot parse location " << document.DebugString() << ": " << FormatExc(e) << Endl;
        }
    };
    reply.CheckSucceeded("location fetch has been unsuccessful");
    reply.ScanDocs(parser);
}

namespace {
    class TSignalSensorsAge: public TUnistatSignal<double> {
    public:
        TSignalSensorsAge()
            : TUnistatSignal<double>({ "sensor-ages" }, NRTLineHistogramSignals::IntervalsTelematicSignals)
        {
        }
    };

    static TSignalSensorsAge SignalSensorsAge;
}

NThreading::TFuture<NDrive::ISensorApi::TSensors> NDrive::TSensorApi::DoGetSensors(TIMEIs imeis, TSensorIds ids, const TSensorsQueryOptions& queryOptions) const {
    if (imeis.empty()) {
        return NThreading::MakeFuture(TSensors());
    }

    TVector<TString> keys;
    for (auto&& imei : imeis) {
        for (auto&& id : ids) {
            keys.push_back(GetSensorUrl(imei, id));
        }
    }
    auto query = NRTLine::TQuery();
    if (queryOptions.GetFetchOptions().HasTimeout()) {
        query.SetTimeout(queryOptions.GetFetchOptions().GetTimeoutRef());
    }
    auto replies = Query(std::move(keys), std::move(query), queryOptions.GetFetchOptions().OptionalMaxKeysPerQuery());
    auto waiter = NThreading::WaitAll(replies);
    return waiter.Apply([
        this,
        queryOptions = std::move(queryOptions),
        replies = std::move(replies)
    ](const NThreading::TFuture<void>& w) {
        ProcessFuture(w);
        ProcessQuorum(replies, queryOptions.GetFetchOptions());
        TSensors sensors;
        for (auto&& reply : replies) {
            if (!reply.HasValue()) {
                continue;
            }
            FillSensors(reply.GetValue(), sensors);
        }
        for (auto&&[imei, s] : sensors) {
            std::sort(s.begin(), s.end());
        }
        return sensors;
    });
}

NThreading::TFuture<NDrive::ISensorApi::TSensors> NDrive::TSensorApi::DoGetSensor(TSensorId sensor, const TSensorsQueryOptions& queryOptions) const {
    auto query = NRTLine::TQuery();
    query.AddProperty("SubId");
    query.AddProperty("Since");
    query.AddProperty("Timestamp");
    query.AddProperty("Value");
    if (queryOptions.GetFetchOptions().HasTimeout()) {
        query.SetTimeout(queryOptions.GetFetchOptions().GetTimeoutRef());
    }
    auto reply = HeavyQuery(ToString(sensor.Id), "Sensor", std::move(query));
    return reply.Apply([this, sensor](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        ProcessFuture(r);
        TSensors sensors;
        FillSensors(r.GetValue(), sensors, sensor);
        return sensors;
    });
}

NThreading::TFuture<NDrive::ISensorApi::TMultiSensor> NDrive::TSensorApi::DoGetSensors(const TString& imei, const TSensorsQueryOptions& queryOptions) const {
    auto query = NRTLine::TQuery();
    query.AddProperty("SubId");
    query.AddProperty("Since");
    query.AddProperty("Timestamp");
    query.AddProperty("Value");
    if (queryOptions.GetFetchOptions().HasTimeout()) {
        query.SetTimeout(queryOptions.GetFetchOptions().GetTimeoutRef());
    }
    auto reply = HeavyQuery(imei, "imei", std::move(query));
    return reply.Apply([this, imei](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
        ProcessFuture(r);
        TSensors sensors;
        FillSensors(r.GetValue(), sensors);
        auto result = std::move(sensors[imei]);
        std::sort(result.begin(), result.end());
        return result;
    });
}

void NDrive::TSensorApi::FillSensors(const NRTLine::TSearchReply& reply, TSensors& sensors, TMaybe<TSensorId> expected) const {
    const TInstant nowSeconds = Now();
    auto parser = [&sensors, &nowSeconds, expected] (const NMetaProtocol::TDocument& document) {
        TString imei;
        TSensor sensor;
        try {
            std::tie(sensor, imei) = BuildSensorFromDocument<NMetaProtocol::TDocument, TSearchDocumentDataAccessor>(document);
        } catch (const std::exception& e) {
            ERROR_LOG << "cannot parse sensor " << document.DebugString() << ": " << FormatExc(e) << Endl;
            return;
        }
        if (!sensor) {
            return;
        }
        if (expected && !(*expected == sensor)) {
            return;
        }

        if (sensor.Timestamp < nowSeconds) {
            SignalSensorsAge.Signal((nowSeconds - sensor.Timestamp).MilliSeconds());
        }
        sensors[imei].push_back(std::move(sensor));
    };

    reply.CheckSucceeded("sensor fetch has been unsuccessful");
    reply.ScanDocs(parser);
    for (auto&& s : sensors) {
        std::sort(s.second.begin(), s.second.end());
    }
}

template <typename TFuture>
void NDrive::TSensorApi::ProcessFuture(const TFuture& f) const {
    if (f.HasValue()) {
        TUnistatSignalsCache::SignalAdd("sensor_api_" + Name, "success", 1);
    } else {
        TUnistatSignalsCache::SignalAdd("sensor_api_" + Name, "exception", 1);
    }
}

template <typename TFuture>
void NDrive::TSensorApi::ProcessQuorum(const TVector<TFuture>& futures, const TFetchOptions& fetchOptions) const {
    if (fetchOptions.HasQuorum()) {
        size_t count = futures.size();
        size_t success = 0;
        for (auto&& future : futures) {
            if (future.HasValue()) {
                success += 1;
            }
        }
        if (success == count) {
            return;
        }
        auto threshold = fetchOptions.GetQuorumRef() * count;
        if (success >= threshold) {
            TUnistatSignalsCache::SignalAdd("sensor_api_" + Name, "partial_quorum", 1);
            return;
        }
    }
    for (auto&& future : futures) {
        future.GetValue();
    }
}
