#include "history.h"

#include <drive/telematics/server/sensors/cache.h>

#include <rtline/api/search_client/neh_client.h>
#include <rtline/util/algorithm/container.h>

namespace NDrive {

    NThreading::TFuture<NDrive::TSensorHistoryApi::TCarsSensorsHistory> TSensorHistoryApi::GetSensorHistory(TSensorId sensorId) const {
        NRTLine::TQuery query;
        query.SetText(::ToString(sensorId.Id)).AddExtraParam("key_name", "Id");
        auto replyGuard = Client.SendAsyncQueryF(query, Timeout);

        return replyGuard.Apply([this](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            const NRTLine::TSearchReply& reply = r.GetValue();
            TCarsSensorsHistory result;
            auto parser = [&result, this](const NMetaProtocol::TDocument& document) {
                try {
                    auto[imei, sensorId, history] = ParseHistoryItem(document);
                    result[imei].emplace(sensorId, std::move(history));
                } catch (const std::exception& e) {
                    ERROR_LOG << "cannot parse sensor history" << document.DebugString() << "/" << FormatExc(e) << Endl;
                }
                return true;
            };

            reply.CheckSucceeded();
            reply.ScanDocs(parser);
            return result;
        });
    }

    NThreading::TFuture<NDrive::TSensorHistoryApi::TSensorHistory> TSensorHistoryApi::GetSensorsHistory(const TString& imei) const {
        NRTLine::TQuery query;
        query.SetText(imei).AddExtraParam("key_name", "IMEI");
        auto replyGuard = Client.SendAsyncQueryF(query, Timeout);

        return replyGuard.Apply([this](const NThreading::TFuture<NRTLine::TSearchReply>& r) {
            const NRTLine::TSearchReply& reply = r.GetValue();
            TSensorHistory result;
            auto parser = [&result, this](const NMetaProtocol::TDocument& document) {
                try {
                    auto [imei, sensorId, history] = ParseHistoryItem(document);
                    Y_UNUSED(imei);
                    result.emplace(sensorId, std::move(history));
                } catch (const std::exception& e) {
                    ERROR_LOG << "cannot parse sensor history" << document.DebugString() << "/" << FormatExc(e) << Endl;
                }
                return true;
            };

            reply.CheckSucceeded();
            reply.ScanDocs(parser);
            return result;
        });
    }

    std::tuple<TString, NDrive::TSensorId, TSensorHistoryApi::THistory> TSensorHistoryApi::ParseHistoryItem(const NMetaProtocol::TDocument& document) const {
        TString imei;
        ui16 sId(0), subId(0);
        std::tie(imei, sId, subId) = NDrive::TSensorHistoryApi::ParseHistoryDocId(document.GetUrl());

        TString timelineStr;
        TInstant lastHistoryUpdate;

        for (auto&& propertie : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
            const auto& key = propertie.GetKey();
            const auto& value = propertie.GetValue();
            if (key == "LastUpdateTs") {
                lastHistoryUpdate = TInstant::Seconds(FromString<ui64>(value));
                continue;
            }
            if (key == "Timeline") {
                timelineStr = value;
                continue;
            }
        }

        TVector<TString> events;
        StringSplitter(timelineStr).SplitBySet(" :").SkipEmpty().Collect(&events);
        Y_ENSURE(events.size() % 2 == 0, "Invalid timeline format " << timelineStr);

        TVector<TSensorHistoryApi::TTimelineEvent> timeline;
        timeline.reserve(events.size() / 2);

        for (ui32 i = 0; i < events.size(); i += 2) {
            timeline.emplace_back(TInstant::Seconds(FromString<ui32>(events[i])) , NDrive::SensorValueFromString(events[i + 1], sId));
        }

        return { imei, TSensorId(sId, subId), TSensorHistoryApi::THistory(std::move(timeline), lastHistoryUpdate) };
    }
}

NThreading::TFuture<NDrive::TSensorHistoryClient::TResult> NDrive::TSensorHistoryClient::Get(
    TConstArrayRef<TString> objectIds,
    TConstArrayRef<TString> imeis,
    TInstant since,
    TInstant until,
    TConstArrayRef<TSensorId> sensorIds
) const {
    auto timestamp = MakeRange<TInstant>(
        since,
        until != TInstant::Max() ? MakeMaybe(until) : Nothing()
    );
    auto sensors = Client.Get(NDrive::TClickHouseClient::TSensorHistoryQuery()
        .SetObjectIds(MakeVector(objectIds))
        .SetImeis(MakeVector(imeis))
        .SetSensors(MakeVector(sensorIds))
        .SetTimestamp(timestamp)
    );
    return sensors.Apply([sensors](const NThreading::TFuture<NDrive::TClickHouseClient::TSensorHistoryResult>& /*s*/) mutable {
        TResult result;
        auto s = sensors.ExtractValue();
        for (auto&& [id, ss] : s) {
            std::sort(ss.begin(), ss.end(), [](const TSensor& left, const TSensor& right) {
                return left.Timestamp < right.Timestamp;
            });
            auto cache = MakeAtomicShared<NDrive::TSensorsCache>(std::numeric_limits<ui32>::max());
            cache->Add(std::move(ss));
            result.emplace(id, std::move(cache));
        }
        return result;
    });
}
