#include "interface.h"

#include <drive/telematics/server/sensors/cache.h>

#include <drive/library/cpp/threading/future.h>

#include <library/cpp/logger/global/global.h>

#include <rtline/util/algorithm/container.h>


NThreading::TFuture<NDrive::ISensorApi::THeartbeats> NDrive::ISensorApi::GetHeartbeats(TIMEIs imeis, TStringBuf name) const {
    THeartbeatsQueryOptions queryOptions;
    queryOptions.SetName(name);
    return GetHeartbeats(imeis, queryOptions);
}

NThreading::TFuture<NDrive::ISensorApi::THeartbeats> NDrive::ISensorApi::GetHeartbeats(TStringBuf name) const {
    THeartbeatsQueryOptions queryOptions;
    queryOptions.SetName(name);
    return GetHeartbeats(queryOptions);
}

NThreading::TFuture<TMaybe<NDrive::THeartbeat>> NDrive::ISensorApi::GetHeartbeat(const TString& imei, TStringBuf name) const {
    auto heartbeats = GetHeartbeats(NContainer::Scalar(imei), name);
    return heartbeats.Apply([imei](const NThreading::TFuture<THeartbeats>& h) -> TMaybe<NDrive::THeartbeat> {
        const auto& heartbeats = h.GetValue();
        if (!heartbeats.empty()) {
            Y_ENSURE(heartbeats.size() == 1);
            Y_ENSURE(heartbeats.begin()->first == imei);
            return heartbeats.begin()->second;
        } else {
            return {};
        }
    });
}

NThreading::TFuture<NDrive::ISensorApi::THeartbeats> NDrive::ISensorApi::GetHeartbeats(TIMEIs imeis, THeartbeatsQueryOptions queryOptions) const {
    return DoGetHeartbeats(imeis, queryOptions);
}

NThreading::TFuture<NDrive::ISensorApi::THeartbeats> NDrive::ISensorApi::GetHeartbeats(THeartbeatsQueryOptions queryOptions) const {
    return DoGetHeartbeats(queryOptions);
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> NDrive::ISensorApi::GetLocations(TIMEIs imeis, TStringBuf name) const {
    TLocationsQueryOptions options;
    options.SetName(name);
    return GetLocations(imeis, options);
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> NDrive::ISensorApi::GetLocations(TStringBuf name) const {
    TLocationsQueryOptions options;
    options.SetName(name);
    return GetLocations(options);
}

NThreading::TFuture<TMaybe<NDrive::TLocation>> NDrive::ISensorApi::GetLocation(const TString& imei, TStringBuf name) const {
    auto locations = GetLocations(NContainer::Scalar(imei), name);
    return locations.Apply([imei](const NThreading::TFuture<TLocations>& l) -> TMaybe<NDrive::TLocation> {
        const auto& locations = l.GetValue();
        if (!locations.empty()) {
            Y_ENSURE(locations.size() == 1);
            Y_ENSURE(locations.begin()->first == imei);
            return locations.begin()->second;
        } else {
            return {};
        }
    });
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> NDrive::ISensorApi::GetLocations(TIMEIs imeis, TLocationsQueryOptions queryOptions) const {
    return DoGetLocations(imeis, queryOptions);
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> NDrive::ISensorApi::GetLocations(TLocationsQueryOptions queryOptions) const {
    return DoGetLocations(queryOptions);
}

NThreading::TFuture<TMaybe<NDrive::TSensor>> NDrive::ISensorApi::GetSensor(const TString& imei, TSensorId id, const TSensorsQueryOptions& queryOptions, ui32 replication, TMaybe<ui32> minSuccessfulReadsCountExt) const {
    auto sensors = GetSensors(imei, NContainer::Scalar(id), queryOptions, replication, minSuccessfulReadsCountExt);
    return sensors.Apply([](const NThreading::TFuture<TMultiSensor>& s) -> TMaybe<NDrive::TSensor> {
        const auto& sensors = s.GetValue();
        if (!sensors.empty()) {
            Y_ENSURE(sensors.size() == 1);
            return sensors[0];
        } else {
            return {};
        }
    });
}

NThreading::TFuture<NDrive::ISensorApi::TMultiSensor> NDrive::ISensorApi::GetSensors(const TString& imei, TSensorIds ids, const TSensorsQueryOptions& queryOptions, ui32 replication, TMaybe<ui32> minSuccessfulReadsCountExt) const {
    auto sensors = GetSensors(NContainer::Scalar(imei), ids, queryOptions, replication, minSuccessfulReadsCountExt);
    return sensors.Apply([imei](const NThreading::TFuture<TSensors>& s) -> TMultiSensor {
        const auto& sensors = s.GetValue();
        if (!sensors.empty()) {
            Y_ENSURE(sensors.size() == 1);
            Y_ENSURE(sensors.begin()->first == imei);
            return std::move(sensors.begin()->second);
        } else {
            return {};
        }
    });
}

NThreading::TFuture<NDrive::ISensorApi::TSensors> NDrive::ISensorApi::GetSensor(TIMEIs imeis, TSensorId id, const TSensorsQueryOptions& queryOptions, ui32 replication, TMaybe<ui32> minSuccessfulReadsCountExt) const {
    return GetSensors(imeis, NContainer::Scalar(id), queryOptions, replication, minSuccessfulReadsCountExt);
}

NThreading::TFuture<NDrive::ISensorApi::TSensors> NDrive::ISensorApi::GetSensors(TIMEIs imeis, TSensorIds ids, const TSensorsQueryOptions& queryOptions, ui32 replication, TMaybe<ui32> minSuccessfulReadsCountExt) const {
    if (replication > 1) {
        TVector<NThreading::TFuture<TSensors>> sensors;
        for (size_t i = 0; i < replication; ++i) {
            sensors.push_back(DoGetSensors(imeis, ids, queryOptions));
        }
        return Merge(std::move(sensors), minSuccessfulReadsCountExt);
    } else {
        return DoGetSensors(imeis, ids, queryOptions);
    }
}

NThreading::TFuture<NDrive::ISensorApi::TMultiSensor> NDrive::ISensorApi::GetSensors(const TString& imei, const TSensorsQueryOptions& queryOptions, ui32 replication, TMaybe<ui32> minSuccessfulReadsCountExt) const {
    if (replication > 1) {
        TVector<NThreading::TFuture<TMultiSensor>> sensors;
        for (size_t i = 0; i < replication; ++i) {
            sensors.push_back(DoGetSensors(imei, queryOptions));
        }
        return Merge(std::move(sensors), minSuccessfulReadsCountExt);
    } else {
        return DoGetSensors(imei, queryOptions);
    }
}


NThreading::TFuture<NDrive::ISensorApi::TSensors> NDrive::ISensorApi::GetSensor(TSensorId id, const TSensorsQueryOptions& queryOptions, ui32 replication, TMaybe<ui32> minSuccessfulReadsCountExt) const {
    if (replication > 1) {
        TVector<NThreading::TFuture<TSensors>> sensors;
        for (size_t i = 0; i < replication; ++i) {
            sensors.push_back(DoGetSensor(id, queryOptions));
        }
        return Merge(std::move(sensors), minSuccessfulReadsCountExt);
    } else {
        return DoGetSensor(id, queryOptions);
    }
}


TMaybe<NDrive::TSensor> NDrive::ISensorApi::GetSensor(const NDrive::ISensorApi::TSensors& sensors, const TString& imei, TSensorId id) {
    auto p = sensors.find(imei);
    if (p != sensors.end()) {
        return FindSensor(p->second, id);
    } else {
        return {};
    }
}

namespace {
    template <class T>
    NThreading::TFuture<THolder<T>> Extractable(NThreading::TFuture<T>&& value) {
        return value.Apply([value](const NThreading::TFuture<T>& /*v*/) mutable {
            return MakeHolder<T>(value.ExtractValue());
        });
    }

    template <class T>
    NThreading::TFutures<THolder<T>> Extractable(NThreading::TFutures<T>&& values) {
        NThreading::TFutures<THolder<T>> result;
        for (auto&& value : values) {
            result.push_back(Extractable(std::move(value)));
        }
        return result;
    }

    template <class T>
    NThreading::TFuture<T> Merge(TVector<NThreading::TFuture<T>>&& values, TMaybe<ui32> minSuccessfulReadsCountExt) {
        auto extractables = Extractable(std::move(values));
        auto waiter = NThreading::WaitAll(extractables);
        return waiter.Apply([extractables = std::move(extractables), minSuccessfulReadsCountExt](const NThreading::TFuture<void>& /* waiter */) mutable {
            ui32 successfulReadsCount = 0;
            T result;
            for (auto&& i : extractables) {
                if (i.HasValue()) {
                    ++successfulReadsCount;
                    result = NDrive::ISensorApi::Merge(std::move(result), std::move(*i.GetValue()));
                }
            }
            ui32 minSuccessfulReadsCount = minSuccessfulReadsCountExt.GetOrElse(extractables.size() / 2 + 1);
            if (successfulReadsCount < minSuccessfulReadsCount) {
                NJson::TJsonValue replicasErrors;
                for (auto&& i : extractables) {
                    if (i.HasException()) {
                        replicasErrors.AppendValue(NThreading::GetExceptionMessage(i));
                    }
                }
                ythrow yexception() << "can't read from necessary number of replicas; "
                                    << successfulReadsCount << " / " << minSuccessfulReadsCount << " "
                                    << "errors: " << replicasErrors.GetStringRobust();
            }
            return result;
        });
    }
}

NThreading::TFuture<NDrive::ISensorApi::TMultiSensor> NDrive::ISensorApi::Merge(TVector<NThreading::TFuture<TMultiSensor>>&& sensors, TMaybe<ui32> minSuccessfulReadsCountExt) {
    return ::Merge(std::move(sensors), minSuccessfulReadsCountExt);
}

NThreading::TFuture<NDrive::ISensorApi::TSensors> NDrive::ISensorApi::Merge(TVector<NThreading::TFuture<TSensors>>&& sensors, TMaybe<ui32> minSuccessfulReadsCountExt) {
    return ::Merge(std::move(sensors), minSuccessfulReadsCountExt);
}

NDrive::ISensorApi::TMultiSensor NDrive::ISensorApi::Merge(TMultiSensor&& base, TMultiSensor&& delta) {
    if (base.size() < delta.size()) {
        std::swap(base, delta);
    }

    base.insert(base.end(), delta.begin(), delta.end());
    std::sort(base.begin(), base.end(), [](const TSensor& left, const TSensor& right) {
        return std::tie(left.Id, left.SubId, left.Timestamp) > std::tie(right.Id, right.SubId, right.Timestamp);
    });
    base.erase(std::unique(base.begin(), base.end()), base.end());
    std::reverse(base.begin(), base.end());
    return base;
}

NDrive::ISensorApi::TSensors NDrive::ISensorApi::Merge(TSensors&& base, TSensors&& delta) {
    if (base.size() < delta.size()) {
        std::swap(base, delta);
    }
    auto i = base.begin();
    auto j = delta.begin();
    while (i != base.end() && j != delta.end()) {
        if (i->first < j->first) {
            i++;
            continue;
        }
        if (j->first < i->first) {
            base.insert(std::move(*j));
            j++;
            continue;
        }
        i->second = Merge(std::move(i->second), std::move(j->second));
        i++;
        j++;
    }
    for (; j != delta.end(); ++j) {
        base.insert(std::move(*j));
    }
    return base;
}


const TVector<NDrive::ISensorHistoryApi::TTimelineEvent>& NDrive::ISensorHistoryApi::THistory::GetTimeline() const {
    return Timeline;
}


TInstant NDrive::ISensorHistoryApi::THistory::GetLastUpdate() const {
    return LastUpdate;
}

NThreading::TFuture<NDrive::TSensorsCachePtr> NDrive::ISensorHistoryClient::Get(const TString& objectId, const TString& imei, TInstant since, TInstant until, TConstArrayRef<TSensorId> sensorIds) const {
    auto objectIds = objectId ? NContainer::Scalar(objectId) : TConstArrayRef<TString>();
    auto imeis = imei ? NContainer::Scalar(imei) : TConstArrayRef<TString>();
    return Get(objectIds, imeis, since, until, sensorIds).Apply([objectId, imei](const NThreading::TFuture<TResult>& r) -> NDrive::TSensorsCachePtr {
        const auto& result = r.GetValue();
        if (objectId) {
            auto p = result.find(objectId);
            if (p != result.end()) {
                return p->second;
            }
        }
        if (imei) {
            auto p = result.find(objectId);
            if (p != result.end()) {
                return p->second;
            }
        }
        return nullptr;
    });
}

template<>
void Out<NDrive::ISensorHistoryApi::TTimelineEvent>(IOutputStream& output, const NDrive::ISensorHistoryApi::TTimelineEvent& data) {
    output << data.first.Seconds() << ":" << NDrive::TSensorValueOperator<NDrive::TSensorValue>::ConvertTo(data.second, TString());
}

