#pragma once

#include <drive/telematics/protocol/sensor.h>
#include <drive/telematics/server/location/heartbeat.h>
#include <drive/telematics/server/location/location.h>
#include <drive/telematics/server/pusher/interface.h>

#include <rtline/util/types/accessor.h>

#include <library/cpp/threading/future/fwd.h>

#include <util/generic/map.h>
#include <util/string/vector.h>


namespace NDrive {
    class TSensorsCache;
    using TSensorsCachePtr = TAtomicSharedPtr<TSensorsCache>;

    class ISensorApi {
    public:
        using TSensorIds = TConstArrayRef<TSensorId>;

        using TIMEIs = TConstArrayRef<TString>;

        using THeartbeats = TMap<TString, THeartbeat>;
        using TLocations = TMap<TString, TLocation>;
        using TMultiSensor = NDrive::TMultiSensor;
        using TSensors = TMap<TString, TMultiSensor>;

        class TFetchOptions {
        public:
            R_OPTIONAL(TDuration, Timeout);
            R_OPTIONAL(ui32, MaxKeysPerQuery);
            R_OPTIONAL(float, Quorum);
        };

        class TLocationsQueryOptions {
        public:
            R_FIELD(TString, Name);
            R_FIELD(TFetchOptions, FetchOptions);
        };

        using THeartbeatsQueryOptions = TLocationsQueryOptions;

        class TSensorsQueryOptions {
        public:
            R_FIELD(TFetchOptions, FetchOptions);
        };

    public:
        virtual ~ISensorApi() = default;

        virtual TPusherPtr GetPusher() const {
            return nullptr;
        }

        NThreading::TFuture<THeartbeats> GetHeartbeats(TIMEIs imeis, TStringBuf name = {}) const;
        NThreading::TFuture<THeartbeats> GetHeartbeats(TStringBuf name = {}) const;
        NThreading::TFuture<TMaybe<THeartbeat>> GetHeartbeat(const TString& imei, TStringBuf name = {}) const;
        NThreading::TFuture<THeartbeats> GetHeartbeats(TIMEIs imeis, THeartbeatsQueryOptions queryOptions) const;
        NThreading::TFuture<THeartbeats> GetHeartbeats(THeartbeatsQueryOptions queryOptions) const;

        NThreading::TFuture<TLocations> GetLocations(TIMEIs imeis, TStringBuf name = {}) const;
        NThreading::TFuture<TLocations> GetLocations(TStringBuf name = {}) const;
        NThreading::TFuture<TMaybe<TLocation>> GetLocation(const TString& imei, TStringBuf name = {}) const;
        NThreading::TFuture<TLocations> GetLocations(TIMEIs imeis, TLocationsQueryOptions queryOptions) const;
        NThreading::TFuture<TLocations> GetLocations(TLocationsQueryOptions queryOptions) const;

        // helpers
        NThreading::TFuture<TMaybe<TSensor>> GetSensor(const TString& imei, TSensorId id, const TSensorsQueryOptions& queryOptions = {}, ui32 replication = 1, TMaybe<ui32> minSuccessfulReadsCountExt = {}) const;
        NThreading::TFuture<TMultiSensor> GetSensors(const TString& imei, TSensorIds ids, const TSensorsQueryOptions& queryOptions = {}, ui32 replication = 1, TMaybe<ui32> minSuccessfulReadsCountExt = {}) const;
        NThreading::TFuture<TSensors> GetSensor(TIMEIs imeis, TSensorId id, const TSensorsQueryOptions& queryOptions = {}, ui32 replication = 1, TMaybe<ui32> minSuccessfulReadsCountExt = {}) const;

        // main functions
        NThreading::TFuture<TSensors> GetSensors(TIMEIs imeis, TSensorIds ids, const TSensorsQueryOptions& queryOptions = {}, ui32 replication = 1, TMaybe<ui32> minSuccessfulReadsCountExt = {}) const;
        NThreading::TFuture<TMultiSensor> GetSensors(const TString& imei, const TSensorsQueryOptions& queryOptions = {}, ui32 replication = 1, TMaybe<ui32> minSuccessfulReadsCountExt = {}) const;
        NThreading::TFuture<TSensors> GetSensor(TSensorId id, const TSensorsQueryOptions& queryOptions = {}, ui32 replication = 1, TMaybe<ui32> minSuccessfulReadsCountExt = {}) const;

    public:
        template <class T>
        static TMaybe<T> FindSensor(const TVector<T>& sensors, const TSensorId id) {
            Y_ASSERT(std::is_sorted(sensors.begin(), sensors.end()));
            auto p = std::lower_bound(sensors.begin(), sensors.end(), id);
            if (p != sensors.end() && TSensorId(p->Id, p->SubId) == id) {
                return *p;
            } else {
                return {};
            }
        }

        static TMaybe<TSensor> GetSensor(const TSensors& sensors, const TString& imei, TSensorId id);

        template <class T>
        static TMaybe<T> GetSensorFamily(const TVector<T>& sensors, const ui16 id) {
            Y_ASSERT(std::is_sorted(sensors.begin(), sensors.end()));
            auto p = std::lower_bound(sensors.begin(), sensors.end(), id);
            if (p != sensors.end() && p->Id == id) {
                return *p;
            } else {
                return {};
            }
        }

        static NThreading::TFuture<TMultiSensor> Merge(TVector<NThreading::TFuture<TMultiSensor>>&& sensors, TMaybe<ui32> minSuccessfulReadsCountExt = {});
        static NThreading::TFuture<TSensors> Merge(TVector<NThreading::TFuture<TSensors>>&& sensors, TMaybe<ui32> minSuccessfulReadsCountExt = {});
        static TMultiSensor Merge(TMultiSensor&& base, TMultiSensor&& delta);
        static TSensors Merge(TSensors&& base, TSensors&& delta);

    private:
        virtual NThreading::TFuture<TLocations> DoGetLocations(TIMEIs imeis, TLocationsQueryOptions queryOptions) const = 0;
        virtual NThreading::TFuture<TLocations> DoGetLocations(TLocationsQueryOptions queryOptions) const = 0;

        virtual NThreading::TFuture<THeartbeats> DoGetHeartbeats(TIMEIs imeis, THeartbeatsQueryOptions queryOptions) const = 0;
        virtual NThreading::TFuture<THeartbeats> DoGetHeartbeats(THeartbeatsQueryOptions queryOptions) const = 0;

        virtual NThreading::TFuture<TSensors> DoGetSensors(TIMEIs imeis, TSensorIds ids, const TSensorsQueryOptions& queryOptions) const = 0;
        virtual NThreading::TFuture<TMultiSensor> DoGetSensors(const TString& imei, const TSensorsQueryOptions& queryOptions) const = 0;
        virtual NThreading::TFuture<TSensors> DoGetSensor(TSensorId id, const TSensorsQueryOptions& queryOptions) const = 0;
    };

    class ISensorHistoryApi {
    public:
        using TTimelineEvent = std::pair<TInstant, NDrive::TSensorValue>;

        class THistory {
        public:
            THistory(TVector<TTimelineEvent>&& timeline, const TInstant lastUpdate)
                : Timeline(std::move(timeline))
                , LastUpdate(lastUpdate)
            {}

            const TVector<TTimelineEvent>& GetTimeline() const;
            TInstant GetLastUpdate() const;
        private:
            TVector<TTimelineEvent> Timeline;
            TInstant LastUpdate;
        };

        using TSensorHistory = TMap<TSensorId, THistory>;
        using TCarsSensorsHistory = TMap<TString, TSensorHistory>;

        virtual ~ISensorHistoryApi() {}
        virtual NThreading::TFuture<TCarsSensorsHistory> GetSensorHistory(TSensorId sensorId) const = 0;
        virtual NThreading::TFuture<TSensorHistory> GetSensorsHistory(const TString& imei) const = 0;

        static std::tuple<TString, ui16, ui16> ParseHistoryDocId(const TString& str) {
            auto parts = SplitString(str, ":");
            Y_ENSURE(parts.size() == 4 && parts[0] == "imei", "Invalid accumulator id format: " << str);
            return { parts[1] , FromString<ui16>(parts[2]), FromString<ui16>(parts[3]) };
        }
    };

    class ISensorHistoryClient {
    public:
        using TResult = TMap<TString, TSensorsCachePtr>;

    public:
        virtual ~ISensorHistoryClient() = default;

        virtual NThreading::TFuture<TResult> Get(
            TConstArrayRef<TString> objectIds,
            TConstArrayRef<TString> imeis,
            TInstant since,
            TInstant until = TInstant::Max(),
            TConstArrayRef<TSensorId> sensorIds = {}
        ) const = 0;
        virtual NThreading::TFuture<TSensorsCachePtr> Get(
            const TString& objectId,
            const TString& imei,
            TInstant since,
            TInstant until = TInstant::Max(),
            TConstArrayRef<TSensorId> sensorIds = {}
        ) const;
    };
}
