#include "local.h"

#include <drive/telematics/protocol/sensor.h>
#include <drive/telematics/server/location/location.h>

#include <library/cpp/threading/future/future.h>

TLocalSensorApi::TLocalSensorApi(TLocalSensorStorage::TPtr localSensorStorage)
    : LocalSensorStorage(localSensorStorage)
    , Pusher(MakeAtomicShared<TLocalPusher>(localSensorStorage))
{
}

TLocalSensorApi::TLocalSensorApi()
    : TLocalSensorApi(TLocalSensorStorage::Instance())
{
}

NThreading::TFuture<NDrive::ISensorApi::THeartbeats> TLocalSensorApi::DoGetHeartbeats(TIMEIs imeis, THeartbeatsQueryOptions queryOptions) const {
    THeartbeats heartbeats;
    for (const auto& imei : imeis) {
        if (auto heartbeat = LocalSensorStorage->GetHeartbeat(imei, queryOptions.GetName())) {
            heartbeats.emplace(imei, std::move(*heartbeat));
        }
    }
    return NThreading::MakeFuture(std::move(heartbeats));
}

NThreading::TFuture<NDrive::ISensorApi::THeartbeats> TLocalSensorApi::DoGetHeartbeats(THeartbeatsQueryOptions queryOptions) const {
    return NThreading::MakeFuture(LocalSensorStorage->GetHeartbeats(queryOptions.GetName()));
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> TLocalSensorApi::DoGetLocations(TIMEIs imeis, TLocationsQueryOptions queryOptions) const {
    TLocations locations;
    for (const auto& imei : imeis) {
        if (auto location = LocalSensorStorage->GetLocation(imei, queryOptions.GetName())) {
            locations.emplace(imei, std::move(*location));
        }
    }
    return NThreading::MakeFuture(std::move(locations));
}

NThreading::TFuture<NDrive::ISensorApi::TLocations> TLocalSensorApi::DoGetLocations(TLocationsQueryOptions queryOptions) const {
    return NThreading::MakeFuture(LocalSensorStorage->GetLocations(queryOptions.GetName()));
}
NThreading::TFuture<NDrive::ISensorApi::TMultiSensor> TLocalSensorApi::DoGetSensors(const TString& imei, const TSensorsQueryOptions& /* queryOptions */) const {
    if (auto sensors = LocalSensorStorage->GetSensors(imei)) {
        return NThreading::MakeFuture(std::move(*sensors));
    }
    return NThreading::MakeFuture(TMultiSensor());
}

NThreading::TFuture<NDrive::ISensorApi::TSensors> TLocalSensorApi::DoGetSensors(TIMEIs imeis, TSensorIds ids, const TSensorsQueryOptions& /* queryOptions */) const {
    TSensors sensors;
    for (const auto& imei : imeis) {
        for (auto sensorId : ids) {
            if (auto sensor = LocalSensorStorage->GetSensor(imei, sensorId.Id, sensorId.SubId)) {
                sensors[imei].push_back(std::move(*sensor));
            }
        }
    }

    for (auto& s : sensors) {
        std::sort(s.second.begin(), s.second.end());
    }
    return NThreading::MakeFuture(std::move(sensors));
}

NThreading::TFuture<NDrive::ISensorApi::TSensors> TLocalSensorApi::DoGetSensor(NDrive::TSensorId id, const TSensorsQueryOptions& /* queryOptions */) const {
    NDrive::ISensorApi::TSensors sensors;
    for (auto&&[imei, sensor] : LocalSensorStorage->GetSensor(id)) {
        sensors[imei] = { sensor };
    }
    return NThreading::MakeFuture(std::move(sensors));
}
