#include "client.h"

#include <drive/telematics/common/handler.h>

#include <drive/library/cpp/threading/container.h>
#include <drive/library/cpp/threading/future.h>

#include <kernel/reqid/reqid.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <rtline/api/indexing_client/neh_client.h>
#include <rtline/api/search_client/neh_client.h>
#include <rtline/library/json/adapters.h>
#include <rtline/library/json/builder.h>
#include <rtline/library/json/parse.h>
#include <rtline/library/scheduler/global.h>
#include <rtline/util/algorithm/container.h>
#include <rtline/util/algorithm/ptr.h>
#include <rtline/util/algorithm/type_traits.h>

#include <util/random/random.h>
#include <util/string/builder.h>
#include <util/system/hostname.h>

class NDrive::TTelematicsClient::TConnectionsUpdater: public TGlobalScheduler::TScheduledItem<NDrive::TTelematicsClient::TConnectionsUpdater> {
private:
    using TBase = TGlobalScheduler::TScheduledItem<NDrive::TTelematicsClient::TConnectionsUpdater>;

public:
    TConnectionsUpdater(NDrive::TTelematicsClient* client, TInstant now, TDuration period)
        : TBase(client->GetName(), client->GetName(), now)
        , Client(client)
        , Period(period)
    {
    }

    THolder<IScheduledItem> GetNextScheduledItem(TInstant now) const override {
        return MakeHolder<TConnectionsUpdater>(Client, now + Period, Period);
    }

    void Process(void* /*threadSpecificResource*/) override {
        INFO_LOG << "ConnectionsUpdater: requesting connections list" << Endl;
        auto cleanup = Hold(this);
        auto asyncConnectionsList = Yensured(Client)->ListConnections();
    }

private:
    NDrive::TTelematicsClient* Client;
    TDuration Period;
};

NDrive::TTelematicsClient::TContext::TContext(TStringBuf baseId)
    : Id(TStringBuilder() << (baseId ? baseId : TStringBuf("anonymous")) << '-' << RandomNumber<ui32>())
{
}

void NDrive::TTelematicsClient::TContext::AddEvent(NJson::TJsonValue&& data, TInstant timestamp /*= TInstant::Zero()*/) {
    DEBUG_LOG << Id << ": " << data.GetStringRobust() << Endl;
    TEvent ev;
    ev.Timestamp = timestamp ? timestamp : Now();
    ev.Data = std::move(data);
    auto guard = Guard(Lock);
    Events.insert(std::move(ev));
}

void NDrive::TTelematicsClient::TContext::SetEvents(TEvents&& value) {
    auto guard = Guard(Lock);
    Events = std::move(value);
}

void NDrive::TTelematicsClient::TContext::SetIMEI(TString value) {
    auto guard = Guard(Lock);
    IMEI = std::move(value);
}

void NDrive::TTelematicsClient::TContext::SetShard(TString value) {
    auto guard = Guard(Lock);
    Shard = std::move(value);
}

void NDrive::TTelematicsClient::TContext::SetStatus(EStatus value) {
    Status = value;
}

void NDrive::TTelematicsClient::TContext::SetWaitConnectionTimeout(TDuration value) {
    WaitConnectionTimeout = value;
}

void NDrive::TTelematicsClient::TContext::SetOptimisticRequestSucceeded(bool value) {
    OptimisticRequestSucceeded = value;
}

void NDrive::TTelematicsClient::TContext::SetTerminationProcessed(bool value) {
    TerminationProcessed = value;
}

template <class T>
THolder<T> NDrive::TTelematicsClient::IResponse::ParseAs(const NJson::TJsonValue& content) {
    auto result = MakeHolder<T>();
    result->FromJson(content);
    return result;
}

THolder<NDrive::TTelematicsClient::IResponse> NDrive::TTelematicsClient::IResponse::Parse(const NJson::TJsonValue& content) {
    const TString& type = content["type"].GetString();
    if (type == "FirstTask" || type == "SequentialTask") {
        auto common = ParseAs<TCommonResponse>(content);
        auto actives = NJson::FromJson<TVector<TString>>(content["active"]);
        auto last = NJson::FromJson<TMaybe<TString>>(content["last"]);
        auto active = !actives.empty() ? MakeMaybe(std::move(actives.front())) : Nothing();
        auto selected = last ? last : active;
        if (selected) {
            for (auto&& task : content["tasks"].GetArraySafe()) {
                auto response = Parse(task);
                Y_ENSURE(response, "cannot parse task from " << task.GetStringRobust());
                auto cr = dynamic_cast<TCommonResponse*>(response.Get());
                Y_ENSURE(cr, "cannot cast parsed task from " << task.GetStringRobust());
                if (cr->Id == *selected) {
                    if (common) {
                        cr->Merge(*common);
                    }
                    return response;
                }
            }
        }
        Y_ENSURE(!selected, "cannot find active or last task " << selected << " in " << type);
    }
    if (type == "CanRequestTask") {
        return ParseAs<TCanResponse>(content);
    }
    if (type == "SendCommandTask" || type == "GetParameterTask") {
        NDrive::NVega::ECommandCode command = NDrive::NVega::ECommandCode::UNKNOWN;
        NJson::ReadField(content, "command", NJson::Stringify(command), true);
        switch (command) {
        case NDrive::NVega::ECommandCode::GET_PARAM:
            return ParseAs<TGetParameterResponse>(content);
        default:
            return ParseAs<TCommandResponse>(content);
        }
    }
    if (type == "ConnectionList") {
        return ParseAs<TConnectionListResponse>(content);
    }
    if (type == "DownloadFileTask") {
        return ParseAs<TDownloadFileResponse>(content);
    }
    return ParseAs<TCommonResponse>(content);
}

void NDrive::TTelematicsClient::TCommonResponse::Merge(const TCommonResponse& other) {
    Tuple() = other.Tuple();
}

void NDrive::TTelematicsClient::TCommonResponse::FromJson(const NJson::TJsonValue& content) {
    Y_ENSURE(NJson::ParseField(content["status"], NJson::Stringify(Status), true), "cannot read status: " << content["status"].GetStringRobust());
    Y_ENSURE(NJson::ParseField(content["id"], Id, true), "cannot read id: " << content["id"].GetStringRobust());
    Y_ENSURE(NJson::ParseField(content["info"], Message), "cannot read info: " << content["info"].GetStringRobust());
    Y_ENSURE(NJson::ParseField(content["events"], Events), "cannot read events: " << content["events"].GetStringRobust());
    Y_ENSURE(NJson::ParseField(content["created"], Created), "cannot read created timestamp: " << content["created"].GetStringRobust());
    Y_ENSURE(NJson::ParseField(content["started"], Started), "cannot read started timestamp: " << content["started"].GetStringRobust());
    Y_ENSURE(NJson::ParseField(content["heartbeat"], Heartbeat), "cannot read heartbeat timestamp: " << content["heartbeat"].GetStringRobust());
    Y_ENSURE(NJson::ParseField(content["finished"], Finished), "cannot read finished timestamp: " << content["finished"].GetStringRobust());
    NJson::ReadField(content, "imei", IMEI);
}

void NDrive::TTelematicsClient::TCanResponse::FromJson(const NJson::TJsonValue& content) {
    TBase::FromJson(content);
    NJson::ReadField(content["frames"], Frames);
}

void NDrive::TTelematicsClient::TCommandResponse::FromJson(const NJson::TJsonValue& content) {
    TBase::FromJson(content);
    Y_ENSURE(NJson::ParseField(content["command"], NJson::Stringify(Code), true), "cannot read command code: " << content["command"].GetStringRobust());
    const auto& result = content["result"];
    if (result.IsDefined()) {
        const auto& r = result["result"];
        if (r.IsDefined()) {
            Result = static_cast<NDrive::NVega::TCommandResponse::EResult>(NJson::FromJson<ui32>(r));
        }
        const auto& e = result["exception"];
        if (e.IsDefined()) {
            Message = NJson::FromJson<TString>(e);
        }
    }
}

void NDrive::TTelematicsClient::TConnectionListResponse::FromJson(const NJson::TJsonValue& content) {
    NJson::ReadField(content, "connections", Connections);
}

void NDrive::TTelematicsClient::TGetParameterResponse::FromJson(const NJson::TJsonValue& content) {
    TBase::FromJson(content);
    const auto& result = content["result"];
    Y_ENSURE(NJson::ParseField(result["sensor"], Sensor), "cannot read sensor " << result["sensor"].GetStringRobust());
}

void NDrive::TTelematicsClient::TDownloadFileResponse::FromJson(const NJson::TJsonValue& content) {
    TBase::FromJson(content);
    const auto& result = content["result"];
    if (!result.IsDefined()) {
        return;
    }

    const auto& data = result["data"];
    if (data.IsDefined()) {
        auto value = Base64Decode(NJson::FromJson<TString>(data));
        Data = TBuffer(value.Data(), value.Size());
        Result = TResult::OK;
    }

    const auto& errorCode = result["error_code"];
    if (errorCode.IsDefined()) {
        Result = static_cast<TResult>(NJson::FromJson<ui32>(errorCode));
    }
}

const NDrive::TTelematicsClient::THandler& NDrive::TTelematicsClient::THandler::Subscribe(
    THandlerCallback&& callback,
    TInstant deadline,
    const NThreading::TFuture<void>& precondition
) const {
    if (deadline) {
        auto now = Now();
        if (deadline > now) {
            auto callbackContainer = MakeThreadSafeContainer(std::move(callback));
            auto wrapper = [callbackContainer = std::move(callbackContainer)](const THandler& handler) {
                auto callback = Yensured(callbackContainer)->Release().GetOrElse(nullptr);
                if (callback) {
                    callback(handler);
                }
            };
            auto self = *this;
            bool scheduled = TGlobalScheduler::Schedule(deadline, [wrapper, self = std::move(self)] {
                wrapper(self);
            });
            if (!scheduled) {
                ERROR_LOG << GetId() << ": cannot schedule deadline trigger" << Endl;
            }
            Y_ASSERT(scheduled);
            Subscribe(std::move(wrapper), /*deadline=*/TInstant::Zero(), precondition);
        } else {
            callback(*this);
        }
    } else {
        auto self = *this;
        if (precondition.Initialized()) {
            auto waiter = NThreading::WaitAll(Response.IgnoreResult(), precondition);
            waiter.Subscribe([callback = std::move(callback), self = std::move(self)](const NThreading::TFuture<void>& /*w*/) {
                callback(self);
            });
        } else {
            Response.Subscribe([callback = std::move(callback), self = std::move(self)](const TAsyncResponse& /*r*/) {
                callback(self);
            });
        }
    }
    return *this;
}

const NDrive::TTelematicsClient::THandler& NDrive::TTelematicsClient::THandler::WaitAndEnsureSuccess(TInstant deadline) const {
    Y_ENSURE(Response.Wait(deadline), Id << ": wait timeout");
    auto status = GetStatus();
    Y_ENSURE(status == EStatus::Success, Id << ": " << status << ' ' << GetMessage());
    return *this;
}

NDrive::TTelematicsClient::TResponsePtr NDrive::TTelematicsClient::THandler::GetResponseImpl() const {
    if (Response.HasValue()) {
        return Response.GetValue();
    } else if (Response.HasException()) {
        return MakeAtomicShared<TErrorResponse>(EStatus::InternalError, NThreading::GetExceptionMessage(Response));
    } else {
        return nullptr;
    }
}

NDrive::TTelematicsClient::TEvents NDrive::TTelematicsClient::THandler::GetEventsImpl() const {
    NDrive::TTelematicsClient::TEvents result;
    if (Context) {
        result = Context->GetEvents();
    }
    if (Response.HasValue()) {
        auto response = std::dynamic_pointer_cast<TCommonResponse>(Response.GetValue());
        if (response) {
            result.insert(response->Events.begin(), response->Events.end());
        }
    }
    return result;
}

TString NDrive::TTelematicsClient::THandler::GetImeiImpl() const {
    if (Context) {
        return Context->GetIMEI();
    } else {
        return {};
    }
}

TString NDrive::TTelematicsClient::THandler::GetMessageImpl() const {
    auto response = GetResponseImpl();
    if (auto common = std::dynamic_pointer_cast<TCommonResponse>(response)) {
        return common->Message;
    }
    if (auto error = std::dynamic_pointer_cast<TErrorResponse>(response)) {
        return error->Message;
    }
    return "null response";
}

TString NDrive::TTelematicsClient::THandler::GetShardImpl() const {
    if (Context) {
        return Context->GetShard();
    } else {
        return {};
    }
}

NDrive::TTelematicsClient::EStatus NDrive::TTelematicsClient::THandler::GetStatusImpl() const {
    if (Response.HasValue()) {
        auto response = Response.GetValue().Get();
        if (response) {
            return response->GetStatus();
        }
    } else if (Response.HasException()) {
        return EStatus::InternalError;
    }
    if (Context) {
        return Context->GetStatus();
    } else {
        return EStatus::Unknown;
    }
}

NDrive::TTelematicsClient::TTelematicsClient(TAtomicSharedPtr<IServiceDiscovery> sd, const TOptions& options, const TAtomicSharedPtr<NTvmAuth::TTvmClient>& tvm)
    : Options(options)
    , Created(Now())
    , Client(MakeAtomicShared<NNeh::THttpClient>(options.MetaConfig))
    , ServiceDiscovery(sd)
    , Tvm(tvm)
{
    for (auto&& shard : GetShards()) {
        Client->RegisterSource(shard, Options.MetaConfig, shard);
    }

    auto saasTvmAuth = Tvm && Options.InfoSaasTvmId ? MakeMaybe<NDrive::TTvmAuth>(Tvm, Options.InfoSaasTvmId) : Nothing();

    if (Options.InfoFetchHost) {
        InfoFetchClient = MakeAtomicShared<NRTLine::TNehSearchClient>(Options.InfoFetchService, Options.InfoFetchHost, Options.InfoFetchPort, Options.MetaConfig, saasTvmAuth, Options.InfoSpMetaSearch);
    }
    if (Options.InfoPushHost) {
        InfoPushClient = MakeAtomicShared<NRTLine::TNehIndexingClient>(Options.InfoPushToken, Options.InfoPushHost, Options.InfoPushPort, Options.MetaConfig);
    }
    GlobalSchedulerRegistrator.emplace(GetName());
    if (Options.ConnectionsShardsUpdatePeriod) {
        Y_ENSURE(TGlobalScheduler::Schedule(MakeHolder<TConnectionsUpdater>(this, Now(), Options.ConnectionsShardsUpdatePeriod)));
    }
}

NDrive::TTelematicsClient::~TTelematicsClient() {
}

NDrive::TTelematicsClient::THandler NDrive::TTelematicsClient::Command(
    TStringBuf imei,
    const NDrive::NVega::TCommand& command,
    TDuration timeout,
    TDuration waitConnectionTimeout,
    TStringBuf externalId
) const {
    auto id = externalId ? TString(externalId) : NDrive::NProtocol::GenerateId(imei, Options.ReqIdClass);
    auto commandData = NJson::ToJson(command);
    auto commandTimeout = timeout ? timeout : Options.DefaultCommandTimeout;
    auto context = MakeAtomicShared<TContext>(id);
    context->SetIMEI(TString{imei});
    context->SetWaitConnectionTimeout(waitConnectionTimeout);
    context->AddEvent(NJson::TMapBuilder
        ("command", commandData)
        ("host", HostName())
        ("id", id)
        ("imei", imei)
        ("timeout", NJson::ToJson(commandTimeout))
        ("type", "CommandInitializing")
    );

    auto request = CreateRequest("/connection/command/", TStringBuilder() << "imei" << '=' << imei);
    auto data = std::move(commandData);
    data["external_id"] = id;
    data["timeout"] = NJson::ToJson(commandTimeout);
    request.SetPostData(data.GetStringRobust());
    return Execute(std::move(id), std::move(request), std::move(context), commandTimeout);
}

NDrive::TTelematicsClient::THandler NDrive::TTelematicsClient::Download(
    TStringBuf imei,
    TStringBuf filename,
    TDuration timeout/* = TDuration::Zero()*/,
    TDuration waitConnection/* = TDuration::Zero()*/,
    TStringBuf externalId/* = {}*/
) const {
    auto id = externalId ? TString(externalId) : NDrive::NProtocol::GenerateId(imei, Options.ReqIdClass);
    auto downloadTimeout = timeout ? timeout : Options.DefaultCommandTimeout;
    auto context = MakeAtomicShared<TContext>(id);
    context->SetIMEI(TString{imei});
    context->SetWaitConnectionTimeout(waitConnection);
    context->AddEvent(NJson::TMapBuilder
        ("filename", filename)
        ("host", HostName())
        ("id", id)
        ("imei", imei)
        ("timeout", NJson::ToJson(downloadTimeout))
        ("type", "DownloadInitializing")
    );

    auto request = CreateRequest("/connection/download/", TStringBuilder() << "imei=" << imei
                                                               << '&' << "filename=" << filename
                                                               << '&' << "external_id=" << id
                                                               << '&' << "timeout=" << downloadTimeout.MicroSeconds());
    return Execute(std::move(id), std::move(request), std::move(context), downloadTimeout);
}

NDrive::TTelematicsClient::THandler NDrive::TTelematicsClient::Upload(
    TStringBuf imei,
    TStringBuf filename,
    TBuffer&& content,
    TDuration timeout /* = TDuration::Zero() */,
    TDuration waitConnectionTimeout /* = TDuration::Zero() */,
    TStringBuf externalId /* = {} */
) const {
    auto id = externalId ? TString(externalId) : NDrive::NProtocol::GenerateId(imei, Options.ReqIdClass);
    auto uploadTimeout = timeout ? timeout : Options.DefaultCommandTimeout;
    auto context = MakeAtomicShared<TContext>(id);
    context->SetIMEI(TString{imei});
    context->SetWaitConnectionTimeout(waitConnectionTimeout);
    context->AddEvent(NJson::TMapBuilder
        ("filename", filename)
        ("host", HostName())
        ("id", id)
        ("imei", imei)
        ("timeout", NJson::ToJson(uploadTimeout))
        ("type", "UploadInitializing")
    );
    auto request = CreateRequest("/connection/upload/", TStringBuilder() << "imei=" << imei
                                                        << '&' << "filename=" << filename
                                                        << '&' << "external_id=" << id
                                                        << '&' << "timeout=" << uploadTimeout.MicroSeconds());
    request.SetPostData(std::move(content));
    return Execute(std::move(id), std::move(request), std::move(context), uploadTimeout);
}

NDrive::TTelematicsClient::THandler NDrive::TTelematicsClient::Execute(TString id, NNeh::THttpRequest&& request, TAtomicSharedPtr<NDrive::TTelematicsClient::TContext> context, TDuration timeout) const {
    auto broadcastResponse = Execute(std::move(request), EStatus::ConnectionNotFound, context);
    auto response = broadcastResponse.Apply([this, id, context, timeout](const TAsyncResponse& br) {
        TResponsePtr broadcastResponse = br.GetValue();
        Y_ENSURE(broadcastResponse);
        if (broadcastResponse->Final()) {
            context->AddEvent(NJson::TMapBuilder
                ("type", "FinalResponse")
            );
            return NThreading::MakeFuture(std::move(broadcastResponse));
        }

        const auto& shard = broadcastResponse->Shard;
        Y_ENSURE(shard);
        if (context) {
            context->SetShard(shard);
            context->SetStatus(broadcastResponse->GetStatus());
        }

        return Wait(shard, id, timeout, context);
    });
    response.Subscribe([context](const TAsyncResponse& r) {
        Y_ENSURE(context);
        if (r.HasValue()) {
            context->AddEvent(NJson::TMapBuilder
                ("type", "CommandFinished")
            );
        } else {
            context->AddEvent(NJson::TMapBuilder
                ("type", "MissingResponseValue")
                ("exception", NThreading::GetExceptionInfo(r))
            );
        }
    });
    response.Subscribe([this, id, context](const TAsyncResponse& r) {
        PushInfo(id, context, r);
    });
    return { std::move(id), std::move(context), std::move(response) };
}

NDrive::TTelematicsClient::THandler NDrive::TTelematicsClient::Restore(const TString& id) const {
    auto context = MakeAtomicShared<TContext>(id);
    auto request = CreateWaitRequest(id, TDuration::Zero());
    auto response = Execute(std::move(request), EStatus::Lost, context);
    response.Subscribe([context](const TAsyncResponse& br) {
        Y_ENSURE(context);
        if (!br.HasValue()) {
            context->AddEvent(NJson::TMapBuilder
                ("type", "MissingBroadcastValue")
                ("exception", NThreading::GetExceptionInfo(br))
            );
            return;
        }
        const auto& broadcastResponse = br.GetValue();
        if (!broadcastResponse) {
            context->AddEvent("NullBroadcastValue");
            return;
        }
        auto commonResponse = std::dynamic_pointer_cast<TCommonResponse>(broadcastResponse);
        if (commonResponse && context) {
            context->SetIMEI(commonResponse->IMEI);
        }
        if (context) {
            context->SetShard(broadcastResponse->Shard);
            context->SetStatus(broadcastResponse->GetStatus());
            context->AddEvent(NJson::TMapBuilder
                ("type", "RestoreFinished")
                ("shard", broadcastResponse->Shard)
            );
        }
    });
    return { id, context, std::move(response) };
}

NDrive::TTelematicsClient::THandler NDrive::TTelematicsClient::FetchInfo(const TString& id) const {
    auto context = MakeAtomicShared<TContext>(id);
    auto response = FetchInfo(id, context);
    return { id, context, std::move(response) };
}

TString NDrive::TTelematicsClient::GetCachedShard(TStringBuf imei) const {
    auto shards = GetConnectionsShards();
    if (!shards) {
        return {};
    }
    auto p = shards->find(imei);
    if (p != shards->end()) {
        return p->second;
    } else {
        return {};
    }
}

NDrive::TTelematicsClient::TAsyncConnectionsList NDrive::TTelematicsClient::ListConnections() const {
    auto context = MakeAtomicShared<TContext>("ListConnections");
    auto request = CreateRequest("/connection/list/");
    auto replies = BroadcastRequest(request, TDuration::Zero(), context);
    auto broadcast = Parse(replies, context);
    auto responses = MakeVector(NContainer::Values(broadcast));
    auto waiter = NThreading::WaitAll(responses);
    if (context) {
        context->AddEvent(NJson::TMapBuilder
            ("type", "WaitAll")
            ("shards", NJson::ToJson(MakeVector<TStringBuf>(NContainer::Keys(broadcast))))
        );
    }
    auto result = waiter.Apply([broadcast = std::move(broadcast), context = std::move(context)](const NThreading::TFuture<void>& /*w*/) {
        TConnectionsList result;
        result.Context = context;
        for (auto&& [shard, response] : broadcast) {
            if (response.HasException()) {
                if (context) {
                    context->AddEvent(NJson::TMapBuilder
                        ("type", "ResponseException")
                        ("shard", shard)
                        ("exception", NThreading::GetExceptionInfo(response))
                    );
                }
                result.UnansweredShards.push_back(shard);
                continue;
            }

            auto r = response.GetValue();
            auto clr = std::dynamic_pointer_cast<TConnectionListResponse>(r);
            if (!clr) {
                if (context) {
                    context->AddEvent(NJson::TMapBuilder
                        ("type", "ResponseNullOrBadCast")
                        ("shard", shard)
                        ("r", static_cast<bool>(r))
                    );
                }
                result.UnansweredShards.push_back(shard);
                continue;
            }

            result.ShardedConnections[shard] = std::move(clr->Connections);
        }
        return result;
    });
    result.Subscribe([this](const TAsyncConnectionsList& r) {
        if (!r.HasValue()) {
            ERROR_LOG << "ListConnection error: " << NThreading::GetExceptionMessage(r) << Endl;
            return;
        }
        const auto& connectionsList = r.GetValue();
        if (!connectionsList.UnansweredShards.empty()) {
            WARNING_LOG << "ListConnection warning: "
                << connectionsList.UnansweredShards.size() << " unanswered shards: "
                << JoinStrings(connectionsList.UnansweredShards, ", ") << Endl;
        }
        auto next = MakeAtomicShared<TConnectionsShards>();
        auto previous = GetConnectionsShards();
        if (previous) {
            *next = *previous;
        }
        for (auto&& [shard, connections] : connectionsList.ShardedConnections) {
            for (auto&& connection : connections) {
                (*next)[connection.IMEI] = shard;
            }
        }
        SetConnectionsShards(next);
    });
    return result;
}

NDrive::TTelematicsClient::TAsyncBroadcast NDrive::TTelematicsClient::Parse(const TMap<TString, NThreading::TFuture<NNeh::THttpReply>>& replies, TContextPtr context) const {
    TAsyncBroadcast result;
    for (auto&&[shard, reply] : replies) {
        result.emplace(shard, Parse(reply, shard, context));
    }
    return result;
}

NDrive::TTelematicsClient::TAsyncResponse NDrive::TTelematicsClient::Parse(const NThreading::TFuture<NNeh::THttpReply>& reply, const TString& shard, TContextPtr context) const {
    return reply.Apply([this, context = std::move(context), shard](const NThreading::TFuture<NNeh::THttpReply>& r) {
        const auto& reply = r.GetValue();
        return Parse(reply, shard, std::move(context));
    });
}

NDrive::TTelematicsClient::TResponsePtr NDrive::TTelematicsClient::Parse(const NNeh::THttpReply& reply, const TString& shard, TContextPtr context) const try {
    auto code = reply.Code();
    if (context) {
        context->AddEvent(NJson::TMapBuilder
            ("type", "ParsingResponse")
            ("shard", shard)
            ("response", reply.Serialize())
        );
    }
    switch (code) {
        case HTTP_OK:
            break;
        case HTTP_INTERNAL_SERVER_ERROR:
        case HTTP_BAD_GATEWAY:
        case HTTP_NOT_FOUND:
            return nullptr;
        default:
            throw yexception() << code << ' ' << reply.ErrorMessage() << ' ' << reply.Content();
    }

    auto content = NJson::ReadJsonFastTree(reply.Content());
    auto result = TCommonResponse::Parse(content);
    result->Shard = shard;
    if (context) {
        context->AddEvent(NJson::TMapBuilder
            ("type", "ParsedResponse")
            ("shard", shard)
        );
    }
    return result;
} catch (const yexception& e) {
    if (context) {
        context->AddEvent(NJson::TMapBuilder
            ("type", "ParsingException")
            ("exception", e.AsStrBuf())
        );
    }
    return nullptr;
}

namespace {
    TString GetHandlerId(TStringBuf id) {
        return TStringBuilder() << "chi:" << id;
    }
}

NDrive::TTelematicsClient::TAsyncResponse NDrive::TTelematicsClient::FetchInfo(const TString& id, TContextPtr context) const {
    Y_ENSURE(InfoFetchClient, "InfoFetchClient is missing");

    NRTLine::TQuery query;
    query.SetText(GetHandlerId(id));
    query.SetReqId(ReqIdGenerate(Options.ReqIdClass.c_str()));
    auto reply = InfoFetchClient->SendAsyncQueryF(query);
    auto result = reply.Apply([context](const NThreading::TFuture<NRTLine::TSearchReply>& r) -> TResponsePtr {
        const NRTLine::TSearchReply& reply = r.GetValue();
        Y_ENSURE(reply.IsSucceeded(), "InfoRequest:" << reply.GetCode() << ':' << reply.GetReqId());
        Y_ENSURE(context, "null Context");

        TString message;
        reply.ScanDocs([context, &message](const NMetaProtocol::TDocument& document) {
            for (auto&& propertie : document.GetArchiveInfo().GetGtaRelatedAttribute()) {
                const auto& key = propertie.GetKey();
                const auto& value = propertie.GetValue();
                if (key == "Events") {
                    context->SetEvents(NJson::FromJson<TEvents>(NJson::ToJson(NJson::JsonString(value))));
                    continue;
                }
                if (key == "Shard") {
                    context->SetShard(value);
                    continue;
                }
                if (key == "Status") {
                    context->SetStatus(FromString<EStatus>(value));
                    continue;
                }
                if (key == "Message") {
                    message = value;
                    continue;
                }
            }
        });
        return MakeAtomicShared<TErrorResponse>(context->GetStatus(), std::move(message));
    });
    return result;
}

NDrive::TTelematicsClient::TAsyncPushResult NDrive::TTelematicsClient::PushInfo(const TString& id, TContextPtr context, const TAsyncResponse& response) const {
    if (!InfoPushClient) {
        return {};
    }

    NRTLine::TAction action;
    NRTLine::TDocument& document = action.AddDocument();
    THandler handler(id, context, MakeCopy(response));
    TInstant timestamp = Now();
    document.SetUrl(GetHandlerId(id));
    document.SetTimestamp(timestamp.Seconds());
    document.SetDeadline(timestamp + Options.InfoLifetime);
    document.AddProperty("Events", NJson::ToJson(handler.GetEvents()).GetStringRobust());
    document.AddProperty("Host", HostName());
    document.AddProperty("Id", handler.GetId());
    document.AddProperty("Shard", handler.GetShard());
    document.AddProperty("Status", handler.GetStatus());
    document.AddProperty("Message", handler.GetMessage());

    NRTLine::TGeoData& geo = document.AddGeoData();
    geo.AddCoordinate({});

    auto reply = InfoPushClient->Send(action);
    reply.Subscribe([id](const NThreading::TFuture<NRTLine::TSendResult>& r) {
        if (r.HasValue()) {
            const NRTLine::TSendResult& reply = r.GetValue();
            if (reply.IsSucceeded()) {
                DEBUG_LOG << id << ": push successful" << Endl;
            } else {
                ERROR_LOG << id << ": push unsuccessful: " << reply.GetCode() << ' ' << reply.GetMessage() << Endl;
            }
        } else {
            ERROR_LOG << id << ": push exception occurred: " << NThreading::GetExceptionMessage(r) << Endl;
        }
    });
    return reply.IgnoreResult();
}

NDrive::TTelematicsClient::TAsyncResponse NDrive::TTelematicsClient::Execute(NNeh::THttpRequest&& request, EStatus fallback, TContextPtr context, bool optimistic) const {
    if (optimistic) {
        auto imei = context ? context->GetIMEI() : TString();
        auto shard = GetCachedShard(imei);
        if (shard) {
            if (context) {
                context->AddEvent(NJson::TMapBuilder
                    ("type", "OptimisticShard")
                    ("shard", shard)
                );
            }
        } else {
            return Execute(std::move(request), fallback, context, /*optimistic=*/false);
        }

        auto reply = MakeRequest(shard, request, TDuration::Zero(), context);
        auto response = Parse(reply, shard, context);
        return response.Apply([context, fallback, request = std::move(request), shard = std::move(shard), this](const TAsyncResponse& r) mutable {
            if (r.HasValue()) {
                if (r.GetValue()) {
                    if (context) {
                        context->SetOptimisticRequestSucceeded(true);
                    }
                    return r;
                } else if (context) {
                    context->AddEvent(NJson::TMapBuilder
                        ("type", "OptimisticNullResponse")
                        ("shard", shard)
                    );
                }
            } else {
                if (context) {
                    context->AddEvent(NJson::TMapBuilder
                        ("type", "OptimisticException")
                        ("shard", shard)
                        ("exception", NThreading::GetExceptionInfo(r))
                    );
                }
            }
            return Execute(std::move(request), fallback, context, /*optimistic=*/false);
        });
    }

    auto waitConnectionTimeout = context ? context->GetWaitConnectionTimeout() : TDuration::Zero();
    if (!waitConnectionTimeout) {
        waitConnectionTimeout = Options.DefaultWaitConnectionTimeout;
    }

    request.AddCgiData(TStringBuilder() << "&wait_timeout" << '=' << waitConnectionTimeout.MicroSeconds());
    auto replies = BroadcastRequest(request, Options.BroadcastTimeout + waitConnectionTimeout, context);
    auto broadcast = Parse(replies, context);
    auto broadcastResponse = Wait(std::move(broadcast), fallback, context);
    return broadcastResponse;
}

NDrive::TTelematicsClient::TAsyncResponse NDrive::TTelematicsClient::Wait(TAsyncBroadcast&& broadcast, EStatus fallback, TContextPtr context) const {
    auto responses = MakeVector(NContainer::Values(broadcast));
    auto waiter = NThreading::WaitAny(responses);
    if (context) {
        context->AddEvent(NJson::TMapBuilder
            ("type", "WaitBroadcast")
            ("shards", NJson::ToJson(MakeVector<TStringBuf>(NContainer::Keys(broadcast))))
        );
    }
    return waiter.Apply([this, broadcast = std::move(broadcast), context = std::move(context), fallback](const NThreading::TFuture<void>& w) {
        w.GetValue();
        bool incorrect = false;
        TAsyncBroadcast remaining;
        for (auto&& [shard, response] : broadcast) {
            if (response.HasException()) {
                if (context) {
                    context->AddEvent(NJson::TMapBuilder
                        ("type", "ResponseException")
                        ("shard", shard)
                        ("exception", NThreading::GetExceptionInfo(response))
                    );
                }
                response.GetValue();
            }
            if (!response.HasValue()) {
                remaining.emplace(shard, std::move(response));
                continue;
            }
            TResponsePtr r = response.GetValue();
            if (r) {
                if (context) {
                    context->AddEvent(NJson::TMapBuilder
                        ("type", "WaitResult")
                        ("shard", shard)
                    );
                }
                return NThreading::MakeFuture(std::move(r));
            }
            incorrect = true;
        }

        Y_ENSURE(remaining.size() < broadcast.size());
        if (!remaining.empty()) {
            return Wait(std::move(remaining), fallback, std::move(context));
        } else {
            if (context) {
                context->AddEvent(NJson::TMapBuilder
                    ("type", "WaitFallback")
                    ("status", ToString(incorrect ? EStatus::Incorrect : fallback))
                );
            }
            return NThreading::MakeFuture<TResponsePtr>(
                MakeAtomicShared<TErrorResponse>(incorrect ? EStatus::Incorrect : fallback,
                    ("broadcast finished: " + ToString(incorrect ? EStatus::Incorrect : fallback))
                )
            );
        }
    });
}

NDrive::TTelematicsClient::TAsyncResponse NDrive::TTelematicsClient::Wait(const TString& shard, TStringBuf id, TDuration timeout, TContextPtr context) const {
    auto request = CreateWaitRequest(id, timeout);
    auto reply = MakeRequest(shard, request, timeout, context);
    auto parsed = Parse(reply, shard, context);
    auto waitTimeout = Options.DefaultWaitTaskTimeout;
    if (waitTimeout) {
        return parsed.Apply([this, id = TString(id), timeout, waitTimeout, context](const TAsyncResponse& r) {
            if (!r.HasValue()) {
                return r;
            }

            const auto& response = r.GetValue();
            auto commonResponse = std::dynamic_pointer_cast<TCommonResponse>(response);
            if (!commonResponse || commonResponse->Status != EStatus::Terminated) {
                return r;
            }

            if (context) {
                context->SetTerminationProcessed(true);
                context->AddEvent(NJson::TMapBuilder
                    ("type", "WaitTaskAfterTermination")
                    ("wait_timeout", NJson::ToJson(waitTimeout))
                );
            }

            auto request = CreateWaitRequest(id, timeout, waitTimeout);
            auto replies = BroadcastRequest(request, timeout + waitTimeout, context);
            auto broadcast = Parse(replies, context);
            return Wait(std::move(broadcast), EStatus::Terminated, context);
        });
    } else {
        return parsed;
    }
}

NNeh::THttpRequest NDrive::TTelematicsClient::CreateWaitRequest(TStringBuf id, TDuration timeout, TDuration waitTimeout) const {
    auto cgi = TStringBuilder() << "id" << '=' << id;
    if (timeout && timeout != TDuration::Max()) {
        cgi << '&' << "timeout" << '=' << timeout.MicroSeconds();
    }
    if (waitTimeout) {
        cgi << '&' << "wait_timeout" << '=' << waitTimeout.MicroSeconds();
    }
    return CreateRequest("/task/wait/", std::move(cgi));
}

NNeh::THttpRequest NDrive::TTelematicsClient::CreateRequest(TString&& req, TString&& cgi) const {
    NNeh::THttpRequest result;
    result.SetUri(std::move(req));
    if (cgi) {
        result.SetCgiData(std::move(cgi));
    }
    result.AddHeader("X-Req-Id", ReqIdGenerate(Options.ReqIdClass.c_str()));
    if (Tvm) {
        result.AddHeader("X-Ya-Service-Ticket", Tvm->GetServiceTicketFor(Options.DestinationClientId));
    }
    return result;
}

TMap<TString, NThreading::TFuture<NNeh::THttpReply>> NDrive::TTelematicsClient::BroadcastRequest(const NNeh::THttpRequest& request, TDuration timeout, TContextPtr context) const {
    auto tout = timeout ? timeout : Options.BroadcastTimeout;
    if (context) {
        auto post = TStringBuf(request.GetPostData().AsCharPtr(), request.GetPostData().Size());
        context->AddEvent(NJson::TMapBuilder
            ("type", "BroadcastRequest")
            ("method", request.GetRequestType())
            ("request", request.GetRequest())
            ("headers", NJson::ToJson(NJson::Dictionary(request.GetHeaders())))
            ("post", NJson::ToJson(Base64Encode(post)))
        );
    }
    TMap<TString, NThreading::TFuture<NNeh::THttpReply>> result;
    for (auto&& shard : GetShards()) {
        if (!Client->HasSource(shard)) {
            Client->RegisterSource(shard, Options.MetaConfig, shard);
        }
        result.emplace(shard, MakeRequestImpl(shard, request, tout));
    }
    return result;
}

NThreading::TFuture<NNeh::THttpReply> NDrive::TTelematicsClient::MakeRequest(const TString& shard, const NNeh::THttpRequest& request, TDuration timeout, TContextPtr context) const {
    auto tout = timeout ? timeout : Options.RequestTimeout;
    auto result = MakeRequestImpl(shard, request, tout);
    if (context) {
        auto post = TStringBuf(request.GetPostData().AsCharPtr(), request.GetPostData().Size());
        context->AddEvent(NJson::TMapBuilder
            ("type", "MakeRequest")
            ("shard", shard)
            ("method", request.GetRequestType())
            ("request", request.GetRequest())
            ("headers", NJson::ToJson(NJson::Dictionary(request.GetHeaders())))
            ("post", NJson::ToJson(Base64Encode(post)))
        );
    }
    return result;
}

NThreading::TFuture<NNeh::THttpReply> NDrive::TTelematicsClient::MakeRequestImpl(const TString& shard, const NNeh::THttpRequest& request, TDuration timeout) const {
    return Client->SendAsync(shard, request, Now() + timeout);
}

NDrive::TTelematicsClient::TConnectionsShardsConstPtr NDrive::TTelematicsClient::GetConnectionsShards() const {
    auto guard = Guard(ConnectionsLock);
    return ConnectionsShards;
}

void NDrive::TTelematicsClient::SetConnectionsShards(TConnectionsShardsPtr value) const {
    auto guard = Guard(ConnectionsLock);
    ConnectionsShards = std::move(value);
}

TString NDrive::TTelematicsClient::GetName() const {
    return TStringBuilder() << "TelematicsClient:" << static_cast<const void*>(this) << ":" << Created.MicroSeconds();
}

TVector<TString> NDrive::TTelematicsClient::GetShards() const {
    Y_ENSURE(ServiceDiscovery);
    auto shards = ServiceDiscovery->GetHosts();
    Y_ENSURE(!shards.empty(), "Cannot find telematics shards");
    return shards;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TTelematicsClient::TConnection& object) {
    NJson::TJsonValue result;
    result["alive"] = object.Alive;
    result["created"] = NJson::ToJson(object.Created);
    result["blackbox"] = NJson::ToJson(object.Blackbox);
    result["heartbeat"] = NJson::ToJson(object.Heartbeat);
    result["imei"] = object.IMEI;
    result["client"] = object.Client;
    result["server"] = object.Server;
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TTelematicsClient::TConnection& result) {
    return
        NJson::ParseField(value["alive"], result.Alive) &&
        NJson::ParseField(value["created"], result.Created) &&
        NJson::ParseField(value["blackbox"], result.Blackbox) &&
        NJson::ParseField(value["heartbeat"], result.Heartbeat) &&
        NJson::ParseField(value["imei"], result.IMEI) &&
        NJson::ParseField(value["client"], result.Client) &&
        NJson::ParseField(value["server"], result.Server);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TTelematicsClient::TEvent& object) {
    if (object.Data.IsMap()) {
        auto result = object.Data;
        if (!result.Has("timestamp")) {
            result["timestamp"] = NJson::ToJson(object.Timestamp);
        }
        return result;
    } else {
        NJson::TJsonValue result;
        result["data"] = object.Data;
        result["timestamp"] = NJson::ToJson(object.Timestamp);
        return result;
    }
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrive::TTelematicsClient::TEvent& result) {
    if (!NJson::TryFromJson(value["timestamp"], result.Timestamp)) {
        return false;
    }
    const NJson::TJsonValue& data = value["data"];
    if (data.IsDefined()) {
        result.Data = data;
    } else {
        result.Data = value;
    }
    return true;
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::TTelematicsClient::THandler& object) {
    NJson::TJsonValue result;
    result["id"] = object.GetId();
    result["events"] = NJson::ToJson(object.GetEvents());
    result["message"] = NJson::ToJson(NJson::Nullable(object.GetMessage()));
    result["shard"] = NJson::ToJson(NJson::Nullable(object.GetShard()));
    result["status"] = ToString(object.GetStatus());
    return result;
}
